/**
 * Copyright (c) 2025 itemis AG - All rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 */
package com.yakindu.base.types.libraries

import com.google.inject.Inject
import com.yakindu.base.types.Declaration
import com.yakindu.base.types.Package
import com.yakindu.base.types.TypeBuilder
import org.eclipse.emf.ecore.resource.impl.ResourceImpl

/**
 * Abstract base class for type library implementations.
 * 
 * It implements an in memory resource with all contributed types. 
 * Custom implementation have to implement 'createLibraryPackage' and 'getLibraryURI'. They also should apply '@Singleton' DI binding.
 * 
 * @author Axel Terfloth - initial contribution
 */
abstract class AbstractTypeLibrary implements ITypeLibrary {
		
	@Inject protected extension TypeBuilder builder
	
	protected var Package libraryPackage;
	
	protected val resource = new ResourceImpl(libraryURI) {
		override protected doUnload() {
			// Do no unload in memory resource
		}
	}
	
	def protected synchronized create() {
		libraryPackage = createLibraryPackage
		resource.URI = libraryURI
		resource.contents += libraryPackage
	}

	def getResource() {
		getLibraryPackage
		resource
	}

	def protected <T extends Declaration> T memberWithName(Class<T> type, String name) {
		getLibraryPackage.member.filter(type).findFirst[it.name==name]
	}
	
	/** 
	 * The type library is expected to provide a package instance. This must contain the custom content.
	 */
	protected def Package createLibraryPackage();

	override getLibraryPackage() {
		if (libraryPackage === null) {
			create();
		}
		return libraryPackage
	}
	
}