/**
 * Copyright (c) 2020-2026 itemis AG - All rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * 
 */
package com.yakindu.base.types.typesystem

import com.google.inject.Singleton
import com.yakindu.base.types.Event

import static com.yakindu.base.types.TypesFactory.*

/** 
 * 
 * @author andreas muelder - Initial contribution and API
 * @author axel terfloth - extensions
 * 
 */
@Singleton
class GenericTypeSystem extends AbstractTypeSystem {

	static final GenericTypeSystem INSTANCE = new GenericTypeSystem()

	public static final String EVENT_VALUE_PROPERTY = "value"
	
	protected new() {
	}

	def static GenericTypeSystem getInstance() {
		return INSTANCE
	}

	override protected void initRegistries() {
		declarePrimitive(STRING)
		declarePrimitive(STRING_LITERAL)
		getType(STRING_LITERAL).setVisible(false)
		declareSuperType(getType(STRING), getType(STRING_LITERAL))
		declareConversion(getType(STRING_LITERAL), getType(STRING))
		declareConversion(getType(STRING), getType(STRING_LITERAL))
		declarePrimitive(REAL)
		declarePrimitive(INTEGER)
		declarePrimitive(UNSIGNED_INTEGER) => [
			visible = false
			abstract = true
		]
		declarePrimitive(BOOLEAN)
		declarePrimitive(VOID)
		declarePrimitive(NULL) => [abstract = true]
		declarePrimitive(ANY) => [abstract = true]
		declareSuperType(getType(ANY), getType(STRING))
		declareSuperType(getType(ANY), getType(BOOLEAN))
		declareSuperType(getType(ANY), getType(INTEGER))
		declareSuperType(getType(INTEGER), getType(REAL))
		declareSuperType(getType(UNSIGNED_INTEGER), getType(INTEGER))
		declareArrayType
		declareMapType
		declareFunctionType
		declareEventValueMetaFeature

		declarePrimitive(UNDEFINED) => [
			visible = false
			abstract = true
		]

	}

	protected def declareEventValueMetaFeature() {
		eINSTANCE.createProperty => [
			name = EVENT_VALUE_PROPERTY
			typeSpecifier = eINSTANCE.createTypeSpecifier => [
				// we assign 'any' as the most general type here 
				// because here we can not express that the type is the type 
				// of the context event. This is only defined within the 
				// context of a feature call. 
				// So the effective type will be inferred based on the feature call. 
				type = ANY.type
			]
			annotations += eINSTANCE.createAnnotation => [
				type = typeAnnotations.createBuiltInTypeAnnotationType()
			]

			metaFeatureRegistry.put(Event, it)
			resource.contents += it
		]
	}

	protected def declareArrayType() {
		val array = createArrayType()
		array.abstract = true
		declareType(array, array.getName())
		getResource().contents += array
	}

	protected def createArrayType() {
		val baseType = eINSTANCE.createTypeParameter => [
			name = "baseType"
		]
		val array = eINSTANCE.createComplexType => [ ar |
			ar.name = ARRAY
			ar.annotations += eINSTANCE.createAnnotation => [
				type = typeAnnotations.createBuiltInTypeAnnotationType()
			]
			ar.typeParameters += baseType
		]
		// operation add(value : <VT>) : void
		array.features += eINSTANCE.createOperation => [
			name = "add"
			typeSpecifier = eINSTANCE.createTypeSpecifier => [
				type = getType(VOID)
			]
			parameters += eINSTANCE.createParameter => [
				name = "element"
				typeSpecifier = eINSTANCE.createTypeSpecifier => [
					type = baseType
				]
			]
		]

		// operation addAll(value[] : <VT>) : void
		array.features += eINSTANCE.createOperation => [
			name = "addAll"
			typeSpecifier = eINSTANCE.createTypeSpecifier => [
				type = getType(VOID)
			]
			parameters += eINSTANCE.createParameter => [
				name = "elements"
				typeSpecifier = eINSTANCE.createTypeSpecifier => [
					type = getType(ARRAY)
					typeArguments += eINSTANCE.createTypeSpecifier => [
						type = baseType
					]
				]
			]
		]

		// operation remove(value : <VT>) : void
		array.features += eINSTANCE.createOperation => [
			name = "remove"
			typeSpecifier = eINSTANCE.createTypeSpecifier => [
				type = getType(VOID)
			]
			parameters += eINSTANCE.createParameter => [
				name = "element"
				typeSpecifier = eINSTANCE.createTypeSpecifier => [
					type = baseType
				]
			]
		]

		// operation removeAll(value : <VT>) : void
		array.features += eINSTANCE.createOperation => [
			name = "removeAll"
			typeSpecifier = eINSTANCE.createTypeSpecifier => [
				type = getType(VOID)
			]
			parameters += eINSTANCE.createParameter => [
				name = "elements"
				typeSpecifier = eINSTANCE.createTypeSpecifier => [
					type = getType(ARRAY)
					typeArguments += eINSTANCE.createTypeSpecifier => [
						type = baseType
					]
				]
			]
		]

		// operation contains(value : <VT>) : void
		array.features += eINSTANCE.createOperation => [
			name = "contains"
			typeSpecifier = eINSTANCE.createTypeSpecifier => [
				type = getType(BOOLEAN)
			]
			parameters += eINSTANCE.createParameter => [
				name = "element"
				typeSpecifier = eINSTANCE.createTypeSpecifier => [
					type = baseType
				]
			]
		]

		// operation clear() : void
		array.features += eINSTANCE.createOperation => [
			name = "clear"
			typeSpecifier = eINSTANCE.createTypeSpecifier => [
				type = getType(VOID)
			]
		]

		// operation size() : integer
		array.features += eINSTANCE.createOperation => [
			name = "size"
			typeSpecifier = eINSTANCE.createTypeSpecifier => [
				type = getType(INTEGER)
			]
		]

		array

	}

	protected def declareMapType() {
		val map = createMapType()
		map.abstract = true
		declareType(map, map.getName())
		getResource().contents += map
	}

	protected def createMapType() {
		val keyType = eINSTANCE.createTypeParameter => [
			name = "keyType"
		]
		val valueType = eINSTANCE.createTypeParameter => [
			name = "valueType"
		]
		val map = eINSTANCE.createComplexType => [ ar |
			ar.name = MAP
			ar.annotations += eINSTANCE.createAnnotation => [
				type = typeAnnotations.createBuiltInTypeAnnotationType()
			]
			ar.typeParameters += keyType
			ar.typeParameters += valueType
		]
		// operation put(key : <KT>, value <VT>) : <VT>
		map.features += eINSTANCE.createOperation => [
			name = "put"
			typeSpecifier = eINSTANCE.createTypeSpecifier => [
				type = valueType
			]
			parameters += eINSTANCE.createParameter => [
				name = "key"
				typeSpecifier = eINSTANCE.createTypeSpecifier => [
					type = keyType
				]
			]
			parameters += eINSTANCE.createParameter => [
				name = "value"
				typeSpecifier = eINSTANCE.createTypeSpecifier => [
					type = valueType
				]
			]
		]
		// operation remove(key : <KT>) : void
		map.features += eINSTANCE.createOperation => [
			name = "remove"
			typeSpecifier = eINSTANCE.createTypeSpecifier => [
				type = getType(VOID)
			]
			parameters += eINSTANCE.createParameter => [
				name = "key"
				typeSpecifier = eINSTANCE.createTypeSpecifier => [
					type = keyType
				]
			]
		]
		// operation size() : integer
		map.features += eINSTANCE.createOperation => [
			name = "size"
			typeSpecifier = eINSTANCE.createTypeSpecifier => [
				type = getType(INTEGER)
			]
		]

		// operation containsKey(key : KT) : boolean
		map.features += eINSTANCE.createOperation => [
			name = "containsKey"
			typeSpecifier = eINSTANCE.createTypeSpecifier => [
				type = getType(BOOLEAN)
			]
			parameters += eINSTANCE.createParameter => [
				name = "key"
				typeSpecifier = eINSTANCE.createTypeSpecifier => [
					type = keyType
				]
			]
		]

		// operation containsValue(value : VT) : boolean
		map.features += eINSTANCE.createOperation => [
			name = "containsKey"
			typeSpecifier = eINSTANCE.createTypeSpecifier => [
				type = getType(BOOLEAN)
			]
			parameters += eINSTANCE.createParameter => [
				name = "key"
				typeSpecifier = eINSTANCE.createTypeSpecifier => [
					type = keyType
				]
			]
		]

		// operation keys() : array<<KT>>
		map.features += eINSTANCE.createOperation => [
			name = "keys"
			typeSpecifier = eINSTANCE.createTypeSpecifier => [
				type = getType(ARRAY)
				typeArguments += eINSTANCE.createTypeSpecifier => [
					type = keyType
				]
			]
		]
		// operation values() : array<<VT>>
		map.features += eINSTANCE.createOperation => [
			name = "values"
			typeSpecifier = eINSTANCE.createTypeSpecifier => [
				type = getType(ARRAY)
				typeArguments += eINSTANCE.createTypeSpecifier => [
					type = valueType
				]
			]
		]
		// operation clear() : void
		map.features += eINSTANCE.createOperation => [
			name = "clear"
			typeSpecifier = eINSTANCE.createTypeSpecifier => [
				type = getType(VOID)
			]
		]
		map
	}
	
	protected def declareFunctionType() {
		val function = createFunctionType()
		function.abstract = true
		declareType(function, function.getName())
		getResource().contents += function
	}

	protected def createFunctionType() {
		val returnType = eINSTANCE.createTypeParameter => [
			name = "returnType"
		]
		val function = eINSTANCE.createComplexType => [ ft |
			ft.name = FUNCTION
			ft.annotations += eINSTANCE.createAnnotation => [
				type = typeAnnotations.createBuiltInTypeAnnotationType()
			]
			typeAnnotations.annotateAsReferenceType(ft)
			ft.typeParameters += returnType
		]
		// operation apply() : <RT>
		function.features += eINSTANCE.createOperation => [
			name = "apply"
			typeSpecifier = eINSTANCE.createTypeSpecifier => [
				type = returnType
			]
		]
		
		function
	}
	
}
