/**
 * Copyright (c) 2021 itemis AG - All rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * 
 * Contributors:
 * 	andreas muelder - itemis AG
 * 
 */
package com.yakindu.base.expressions.validation

import com.google.inject.Inject
import com.yakindu.base.expressions.expressions.BlockExpression
import com.yakindu.base.expressions.expressions.IfExpression
import com.yakindu.base.expressions.expressions.ReturnExpression
import com.yakindu.base.expressions.expressions.SwitchExpression
import com.yakindu.base.types.Expression
import com.yakindu.base.types.Operation
import com.yakindu.base.types.TypesUtil
import com.yakindu.base.types.inferrer.ITypeSystemInferrer
import com.yakindu.base.types.typesystem.ITypeSystem
import com.yakindu.base.types.validation.TypeValidator
import org.eclipse.xtext.validation.Check

/**
 * 
 * @author andreas muelder - Initial contribution and API
 * 
 */
class ReturnValidator extends ExpressionsBaseValidator {

	@Inject
	ITypeSystemInferrer inferrer
	@Inject
	TypeValidator typeValidator
	@Inject
	ITypeSystem typeSystem

	public static val UNREACHABLE_CODE_MSG = "Unreachable expression.";
	public static val UNREACHABLE_CODE_CODE = "UnreachableExpression";
	
	@Inject protected extension TypesUtil

	@Check
	def checkUnreachableCode(ReturnExpression it) {
		if (eContainer instanceof BlockExpression) {
			val block = eContainer as BlockExpression
			val index = block.expressions.indexOf(it)
			if (index < block.expressions.size - 1)
				error(UNREACHABLE_CODE_MSG, block.expressions.get(index + 1), null, UNREACHABLE_CODE_CODE)
		}
	}

	@Check
	def checkReturnTypesCompatible(Operation it) {
		val operationType = inferrer.infer(it)
		val returnExpressions = it.eAllContents.filter(ReturnExpression)
		returnExpressions.map[ex|inferrer.infer(ex)].forEach [ rT |
			typeValidator.assertAssignable(operationType, rT, null, this)
		]
	}

	public static val RETURN_MISSING_MSG = "The operation must return a result of type %s";
	public static val RETURN_MISSING_CODE = "ReturnMissing";

	@Check
	def checkMissingReturn(Operation it) {
		if (it.implementation === null)
			return
		val operationType = inferrer.infer(it)
		if (!typeSystem.isVoid(operationType.type)) {
			if (!implementation.returns) {
				error(String.format(RETURN_MISSING_MSG, it?.type.name), it, null, RETURN_MISSING_CODE)
			}
		}
	}

	def protected dispatch boolean returns(BlockExpression it) {
		if(expressions.isEmpty) false else expressions.lastOrNull.returns
	}

	def protected dispatch boolean returns(IfExpression it) {
		then.returns && ^else !== null && ^else.returns
	}

	def protected dispatch boolean returns(SwitchExpression it) {
		cases.forall[then.returns] && ^default.returns
	}

	def protected dispatch boolean returns(ReturnExpression it) {
		true
	}

	def protected dispatch boolean returns(Expression it) {
		false
	}
}
