/**
 * Copyright (c) 2020 itemis AG - All rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * 
 */
package com.yakindu.sct.model.stext.scoping

import com.google.common.base.Predicate
import com.google.common.collect.Lists
import com.google.inject.Inject
import com.yakindu.base.expressions.expressions.ElementReferenceExpression
import com.yakindu.base.expressions.expressions.FeatureCall
import com.yakindu.base.expressions.expressions.MetaCall
import com.yakindu.base.expressions.scoping.ExpressionsScopeProvider
import com.yakindu.base.expressions.util.ExpressionExtensions
import com.yakindu.base.types.ComplexType
import com.yakindu.base.types.EnumerationType
import com.yakindu.base.types.Event
import com.yakindu.base.types.Expression
import com.yakindu.base.types.MetaComposite
import com.yakindu.base.types.Package
import com.yakindu.base.types.Type
import com.yakindu.base.types.TypesFactory
import com.yakindu.base.types.inferrer.ITypeSystemInferrer
import com.yakindu.base.types.inferrer.ITypeSystemInferrer.InferenceResult
import com.yakindu.base.types.libraries.ITypeLibraryProvider
import com.yakindu.base.types.scoping.IPackageImport2URIMapper
import com.yakindu.base.types.scoping.TypeLibraryScope
import com.yakindu.base.types.typesystem.ITypeSystem
import com.yakindu.sct.model.sgraph.Region
import com.yakindu.sct.model.sgraph.RegularState
import com.yakindu.sct.model.sgraph.Scope
import com.yakindu.sct.model.sgraph.ScopedElement
import com.yakindu.sct.model.sgraph.SpecificationElement
import com.yakindu.sct.model.sgraph.Statechart
import com.yakindu.sct.model.sgraph.util.StatechartUtil
import com.yakindu.sct.model.stext.extensions.STextExtensions
import com.yakindu.sct.model.stext.resource.StextResource
import com.yakindu.sct.model.stext.stext.InterfaceScope
import com.yakindu.sct.model.stext.stext.InternalScope
import com.yakindu.sct.model.stext.stext.StatechartSpecification
import java.util.Collection
import java.util.List
import java.util.Set
import org.eclipse.emf.common.util.EList
import org.eclipse.emf.common.util.URI
import org.eclipse.emf.ecore.EObject
import org.eclipse.emf.ecore.EReference
import org.eclipse.emf.ecore.resource.Resource
import org.eclipse.xtext.EcoreUtil2
import org.eclipse.xtext.naming.IQualifiedNameProvider
import org.eclipse.xtext.naming.QualifiedName
import org.eclipse.xtext.resource.IEObjectDescription
import org.eclipse.xtext.resource.impl.EObjectDescriptionLookUp
import org.eclipse.xtext.scoping.IScope
import org.eclipse.xtext.scoping.Scopes
import org.eclipse.xtext.scoping.impl.FilteringScope
import org.eclipse.xtext.scoping.impl.ImportNormalizer
import org.eclipse.xtext.scoping.impl.ImportScope
import org.eclipse.xtext.scoping.impl.SimpleScope

/** 
 * @author andreas muelder
 * @author axel terfloth
 * @author alexander nyssen Added support for scoping of enumeration literals
 */
class STextScopeProvider extends ExpressionsScopeProvider {

	@Inject
	protected ITypeSystemInferrer typeInferrer
	@Inject
	protected ITypeSystem typeSystem
	@Inject
	protected IQualifiedNameProvider nameProvider
	@Inject
	ContextPredicateProvider predicateProvider
	@Inject
	protected extension STextExtensions utils
	@Inject
	protected StatechartUtil statechartUtil
	
	@Inject protected ITypeLibraryProvider typeLibraries
	@Inject protected extension StextDefaultTypeLibraries
	@Inject protected extension ExpressionExtensions
	@Inject protected ImportUriProvider importUriProvider
	@Inject protected IPackageImport2URIMapper importUriMapper
	
		
	protected extension TypesFactory = TypesFactory.eINSTANCE

	def Set<URI> importedAndDefaultURIs(Resource context) {
		val Collection<com.yakindu.sct.model.stext.stext.ImportScope> importScopes = importUriProvider.getImportScopes(context)

		val uriSet = importScopes
			.map[imports]
			.flatten
			.map[ importUriMapper.findPackageImport(context, it) ]
			.filter[ isPresent ]
			.map[ get.uri ]
			.toSet
			
		uriSet.addAll(defaultTypeLibraries)
		
		return uriSet
	}

	def IScope scope_Annotation_type(EObject context, EReference reference){
		new TypeLibraryScope.AnnotationTypeScope(IScope.NULLSCOPE, typeLibraries, importedAndDefaultURIs(context.eResource));
	}

	def IScope scope_ActiveStateReferenceExpression_value(EObject context, EReference reference) {
		var Statechart statechart = getStatechart(context)
		if(statechart === null) return IScope.NULLSCOPE
		var List<RegularState> allStates = EcoreUtil2.getAllContentsOfType(statechart, RegularState)
		var IScope scope = Scopes.scopeFor(allStates, nameProvider, IScope.NULLSCOPE)
		return new ImportScope(getActiveStateNormalizer(context), scope,
			new EObjectDescriptionLookUp(Lists.newArrayList(scope.getAllElements())), reference.getEReferenceType(),
			false)
	}

	def protected List<ImportNormalizer> getActiveStateNormalizer(EObject context) {
		var List<ImportNormalizer> normalizer = Lists.newArrayList()
		var SpecificationElement contextElement = getContextElement(context)
		if(contextElement === null) return normalizer
		
		var Region containingRegion = EcoreUtil2.getContainerOfType(contextElement, Region)
		if(containingRegion === null) return normalizer
		
		var QualifiedName fullyQualifiedName = nameProvider.getFullyQualifiedName(containingRegion)
		while (!fullyQualifiedName.getSegments().isEmpty()) {
			normalizer.add(new ImportNormalizer(fullyQualifiedName, true, false))
			fullyQualifiedName = fullyQualifiedName.skipLast(1)
		}
		return normalizer
	}

	/** 
	 * Scoping for types and taking imported namespaces into account e.g. in
	 * variable declarations.
	 */

	 
	def IScope scope_TypeSpecifier_type(EObject context, EReference reference) {
		return new GlobalTypeScope(delegate.getScope(context, reference), context.getStatechartResource, reference.EReferenceType)
	}
	
	def IScope scope_ElementReferenceExpression_reference(EObject context, EReference reference) {
		var IScope unnamedScope = getUnnamedTopLevelScope(context, reference)
		var Predicate<IEObjectDescription> predicate = calculateFilterPredicate(context, reference)
		unnamedScope = new FilteringScope(unnamedScope, predicate)
		val IScope packageContainerScope = new SimpleScope(unnamedScope, Scopes.scopedElementsFor(context.getStatechartResource.contents.filter(Package)))
		
		var IScope namedScope = getNamedTopLevelScope(context, reference)
		return new SimpleScope(packageContainerScope, namedScope.getAllElements())
	}

	def IScope scope_FeatureCall_feature(ElementReferenceExpression context, EReference reference) {
		return scope_FeatureCall(context, reference)
	}

	def IScope scope_FeatureCall_feature(FeatureCall context, EReference reference) {
		return scope_FeatureCall(context.owner, reference)
	}
	
	def protected IScope scope_FeatureCall(Expression context, EReference reference){
		var Predicate<IEObjectDescription> predicate = calculateFilterPredicate(context, reference)
		val EObject element = context.featureOrReference		
		if(element instanceof Event){
			return Scopes.scopeFor(typeSystem.getMetaFeatures(element))
		}
		var IScope scope = IScope.NULLSCOPE
		if (element instanceof Package) {
			if(element.eResource instanceof StextResource) return addScopeForInternalPackage(element, scope, predicate, reference)
			else return addScopeForPackage(element, scope, predicate, context, reference)
		}

		var InferenceResult result = typeInferrer.infer(context)
		var Type ownerType = if(result !== null) result.getType() else null
		if (element instanceof Scope) {
			scope = Scopes.scopeFor(element.getDeclarations())
			return new FilteringScope(scope, predicate)
		} else if (ownerType !== null) {
			scope = Scopes.scopeFor(typeSystem.getPropertyExtensions(ownerType))
			scope = Scopes.scopeFor(typeSystem.getOperationExtensions(ownerType), scope)
		}
		if (ownerType instanceof EnumerationType) {
			scope = addScopeForEnumType(ownerType, scope, predicate)
		}
		if (ownerType instanceof ComplexType) {
			scope = scopeForComplexTypeOwner(ownerType, scope, predicate, element)
		}

		return scope
	}
	
	def protected scopeForComplexTypeOwner(ComplexType ownerType,IScope scope,Predicate<IEObjectDescription> predicate, EObject element){
		addScopeForComplexType(ownerType, scope, predicate)
	}

	def IScope scope_FeatureCall_feature(MetaCall context, EReference reference) {

		var Predicate<IEObjectDescription> predicate = calculateFilterPredicate(context, reference);
		var Expression owner = context.getOwner();
		var EObject element = null;

		if (owner instanceof ElementReferenceExpression) {
			element = owner.getReference();
		} else if (owner instanceof FeatureCall) {
			element = owner.getFeature();
		} else {
			return getDelegate().getScope(context, reference);
		}

		var IScope scope = IScope.NULLSCOPE;

		if (element instanceof MetaComposite) {
			if (element.getMetaFeatures().size() > 0) {
				scope = Scopes.scopeFor(element.getMetaFeatures(), scope);
				scope = new FilteringScope(scope, predicate);
			}
		}

		return scope;
	}

	def protected IScope addScopeForEnumType(EnumerationType element, IScope parentScope,
		Predicate<IEObjectDescription> predicate) {
		var scope = parentScope
		scope = Scopes.scopeFor((element).getEnumerator(), scope)
		scope = new FilteringScope(scope, predicate)
		return scope
	}

	def protected IScope addScopeForComplexType(ComplexType type, IScope parentScope,
		Predicate<IEObjectDescription> predicate) {
		var scope = parentScope
		scope = Scopes.scopeFor(type.getAllFeatures(), scope)
		scope = new FilteringScope(scope, predicate)
		return scope
	}

	def protected IScope addScopeForPackage(Package pkg, IScope parentScope, Predicate<IEObjectDescription> predicate,  Expression context, EReference reference) {
		var scope = parentScope
		scope = Scopes.scopeFor(pkg.member, scope)
		scope = new FilteringScope(scope, predicate)
		return scope
	}
	
	def protected IScope addScopeForInternalPackage(Package pkg, IScope parentScope, Predicate<IEObjectDescription> predicate, EReference reference) {
		 
		val namespaceMembers = pkg.member

		var scope = parentScope
		//if(!(pkg.eContainer instanceof Package))
		//	scope = Scopes.scopeFor(#[pkg], scope)
		//else 
			scope = Scopes.scopeFor(namespaceMembers, scope)
		scope = new FilteringScope(scope, predicate)
		return scope
	}

	def protected Predicate<IEObjectDescription> calculateFilterPredicate(EObject context, EReference reference) {
		var Predicate<IEObjectDescription> predicate = null
		var EObject container = context
		var EReference ref = reference
		var break = false;
		while (container !== null && !break) {
			predicate = predicateProvider.getPredicate(container.eClass(), ref)
			if (!(predicate === ContextPredicateProvider.EMPTY_PREDICATE)) {
				break = true
			}
			ref = container.eContainingFeature() as EReference
			container = container.eContainer()
		}
		return predicate
	}

	/** 
	 * Returns the toplevel scope
	 */
	def protected IScope getNamedTopLevelScope(EObject context, EReference reference) {
		var List<EObject> scopeCandidates = Lists.newArrayList()
		var ScopedElement scopedElement = getScopedElement(context)
		if(scopedElement === null) return IScope.NULLSCOPE
		var EList<Scope> scopes = scopedElement.getScopes()
		for (Scope scope : scopes) {
			if (scope instanceof InterfaceScope) {
				var String name = scope.getName()
				if (name !== null && name.trim().length() > 0) {
					scopeCandidates.add(scope)
				}
			}
		}
		return Scopes.scopeFor(scopeCandidates)
	}

	/** 
	 * Returns a scope with all toplevel declarations of unnamed scope
	 */
	def protected IScope getUnnamedTopLevelScope(EObject context, EReference reference) {
		var List<EObject> scopeCandidates = Lists.newArrayList()
		var ScopedElement scopedElement = getScopedElement(context)
		if(scopedElement === null) return IScope.NULLSCOPE
		var EList<Scope> scopes = scopedElement.getScopes()
		for (Scope scope : scopes) {
			if (scope instanceof InterfaceScope) {
				var String name = scope.getName()
				if (name === null || name.trim().length() === 0) {
					scopeCandidates.addAll(scope.getDeclarations())
				}
			} else if (scope instanceof InternalScope) {
				scopeCandidates.addAll(scope.getDeclarations())
			}
		}
		//if(context.getStatechartResource.contents.filter(Package).head !== null)
		//	scopeCandidates.add(context.getStatechartResource.contents.filter(Package).head)
		// Add import scope
		val globalScope = getDelegate().getScope(context, reference)
		val libraryScope = new TypeLibraryScope(globalScope, typeLibraries, importedAndDefaultURIs(context.eResource));
		val scope = new GlobalTypeScope(libraryScope, context.getStatechartResource)
		return Scopes.scopeFor(scopeCandidates, scope)
	}

	def protected ScopedElement getScopedElement(EObject context) {
		val scopedElement = EcoreUtil2.getContainerOfType(context, ScopedElement)
		val spec = EcoreUtil2.getContainerOfType(context, StatechartSpecification)
		if(spec !== null && scopedElement !== null) return scopedElement

		return getStatechart(context)
	}
	
	def protected Statechart getStatechart(EObject context) {
		return utils.getStatechart(context)
	}
	
	def protected Resource getStatechartResource(EObject context) {
		context.statechart?.eResource ?: context.eResource
	}
}
