/**
 * Copyright (c) 2022 itemis AG - All rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * 
 */
package com.yakindu.sct.model.stext.validation

import com.google.inject.Inject
import com.yakindu.base.expressions.expressions.ReactionTrigger
import com.yakindu.sct.model.sgraph.Statechart
import com.yakindu.sct.model.sgraph.Synchronization
import com.yakindu.sct.model.sgraph.validation.SynchronizationValidator
import com.yakindu.sct.model.stext.concepts.ExitTransition
import com.yakindu.sct.model.stext.concepts.StatechartAnnotations
import com.yakindu.sct.model.stext.stext.AlwaysEvent
import com.yakindu.sct.model.stext.stext.ExitPointSpec
import org.eclipse.xtext.EcoreUtil2
import org.eclipse.xtext.validation.Check
import org.eclipse.xtext.validation.CheckType

/**
 * Validation for synhronization nodes that require information from the Stext based context.
 * See {@link SynchronizationValidator} for the other validation rules.
 * 
 * @author Laszlo Kovacs
 */
class STextSynchronizationValidatior extends STextBaseValidator {

	@Inject protected extension ExitTransition
	@Inject protected extension StatechartAnnotations

	public static val String SYNCHRONIZATION_INCOMING_TRIGGER_MSG = "Only one of the incoming transitions should have a trigger to keep the node reachable. Please consider removing trigger(s) or change to CycleBased execution.";
	public static val String SYNCHRONIZATION_INCOMING_TRIGGER_CODE = "synchronization.MoreThanOneIncomingTriggers";

	@Check(CheckType.FAST)
	def checkSyncTooManyTriggersOnIncomingTransition(Synchronization it) {
		if (EcoreUtil2.getContainerOfType(it, typeof(Statechart)).isEventDriven) {
			if (incomingTransitions.filter(
				t | t.trigger !== null &&
				if (t.trigger instanceof ReactionTrigger) {
					val rt = t.trigger as ReactionTrigger
					rt.triggers.filter(a|!(a instanceof AlwaysEvent)).size > 0
				} else {
					false
				}
			).map[trigger].size > 1) {
				warning(SYNCHRONIZATION_INCOMING_TRIGGER_MSG, it, null, -1, SYNCHRONIZATION_INCOMING_TRIGGER_CODE);
			}
		}
	}

	public static val String SYNCHRONIZATION_EXITPOINT_SPEC_MSG = "This transition is a synchronization transition, thus it can't be an exit transition from exit '%s'. Please remove exit point specification or reconsider using synchronization node.";
	public static val String SYNCHRONIZATION_EXITPOINT_SPEC_CODE = "synchronization.ExitPointSpec";

	@Check(CheckType.FAST)
	def checkJoinHasNoIncomingExitPointSpecTransition(Synchronization it) {
		if (incomingTransitions.size !== 1) {
			for (transition : incomingTransitions) {
				if (transition.isNamedExitTransition) {
					error(String.format(SYNCHRONIZATION_EXITPOINT_SPEC_MSG, transition.properties.filter(ExitPointSpec).head.exitpoint), transition, null, -1,
						SYNCHRONIZATION_EXITPOINT_SPEC_CODE);
				}
			}
		}
	}
}
