/**
 * Copyright (c) 2020-2025 itemis AG - All rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * 
 */
package com.yakindu.sct.model.stext.validation

import com.google.inject.Inject
import com.yakindu.base.expressions.expressions.EventSpec
import com.yakindu.base.expressions.expressions.ReactionTrigger
import com.yakindu.base.expressions.expressions.RegularEventSpec
import com.yakindu.base.expressions.expressions.TimeEventSpec
import com.yakindu.base.types.Trigger
import com.yakindu.sct.model.sgraph.Choice
import com.yakindu.sct.model.sgraph.Exit
import com.yakindu.sct.model.sgraph.FinalState
import com.yakindu.sct.model.sgraph.ReactionProperty
import com.yakindu.sct.model.sgraph.RegularState
import com.yakindu.sct.model.sgraph.Transition
import com.yakindu.sct.model.sgraph.Vertex
import com.yakindu.sct.model.sgraph.util.SgraphExtensions
import com.yakindu.sct.model.stext.concepts.CompletionTransition
import com.yakindu.sct.model.stext.concepts.ExitTransition
import com.yakindu.sct.model.stext.stext.AlwaysEvent
import com.yakindu.sct.model.stext.stext.DefaultTrigger
import com.yakindu.sct.model.stext.stext.EntryPointSpec
import com.yakindu.sct.model.stext.stext.ExitPointSpec
import com.yakindu.sct.model.stext.stext.LocalReaction
import java.util.HashSet
import java.util.Iterator
import org.eclipse.emf.common.util.EList
import org.eclipse.gmf.runtime.notation.Diagram
import org.eclipse.gmf.runtime.notation.View
import org.eclipse.xtext.validation.Check
import org.eclipse.xtext.validation.CheckType
import com.yakindu.sct.model.stext.stext.SubmachineReferenceExpression

class TransitionValidator extends STextBaseValidator {

	public static final String MULTIPLE_COMPLETION_TRANSITION = "Multiple completion/default transition. Only one completion/default transition is allowed from state '%s'";
	public static final String DEAD_TRANSITION_DUE_COMPLETION = "Dead transition from state '%s'. This transition is never taken due to the precedence of completion transition.";
	public static final String NOT_ALLOWED_LOCAL_REACTION = "Local reactions of state '%s' are never executed due to completion transition.";
	public static final String SOURCE_STATE_NOT_COMPLETABLE = "Dead transition. This completion transition is never taken because the source state '%s' is not completable.";
	public static final String INFINITE_CYCLE = "The completion transition of state '%s' results in an infinite cycle.";
	public static final String COMPLETION_TRANSITION_IGNORES_SUBMACHINE = "The completion transition of state '%s' will ignore the submachine state.";

	@Inject protected extension CompletionTransition
	@Inject protected extension SgraphExtensions
	@Inject protected extension ExitTransition

	def protected boolean resultsInfiniteCycle(Transition t, Vertex s, HashSet<Transition> traversedTransitions) {
		val outgoingCompletionTransitions = t.target.outgoingTransitions.filter(tr | tr.isCompletionTransition)
		if (t.target.outgoingTransitions.exists[isCompletionTransition && target == s]) {
			return true
		} else if(outgoingCompletionTransitions.exists[!traversedTransitions.contains(it)]){
			traversedTransitions.addAll(outgoingCompletionTransitions)
			return outgoingCompletionTransitions.exists[resultsInfiniteCycle(s,traversedTransitions)]
		} else false
	}

	@Check(CheckType.FAST)
	def void completionTransitions(com.yakindu.sct.model.sgraph.State state) {
		
		val completionTransitions = state.outgoingTransitions.filter[completionTransition].toList

		if (state.incomingTransitions.filter(intrans | intrans.isCompletionTransition).size > 0 && completionTransitions.exists[resultsInfiniteCycle(it.source, newHashSet)]) {
			warning(String.format(INFINITE_CYCLE, state.name), state, null, -1)
		}

		if (!state.isLeaf && completionTransitions.size > 0) {
			if (state.regions.map(r|r.vertices).flatten.filter[v|(v instanceof FinalState)].size < state.regions.size)
				completionTransitions.forEach[error(String.format(SOURCE_STATE_NOT_COMPLETABLE, source.name), it, null, -1)]
		}

		if (state.isLeaf && completionTransitions.size > 0 && state.scopes !== null && state.scopes.filter( s |
			s.members !== null
		).map(s|s.members).flatten.filter(typeof(LocalReaction)).filter(
			lr |
				(lr.trigger as ReactionTrigger).triggers.empty || !(lr.trigger as ReactionTrigger).triggers.filter( t |
					t instanceof RegularEventSpec || t instanceof TimeEventSpec || t instanceof AlwaysEvent
				).toList.empty
		).size > 0) {
			error(String.format(NOT_ALLOWED_LOCAL_REACTION, state.name), state, null, -1)
		}

		if (completionTransitions.size > 1) {
			completionTransitions.forEach [
				error(String.format(MULTIPLE_COMPLETION_TRANSITION, source.name), it, null, -1)
			]
		}

		if (state.isLeaf) {
			val regularTransitions = state.outgoingTransitions.filter(
				t |
					!completionTransitions.contains(t) && t.trigger !== null
			)
			if (completionTransitions.size == 1 && regularTransitions.size > 0) {
				regularTransitions.forEach[warning(String.format(DEAD_TRANSITION_DUE_COMPLETION, source.name), it, null, -1)]
			}
		}
		
		val submachineReferences = state.scopes.map[members].flatten.filter(SubmachineReferenceExpression)
		if ( !submachineReferences.empty && !completionTransitions.empty) {
			warning(String.format(COMPLETION_TRANSITION_IGNORES_SUBMACHINE, state.name), state, null, -1)
		}
	}

	
	public static final String DEAD_TRANSITION = "Dead transition. This transition can not be taken due to previous transition with '%s' trigger.";
	public static final String ALWAYS_TRUE_TRANSITION_USED = "Transition with '%s' should be evaluate as the last transition of the state as following transitions won't be evaluated. Change the transition order (in the Properties view).";

	@Check(CheckType.FAST)
	def void checkAlwaysTransitionHasLowestPriority(RegularState state) {
		var Iterator<Transition> iterator = state.outgoingTransitions.iterator()
		var Transition deadTransition = null
		while (iterator.hasNext()) {
			var Transition transition = iterator.next()
			var Trigger trigger = transition.trigger
			if (deadTransition !== null) {
				warning(String.format(DEAD_TRANSITION, getTransitionDeclaration(deadTransition)), transition, null, -1)
			}
			// check default/else trigger
			if (trigger instanceof DefaultTrigger && iterator.hasNext()) {
				warning(String.format(ALWAYS_TRUE_TRANSITION_USED, transition.getSpecification()), transition, null, -1)
				if (deadTransition === null) {
					deadTransition = transition
				}
			} else // check always/oncycle trigger
			if (trigger instanceof ReactionTrigger) {
				var EList<EventSpec> triggers = trigger.triggers
				if (triggers.size() === 1 && trigger.guard === null) {
					var EventSpec eventSpec = triggers.get(0)
					if (eventSpec instanceof AlwaysEvent && iterator.hasNext()) {
						warning(String.format(ALWAYS_TRUE_TRANSITION_USED, getTransitionDeclaration(transition)),
							transition, null, -1)
						if (deadTransition === null) {
							deadTransition = transition
						}
					}
				}
			}
		}
	}
	
	public static final String TRANSITION_SOURCE_ISNT_VERTEX = "The source of the transition must be a vertex. First move the source and target of the transition to a Node. It can only be deleted after fixing the source/target!";
	
	@Check(CheckType.FAST)
	def void checkEContainer(Transition transition) {
		if (!(transition.eContainer instanceof Vertex) && hasNotationView(transition)) {
			error(TRANSITION_SOURCE_ISNT_VERTEX, transition, null, -1)
		}
	}

	def hasNotationView(Transition transition) {
		transition.eResource.contents.filter(Diagram).map[eAllContents.toList].flatten.filter(View).
			exists[element == transition]
	}

	public static final String TRANSITION_ENTRY_SPEC_NOT_COMPOSITE = "Target state '%s' isn't composite.";
	public static final String TRANSITION_EXIT_SPEC_NOT_COMPOSITE = "Source state '%s' isn't composite.";
	public static final String TRANSITION_EXIT_SPEC_ON_MULTIPLE_SIBLINGS = "ExitPointSpec '%s' can't be used on transition siblings.";
	public static final String TRANSITION_NOT_EXISTING_NAMED_EXIT_POINT = "Source state '%s' needs at least one region with the named exit point.";

	@Check(CheckType.FAST)
	def void checkTransitionPropertySpec(Transition transition) {
		for (ReactionProperty property : transition.getProperties()) {
			if (property instanceof EntryPointSpec) {
				if (transition.target instanceof com.yakindu.sct.model.sgraph.State) {
					var com.yakindu.sct.model.sgraph.State state = (transition.
						target as com.yakindu.sct.model.sgraph.State)
					if (!state.isComposite()) {
						warning(String.format(TRANSITION_ENTRY_SPEC_NOT_COMPOSITE, transition.target.name), transition, null, -1)
					}
				}
			} else if (property instanceof ExitPointSpec) {
				if (transition.source instanceof com.yakindu.sct.model.sgraph.State) {
					var com.yakindu.sct.model.sgraph.State state = (transition.
						source as com.yakindu.sct.model.sgraph.State)
					if (!state.isComposite()) {
						warning(String.format(TRANSITION_EXIT_SPEC_NOT_COMPOSITE, transition.source.name), transition, null, -1)
					} else {
						// Validate an exit point is continued on one transition
						// only.						
						if (!state.outgoingTransitions.filter( t |
							transition !== t && t.isNamedExitTransition(property.exitpoint)
						).empty) {
							warning(String.format(TRANSITION_EXIT_SPEC_ON_MULTIPLE_SIBLINGS, property.exitpoint), transition, null, -1)
						}
						// Validate the state has minimally one named exit
						// region
						if (state.allPseudoStates.filter(Exit).filter(e|property.exitpoint.equals(e.name)).empty) {
							error(String.format(TRANSITION_NOT_EXISTING_NAMED_EXIT_POINT, state.name), transition, null, -1)
						}
					}
				}
			}
		}
	}

	@Check(CheckType.FAST)
	def void checkAlwaysAndDefaultTransitionInChoices(Choice choice) {
		var Transition deadTransition = null
		var EList<Transition> outgoingTransitions = choice.outgoingTransitions
		var int size = outgoingTransitions.size()
		var int deadTransitionIndex = 0
		for (var int i = 0; i < size; i++) {
			var Transition transition = outgoingTransitions.get(i)
			if (deadTransition !== null) {
				warning(String.format(DEAD_TRANSITION, getTransitionDeclaration(deadTransition)), transition, null, -1)
			}
			var Trigger trigger = transition.trigger
			if (trigger instanceof ReactionTrigger) {
				var EList<EventSpec> triggers = trigger.triggers
				if (triggers.size() === 1 && trigger.guard === null) {
					if (triggers.get(0) instanceof AlwaysEvent) {
						if (i !== size - 1) {
							warning(String.format(ALWAYS_TRUE_TRANSITION_USED, transition.getSpecification()),
								transition, null, -1)
						}
						if (deadTransition === null) {
							deadTransition = transition
							deadTransitionIndex = i
						}
					}
				}
			}
		}
		// if we got a dead transition, we need to re-check if a default was used before
		if (deadTransition !== null) {
			for (var int i = 0; i < deadTransitionIndex; i++) {
				var Transition transition = outgoingTransitions.get(i)
				var Trigger trigger = transition.trigger
				if (trigger instanceof DefaultTrigger || trigger === null) {
					warning(String.format(DEAD_TRANSITION, getTransitionDeclaration(deadTransition)), transition, null,
						-1)
				}
			}
		}
	}

	static final String KEYWORD_ONCYCLE = "oncycle"
	static final String KEYWORD_ALWAYS = "always"

	def protected String getTransitionDeclaration(Transition transition) {
		var String specification = transition.getSpecification()
		if (KEYWORD_ALWAYS.contains(specification)) {
			return KEYWORD_ALWAYS
		} else if (KEYWORD_ONCYCLE.contains(specification)) {
			return KEYWORD_ONCYCLE
		}
		return specification
	}

}
