/**
 * Copyright (c) 2022 itemis AG - All rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * 
 * Contributors:
 * 	Andreas Muelder - itemis AG
 * 
 */
package com.yakindu.sctunit.generator.base.extensions

import com.google.inject.Inject
import com.yakindu.base.base.NamedElement
import com.yakindu.base.expressions.expressions.AdditiveOperator
import com.yakindu.base.expressions.expressions.ArgumentExpression
import com.yakindu.base.expressions.expressions.AssignmentExpression
import com.yakindu.base.expressions.expressions.ElementReferenceExpression
import com.yakindu.base.expressions.expressions.EventRaisingExpression
import com.yakindu.base.expressions.expressions.ExpressionsFactory
import com.yakindu.base.expressions.expressions.FeatureCall
import com.yakindu.base.expressions.expressions.FloatLiteral
import com.yakindu.base.expressions.expressions.IntLiteral
import com.yakindu.base.expressions.expressions.LogicalAndExpression
import com.yakindu.base.expressions.expressions.LogicalNotExpression
import com.yakindu.base.expressions.expressions.LogicalOrExpression
import com.yakindu.base.expressions.expressions.LogicalRelationExpression
import com.yakindu.base.expressions.expressions.MultiplicativeOperator
import com.yakindu.base.expressions.expressions.NumericalAddSubtractExpression
import com.yakindu.base.expressions.expressions.NumericalMultiplyDivideExpression
import com.yakindu.base.expressions.expressions.NumericalUnaryExpression
import com.yakindu.base.expressions.expressions.PrimitiveValueExpression
import com.yakindu.base.expressions.expressions.RelationalOperator
import com.yakindu.base.expressions.expressions.StringLiteral
import com.yakindu.base.types.Property
import com.yakindu.base.types.Expression
import com.yakindu.base.types.Parameter
import com.yakindu.base.types.Type
import com.yakindu.base.types.inferrer.ITypeSystemInferrer
import com.yakindu.base.types.typesystem.GenericTypeSystem
import com.yakindu.sct.generator.core.templates.ExpressionsGenerator
import com.yakindu.sct.model.stext.stext.ActiveStateReferenceExpression
import com.yakindu.sct.model.stext.stext.EventDefinition
import com.yakindu.sct.model.stext.stext.VariableDefinition
import com.yakindu.sctunit.inferrer.TypesProvider
import com.yakindu.sctunit.sCTUnit.AssertionStatement
import com.yakindu.sctunit.sCTUnit.ProceedExpression
import com.yakindu.sctunit.sCTUnit.ProceedUnit
import com.yakindu.sctunit.sCTUnit.SCTUnitOperation
import com.yakindu.sctunit.sCTUnit.StatechartActiveExpression
import com.yakindu.sctunit.sCTUnit.StatechartFinalExpression
import com.yakindu.sctunit.sCTUnit.TestStatement
import org.eclipse.emf.ecore.EObject
import org.eclipse.xtext.naming.IQualifiedNameProvider

import static extension org.eclipse.emf.ecore.util.EcoreUtil.*

/**
 * 
 * @author andreas muelder - Initial contribution and API
 * 
 */
abstract class BaseExpressionExtensions extends ExpressionsGenerator {

	abstract def CharSequence generateNotLocalAssignment(AssignmentExpression it)

	abstract def CharSequence proceedTime(ProceedExpression it)

	abstract def CharSequence proceedCycles(ProceedExpression it)

	@Inject protected extension IQualifiedNameProvider
	@Inject protected extension ITypeSystemInferrer
	@Inject protected TypesProvider provider
	@Inject protected extension BaseNavigationExtensions
	@Inject protected extension BaseStatementExtensions stms
	@Inject protected extension BaseNamingExtensions
	
	extension ExpressionsFactory factory = ExpressionsFactory.eINSTANCE

	def getTypeSystem(EObject context) {
		return provider.getTypeSystem(context);
	}
	
	def dispatch code(Parameter it) {
		it.name
	}

	override dispatch CharSequence code(Expression expression) {
		'''/* generate «expression» not yet implemented */'''
	}

	def dispatch CharSequence code(EventRaisingExpression expression) {

		if (expression.event instanceof FeatureCall) {
			'''raiseEvent("«(expression.event as FeatureCall).owner.referenceName.toString».«((expression.event as FeatureCall).
				feature as EventDefinition).name»"«IF expression.value !== null»,«expression.value.code»«ENDIF»)'''
		} else {
			'''raiseEvent("«expression.event.referenceName»"«IF expression.value !== null»,«expression.value.code»«ENDIF»)'''
		}
	}

	def dispatch String referenceName(ElementReferenceExpression expression) {
		'''«(expression.reference as NamedElement).name»'''
	}
	
	def dispatch String referenceName(Property it) {
		'''«name»'''
	}
	
	def dispatch String referenceName(EObject it) {
		'''referenceName «it.eClass» not yet implemented'''.toString
	}
	
	def dispatch String referenceName(FeatureCall featureCall) {
		return featureCall.feature.referenceName
	}

	override dispatch CharSequence code(AssignmentExpression expression) {
		if ((expression.varRef).isLocal) {
			return '''«expression.varRef.code» «expression.operator.toString» «expression.expression.code»'''
		} else if((expression.varRef).isStatechartLocal) {
			expression.generateNotLocalAssignment
		} else {
			expression.generateExternalAssignment
		}
	}

	protected def dispatch generateAssignmentForIfExpr(VariableDefinition it, TestStatement stm) {
		'''«it.name» = «stms.generate(stm)»'''
	}
	
	def generateExternalAssignment(AssignmentExpression it) {
		'''Statechart External Value'''
	}

	protected def dispatch generateAssignmentForIfExpr(AssertionStatement it, TestStatement stm) {
		'''assertTrue(«IF !errorMsg.isNullOrEmpty»"«errorMsg»",«ENDIF»«stms.generate(stm)»);'''
	}

	def dispatch CharSequence code(ProceedExpression it) {
		if (unit == ProceedUnit.CYCLES) {
			return proceedCycles
		} else {
			return proceedTime
		}
	}

	def dispatch CharSequence code(VariableDefinition definition) {
		return '''«definition.name»'''
	}
	
	def dispatch CharSequence code(Void it) {
		''''''
	}

	def dispatch CharSequence code(SCTUnitOperation operation) {
		return '''«operation.name»()'''
	}

	override dispatch CharSequence code(ElementReferenceExpression it) {
		if (isExpressionVoidOperation) {
			return '''«referenceName»(«FOR expr : expressions SEPARATOR ', '»«expr.code»«ENDFOR»)'''
		}
		if (!reference.isLocal && reference.isStatechartLocal) {
			return generateNotLocalElemRefExpr
		} else if(!reference.isStatechartLocal) {
			referenceName
		}
		'''«referenceName»'''
	}

	def generateNotLocalElemRefExpr(ElementReferenceExpression it) {
		return '''«reference.valueGetter»("«reference.fullyQualifiedName»")'''
	}

	def dispatch CharSequence code(ActiveStateReferenceExpression expression) {
		'''isStateActive("«expression.value.name»")'''
	}

	def dispatch CharSequence code(StatechartFinalExpression stm) {
		'''interpreter.isFinal()'''
	}

	def dispatch CharSequence code(StatechartActiveExpression stm) {
		'''interpreter.isActive()'''
	}

	override dispatch CharSequence code(FeatureCall it) {
		owner.code(feature)
	}
	
	
	override dispatch code(Expression owner, Object object) '''
		Cannot handle «object.class.name» in context of «owner.eContainer.eClass.name»
	'''
	
	def dispatch code(Expression owner, EObject feature) '''
		Cannot find FeatureCall for «owner.eContainer.eClass.name»
		for Feature «feature.eClass.name»
	'''
	
	
	def dispatch code(Expression owner, Void feature)'''
		Feature for «owner.eContainer.eClass.name» must not be null!
	'''

	def dispatch CharSequence code(LogicalAndExpression expression) {
		'''«expression.leftOperand.code» && «expression.rightOperand.code»'''
	}

	def dispatch CharSequence code(LogicalOrExpression expression) {
		'''«expression.leftOperand.code» || «expression.rightOperand.code»'''
	}

	def dispatch CharSequence code(LogicalNotExpression expression) {
		'''!«expression.operand.code»'''
	}

	def dispatch CharSequence code(LogicalRelationExpression expression) {
		if (expression.leftOperand.infer.type.isStringType && expression.operator == RelationalOperator::EQUALS) {
			'''«expression.leftOperand.code».equals(«expression.rightOperand.generateString»)'''
		} else
			'''«expression.leftOperand.code» «expression.operator.literal» «expression.rightOperand.code»'''
	}

	def generateString(Expression expression) {
		// NumericvalAddSubtractExpression
		if (expression instanceof NumericalAddSubtractExpression) {
			if ((expression as NumericalAddSubtractExpression).operator == AdditiveOperator::PLUS) {
				return '''«expression.generateStringsForNonStrings»'''
			}
		}
		return '''«expression.code»'''
	}

	def CharSequence generateStringsForNonStrings(Expression exp) {
		var result = ""
		if (exp instanceof NumericalAddSubtractExpression) {
			result += exp.leftOperand.generateStringsForNonStrings
			result += ''' + '''
			result += exp.rightOperand.generateStringsForNonStrings
		} else if (exp instanceof PrimitiveValueExpression) {
			if (exp.value instanceof StringLiteral) {
				result += '''«exp.code»'''
			} else {
				result += '''String.valueOf(«exp.code»)'''
			}
		} else if (exp instanceof ElementReferenceExpression) {
			if (exp.reference.infer.type.isStringType) {
				result += '''«exp.code»'''
			} else {
				result += '''String.valueOf(«exp.code»)'''
			}
		} else {
			result += "/*ERROR in StringConversion*/"
		}
		return result
	}

	def boolean isStringType(Type type) {
		return type.getTypeSystem.isSame(type, type.getTypeSystem.getType(GenericTypeSystem.STRING))
	}

	def dispatch CharSequence code(NumericalAddSubtractExpression expression) {
		'''«expression.leftOperand.code»«expression.operator.literal»«expression.rightOperand.code»'''
	}

	def dispatch CharSequence code(NumericalMultiplyDivideExpression expression) {
		'''«expression.leftOperand.code»«expression.operator.literal»«expression.rightOperand.code»'''
	}

	def dispatch CharSequence code(NumericalUnaryExpression expression) {
		'''«expression.operator.literal» «expression.operand.code»'''
	}

	override dispatch code(IntLiteral literal) {
		return '''«literal.value»l'''
	}

	override dispatch code(FloatLiteral it) {
		return value + "f"
	}

	def dispatch isExpressionVoidOperation(Expression it) {
		!eAllContents.filter(ElementReferenceExpression).map[ref|ref instanceof SCTUnitOperation].isEmpty
	}

	def dispatch isExpressionVoidOperation(ElementReferenceExpression it) {
		reference instanceof SCTUnitOperation
	}
	
	def toMilliseconds(Expression valueExp, ProceedUnit unit) {
		switch (unit) {
			case MILLISECOND: valueExp.copy
			case MICROSECOND: valueExp.copy.parenthesis.div(1000.intLiteral)
			case NANOSECOND: valueExp.copy.parenthesis.div(1000000.intLiteral)
			case SECOND: valueExp.copy.parenthesis.mult(1000.intLiteral)
			default: valueExp.copy
		}
	}
	
	def dispatch parenthesis(Expression exp) {
		createParenthesizedExpression => [expression = exp]
	}
	
	def dispatch parenthesis(PrimitiveValueExpression exp) {
		exp // no need for parenthesis
	}
	
	def dispatch parenthesis(ArgumentExpression exp) {
		exp // no need for parenthesis
	}
	
	def div(Expression left, Expression right) {
		createNumericalMultiplyDivideExpression => [
				leftOperand = left
				operator = MultiplicativeOperator.DIV
				rightOperand = right
			]
	}
	
	def mult(Expression left, Expression right) {
		createNumericalMultiplyDivideExpression => [
				leftOperand = left
				operator = MultiplicativeOperator.MUL
				rightOperand = right
			]
	}
	
	def intLiteral(int v) {
		createPrimitiveValueExpression => [value = createIntLiteral => [value = v]]
	}
}
