/**
 * Copyright (c) 2016-2026 itemis AG - All rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * 
 * Contributors:
 * 	Andreas Muelder - itemis AG
 * 	Axel Terfloth - itemis AG
 * 
 */
package com.yakindu.sctunit.simulation.core.interpreter

import com.google.inject.Inject
import com.google.inject.Singleton
import com.yakindu.base.expressions.expressions.ElementReferenceExpression
import com.yakindu.base.expressions.expressions.EventRaisingExpression
import com.yakindu.base.expressions.expressions.EventValueReferenceExpression
import com.yakindu.base.expressions.expressions.FeatureCall
import com.yakindu.base.expressions.interpreter.base.IInterpreter
import com.yakindu.base.expressions.interpreter.context.IExecutionSlotResolver
import com.yakindu.base.expressions.interpreter.scheduling.ITimeTaskScheduler.TimeTask
import com.yakindu.base.expressions.interpreter.scheduling.ITimeTaskScheduler.TimeTask.Priority
import com.yakindu.base.types.Expression
import com.yakindu.sct.model.sexec.ExecutionFlow
import com.yakindu.sct.model.sexec.transformation.IModelSequencer
import com.yakindu.sct.model.sgraph.Statechart
import com.yakindu.sct.model.sruntime.ExecutionContext
import com.yakindu.sct.model.sruntime.ExecutionSlot
import com.yakindu.sct.model.stext.concepts.StatechartAnnotations
import com.yakindu.sct.simulation.core.sexec.interpreter.MultiStatechartInjector
import com.yakindu.sct.simulation.core.sexec.interpreter.StatechartRuntimeExtension
import com.yakindu.sctunit.sCTUnit.EnterExpression
import com.yakindu.sctunit.sCTUnit.ExitExpression
import com.yakindu.sctunit.sCTUnit.SCTUnitClass
import com.yakindu.sctunit.sCTUnit.SCTUnitOperation
import com.yakindu.sctunit.sCTUnit.StatechartActiveExpression
import com.yakindu.sctunit.sCTUnit.StatechartFinalExpression
import com.yakindu.sctunit.sCTUnit.TriggerWithoutEventExpression
import java.util.Optional
import org.eclipse.xtext.EcoreUtil2
import org.eclipse.emf.ecore.EObject

/**
 * 
 * @author andreas muelder - Initial contribution and API
 * @author axel terfloth - Integration of new interpreter 
 * 
 */
@Singleton
class ExecutionFlowSCTUnitTestCaseInterpreter extends BaseSCTUnitTestCaseInterpreter {

	@Inject extension StatechartAnnotations
	@Inject protected IModelSequencer sequencer
	@Inject protected IInterpreter interpreter
	@Inject protected extension StatechartRuntimeExtension

	@Inject protected extension IExecutionSlotResolver
	@Inject protected extension MultiStatechartInjector injector

	protected ExecutionFlow flow;

	override init(SCTUnitClass unitClass) {
		flow = sequencer.transform(unitClass.statechart)
		context = flow.newInstance
		context.init

		sctUnitContextInitializer.initialize(context, unitClass)
		// Has to be configured here cause no engine is used by SCTUnit
		if (unitClass.statechart.isCycleBased) {
			cyclePeriod = getCyclePeriod(unitClass.statechart)
			var cycleTask = new TimeTask(TimeTask.CYCLE_TASK_NAME, [context.runCycle()], Priority.LOW);
			schedulingService.scheduleTimeTask(cycleTask, true, cyclePeriod);
		}

		// create state machine instances for contained statechart references
		context.inject
	}

	override tearDown() {
		super.tearDown
		interpreter.reset
	}

	def dispatch Object execute(StatechartFinalExpression expression) {
		context.isFinal
	}

	override dispatch Object execute(EventValueReferenceExpression expression) {
		interpreter.evaluate(context, expression)
	}

	def dispatch Object execute(StatechartActiveExpression expression) {
		context.isActive
	}

	override dispatch Object execute(ElementReferenceExpression expression) {
		if (expression.expressionSCTUnitOperation !== null) {
			val owner = EcoreUtil2.getContainerOfType(expression, SCTUnitOperation)
			return expression.expressionSCTUnitOperation.executeStatement(expression.expressions, testCase, owner)
		} else if (expression.requiresStatechartScopeExecution) {
			return interpreter.evaluate(context, expression)
		}
		return executeElementReferenceExpression(expression)
	}

	override dispatch Object execute(EventRaisingExpression eventRaising) {

		val event = interpreter.evaluate(context, eventRaising.event)
		val value = eventRaising.value?.execute
		interpreter.raiseEvent(event, value)

		return null
	}

	override dispatch execute(FeatureCall call) {
		if (call.expressionSCTUnitOperation !== null) {
			val owner = EcoreUtil2.getContainerOfType(call, SCTUnitOperation)
			return call.expressionSCTUnitOperation.executeStatement(call.expressions, testCase, owner)
		} else if (call.requiresStatechartScopeExecution) {
			return interpreter.evaluate(context, call)
		} else {
			return executeFeatureCall(call)
		}
	}

	override protected Optional<ExecutionSlot> resolveSlot(ExecutionContext context, Expression expression) {
		if (expression.requiresStatechartScopeExecution) {
			val result = interpreter.evaluate(context, expression)
			return if(result !== null && result instanceof ExecutionSlot) Optional.of(
				result as ExecutionSlot) else Optional.empty
		} else
			super.resolveSlot(context, expression)
	}

	override protected setValue(ExecutionSlot slot, Object value, EObject target) {
		if (target.requiresStatechartScopeExecution) {
			interpreter.setValue(slot, value)
			return value
		} else {
			super.setValue(slot, value, target)
		}
	}
	

	def dispatch protected boolean requiresStatechartScopeExecution(Object it) {
		false
	}

	def dispatch protected boolean requiresStatechartScopeExecution(EObject it) {
		EcoreUtil2.getContainerOfType(it, Statechart) !== null
		|| EcoreUtil2.getContainerOfType(it, SCTUnitClass) === null
	}

	def dispatch protected boolean requiresStatechartScopeExecution(FeatureCall it) {
		owner.requiresStatechartScopeExecution
	}

	def dispatch protected boolean requiresStatechartScopeExecution(ElementReferenceExpression it) {
		reference.requiresStatechartScopeExecution
	}


	def dispatch Object execute(EnterExpression stm) {
		context.enter
		null
	}
	
	def dispatch Object execute(TriggerWithoutEventExpression stm) {
		context.triggerWithoutEvent
		null
	}

	def dispatch Object execute(ExitExpression stm) {
		context.exit
		null
	}

}
