/**
 * Copyright (c) 2022 itemis AG - All rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * 
 * Contributors:
 * 	Andreas Muelder - itemis AG
 * 	Jonathan Thoene - itemis AG
 */

package com.yakindu.sctunit.generator.python

import com.google.inject.Inject
import com.yakindu.base.expressions.expressions.ElementReferenceExpression
import com.yakindu.base.expressions.expressions.FeatureCall
import com.yakindu.base.types.Expression
import com.yakindu.sct.generator.core.extensions.StringHelper
import com.yakindu.sct.generator.core.library.ICoreLibraryHelper
import com.yakindu.sct.generator.python.files.PackageInit
import com.yakindu.sct.generator.python.naming.Naming
import com.yakindu.sct.model.sexec.extensions.SExecExtensions
import com.yakindu.sct.model.sgen.GeneratorEntry
import com.yakindu.sct.model.sgraph.Scope
import com.yakindu.sct.model.sgraph.Statechart
import com.yakindu.sct.model.stext.concepts.StatechartAnnotations
import com.yakindu.sct.model.stext.stext.InternalScope
import com.yakindu.sctunit.generator.base.ISCTUnitGenerator
import com.yakindu.sctunit.generator.base.extensions.BaseNavigationExtensions
import com.yakindu.sctunit.generator.python.extensions.CallbackExtensions
import com.yakindu.sctunit.generator.python.extensions.JUnitTest
import com.yakindu.sctunit.generator.python.extensions.NamingExtensions
import com.yakindu.sctunit.generator.python.extensions.PythonStatementExtensions
import com.yakindu.sctunit.generator.python.extensions.PythonVirtualTimerExtensions
import com.yakindu.sctunit.generator.python.extensions.UnittestMockingExtensions
import com.yakindu.sctunit.generator.python.features.PythonSCTUnitGenmodelEntries
import com.yakindu.sctunit.sCTUnit.SCTUnitClass
import com.yakindu.sctunit.sCTUnit.SCTUnitElement
import com.yakindu.sctunit.sCTUnit.SCTUnitOperation
import com.yakindu.sctunit.sCTUnit.SCTUnitSuite
import com.yakindu.sctunit.sCTUnit.TestPackage
import com.yakindu.sctunit.sCTUnit.VariableDefinitionStatement
import java.util.Collections
import java.util.List
import org.eclipse.emf.ecore.util.EcoreUtil
import org.eclipse.xtext.generator.IFileSystemAccess

import static com.yakindu.sct.generator.core.filesystem.ISCTFileSystemAccess.*

/**
 * 
 * @author andreas muelder - Initial contribution and API
 * @author Johannes Dicks - Refactoring based on EFS encapsulation
 * @author jonathan thoene - Expanding templates to generate mocking statements 
 * 
 */
class PythonSCTUnitGenerator implements ISCTUnitGenerator {

	public val static String SCTUNIT_INFERRER = "com.yakindu.sctunit.generator.python.PythonSCTUnitGenerator.typeInferrer"

	@Inject extension SExecExtensions
	@Inject extension StatechartAnnotations
	@Inject extension NamingExtensions
	@Inject extension PythonStatementExtensions
	@Inject extension UnittestMockingExtensions mockExt
	@Inject extension CallbackExtensions
	@Inject extension BaseNavigationExtensions
	@Inject extension PythonSCTUnitGenmodelEntries
	@Inject extension PythonVirtualTimerExtensions timer
	@Inject extension JUnitTest jUnitTest
	@Inject extension PythonSubchartInitializer
	@Inject extension Naming
	@Inject extension StringHelper
	@Inject extension ICoreLibraryHelper
	@Inject extension PackageInit

	GeneratorEntry entry

	override generate(GeneratorEntry entry, IFileSystemAccess fsa) {
		this.entry = entry;

		var testPackage = entry.testPackage
		var fileName = testPackage.fileName.toLowerCase
		val element = testPackage.SCTUnitElement
		var content = element.toCode
		
		generateJUnitWrapper(entry, testPackage, fsa)
		
		if (element instanceof SCTUnitClass) {
			if (element.statechart.needsTimer) {
				timer.generate(entry, fsa, "/vtimer/virtual_timer.py")
				fsa.generatePyPackageInit("/vtimer", LIBRARY_TARGET_FOLDER_OUTPUT, entry)
			}
		}
		fsa.generateFile(fileName, content)
	}
	
	protected def void generateJUnitWrapper(GeneratorEntry entry, TestPackage testPackage, IFileSystemAccess fsa) {
		if (entry.isJUnitWrapper) {
			var targetProject = entry.targetProjectValue.stringValue
			jUnitTest.generateJUnitTest(testPackage, fsa, targetProject.toPath)
		}
	}

	protected def getFileName(TestPackage it) {
		return '''«SCTUnitElement.testClassName.py»'''.toString.toLowerCase
	}
	
	protected def containsTestUnitsFromMultiplePackages(SCTUnitSuite suite){
		var String comperator = suite.SCTUnitClasses.get(0).statechartName
		for(SCTUnitClass uc : suite.SCTUnitClasses){
			if (!uc.statechartName.equals(comperator)){
				return true
			}
		}
		return false
	}

	protected def getTestPackage(GeneratorEntry entry) {
		return entry.elementRef.eContainer as TestPackage
	}

	def dispatch CharSequence toCode(SCTUnitSuite it) {
		'''
			«getLicenseText(entry)»
			
			# Suite: «testClassClassName»
			
			«IF containsTestUnitsFromMultiplePackages»import glob, string«ENDIF»
			import unittest
			import warnings
			
			«IF containsTestUnitsFromMultiplePackages»
				«createSuiteBodyMultiplePackages»
			«ELSE»
				«createSuiteBodySinglePackage»
			«ENDIF»

		'''
	}
	
	protected def createSuiteBodySinglePackage(SCTUnitSuite it)'''
		
		«FOR group : SCTUnitClasses»
			from «group.testClassName.toLowerCase» import «group.testClassClassName»TestCase
		«ENDFOR»
		
		def run_suite():
			suite = unittest.TestSuite()
			result = unittest.TestResult()
			«FOR group : SCTUnitClasses SEPARATOR ','»
				suite.addTest(unittest.makeSuite(«group.testClassClassName»TestCase))
			«ENDFOR»
			runner = unittest.TextTestRunner(verbosity=2)
			print(runner.run(suite))
			
		run_suite()
	'''
	
	protected def createSuiteBodyMultiplePackages(SCTUnitSuite it)'''
		# testcases listed / specified in SCT Unit file:
		target_tests = [\
			«FOR group : SCTUnitClasses»
				'«group.statechartName».tests.«group.testClassName»',\
			«ENDFOR»
			]
		# recursive search for all testcases in current directory:
		found_tests_raw = glob.glob('**/test_*.py', recursive=True)
		found_tests = [test_file[0:len(test_file)-3].replace('\\','.') for test_file in found_tests_raw]
		# match testcases found with user specified testcases:
		testcases = set(target_tests).intersection(found_tests)
		print('Number of TestCases specified by user: ', len(target_tests))
		print('Number of TestCases found by search: ', len(found_tests))
		print('Number of matching TestCases: ', len(testcases))
		
		with warnings.catch_warnings(record=False):
		    suites = [unittest.defaultTestLoader.loadTestsFromName(test_file) for test_file in testcases]
		    test_suite = unittest.TestSuite(suites)
		    test_runner = unittest.TextTestRunner(verbosity=2).run(test_suite)
	'''
	
	protected def createStatemachineImports(SCTUnitElement it, GeneratorEntry entry) '''
		import sys
		import os
		«IF it instanceof SCTUnitClass»
			from «entry.statechartBasePackage.dot(statemachineClassName)» import «statechartName.toFirstUpper»
		«ENDIF»
		«FOR subchart : getSCTUnitClass?.statechart.subchartImports.keySet»
			from «entry.getStatechartBasePackage(subchart).dot(subchart.statemachineClassName)» import «subchart.statemachineName»
		«ENDFOR»
		«IF getSCTUnitClass?.statechart.needsTimer»
			from «entry.statechartLibraryPackage.dot("vtimer.virtual_timer")» import VirtualTimer«IF getSCTUnitClass?.statechart.isCycleBased», CycleTimeEventTask«ENDIF»
		«ENDIF»
		sys.path.insert(0, os.path.abspath(os.path.join(os.path.dirname(__file__), '..')))
	'''

	protected def getAllScopesToMock(SCTUnitClass it) {
		statechart.scopes.filter[!operations.nullOrEmpty].toList
	}

	def dispatch CharSequence toCode(SCTUnitClass it) {
		var scopes = allScopesToMock
		val uninitializedVariableDefinitions = EcoreUtil.copyAll(variableDefinitions).toList
		uninitializedVariableDefinitions.forEach[definition.initialValue = null]
		'''
			«getLicenseText(entry)»
			
			import unittest
			«IF isMockingRequired || statechart.hasInternalOperation»
				from unittest import mock
			«ENDIF»
			
			«createStatemachineImports(entry)»
			
			# Unit TestCase for «statechart.name»
			class «testClassClassName»TestCase(unittest.TestCase):
				«FOR variable: uninitializedVariableDefinitions»
					«variable.generate»
				«ENDFOR»
				
				
				«setup»
			
				def tearDown(self):
					«FOR Scope scope : scopes»
						«scope.unsetOperationCallback»
					«ENDFOR»
					«IF (statechart.isCycleBased || statechart.timed)»
						self.timer_service.stop()
					«ENDIF»
					self.«statemachineReferenceName».exit()
				
				«FOR operation : SCTUnitOperations»
					«operation.toCode»
				«ENDFOR»
		'''
	}

	def dispatch CharSequence toCode(SCTUnitOperation it) {
		if(isTest)
			mockExt.clearMockedOperations
		'''
			«IF isIgnore»@unittest.skip("Ignored. See related .sctunit file.")«ENDIF»
			def «IF it.isTest»test_«ENDIF»«methodName»(self«IF !parameters.empty», «ENDIF»«generateParameter»):
				«generateSideEffectList.generateSideEffects»
				«FOR  variable: SCTUnitClass.variableDefinitions»
					global «variable.definition.identifier»
				«ENDFOR»
				«IF body.code.nullOrEmpty»
				pass
				«ENDIF»
				«FOR statement : body.code»
					«statement.generate»
				«ENDFOR»
				
		'''
	}
	
	protected def dispatch List<Expression> argumentExpressions(Expression it){}
	
	protected def dispatch List<Expression> argumentExpressions(ElementReferenceExpression it){
		if(expressions.nullOrEmpty){
			return Collections.emptyList
		}else {
			return expressions
		}
	}
	
	protected def dispatch List<Expression> argumentExpressions(FeatureCall it){
		if(expressions.nullOrEmpty){
			return Collections.emptyList
		}else {
			return expressions
		}
	}

	def dispatch CharSequence toCode(List<VariableDefinitionStatement> varDefs) '''
		«FOR varDef : varDefs»
			«varDef.toCode»
		«ENDFOR»
	'''		

	
	def dispatch CharSequence toCode(VariableDefinitionStatement it) '''
		«generate»
	'''
 	
 	protected def hasInternalOperation(Statechart statechart){
 		var InternalScope internalScope = null
 		for(s : statechart.scopes){
			if(s instanceof InternalScope){
				internalScope = s
			}
		}
		if(internalScope !== null){
			if(internalScope.hasOperations){
				return true
			}	
		}
		return false
 	}
 	
	protected def setup(SCTUnitClass it) throws Exception{
		var scopes = allScopesToMock
		
		'''
			def setUp(self):
				«FOR scope : scopes»		
					«scope.defaultMock»
				«ENDFOR»
				self.«statemachineReferenceName» = «statechartName.toFirstUpper»()
				«IF (statechart.needsTimer)»
					self.timer_service = VirtualTimer(«cyclePeriod»)
				«ENDIF»
				«IF statechart.isCycleBased»
					self.timer_service.schedule_periodical_task(CycleTimeEventTask(self.«statemachineReferenceName»), «cyclePeriod», «cyclePeriod»)
				«ENDIF»
				«IF statechart.isTimed»
					self.«statemachineReferenceName».timer_service = self.timer_service
				«ENDIF»
				«FOR Scope scope : scopes»
					«scope.mockOperationCallbacks»
				«ENDFOR»
				«FOR variable: variableDefinitions»
					global «variable.definition.identifier»
					«variable.generate»
				«ENDFOR»
				«statechart.subchartInitSequence»
		'''
	}

	protected def generateParameter(SCTUnitOperation it){
		'''«FOR param:parameters SEPARATOR ', '»«param.identifier»«ENDFOR»'''
	}

}
