/**
 * Copyright (c) 2022 itemis AG - All rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * 
 * Contributors:
 * 	Andreas Muelder - itemis AG
 *  Jonathan Thoene - itemis AG
 * 
 */

package com.yakindu.sctunit.generator.c.extensions

import com.google.inject.Inject
import com.google.inject.name.Named
import com.yakindu.base.expressions.expressions.ElementReferenceExpression
import com.yakindu.base.types.inferrer.ITypeSystemInferrer
import com.yakindu.sct.generator.c.extensions.FileNaming
import com.yakindu.sct.generator.c.extensions.Naming
import com.yakindu.sct.generator.c.types.CTypes
import com.yakindu.sct.generator.core.types.ICodegenTypeSystemAccess
import com.yakindu.sct.model.sexec.ExecutionFlow
import com.yakindu.sct.model.sexec.concepts.RunCycleMethod
import com.yakindu.sct.model.sexec.naming.INamingService
import com.yakindu.sct.model.sexec.transformation.config.IFlowConfiguration
import com.yakindu.sct.model.sgen.GeneratorEntry
import com.yakindu.sct.model.sgraph.Statechart
import com.yakindu.sct.model.stext.concepts.StatechartAnnotations
import com.yakindu.sct.model.stext.stext.OperationDefinition
import com.yakindu.sctunit.generator.c.features.CSCTUnitGenmodelEntries
import com.yakindu.sctunit.sCTUnit.MockReturnStatement
import com.yakindu.sctunit.sCTUnit.SCTUnitClass
import com.yakindu.sctunit.sCTUnit.SCTUnitOperation
import org.eclipse.xtext.EcoreUtil2
import org.eclipse.xtext.generator.IFileSystemAccess

/**
 * 
 * @author Markus Muehlbrandt - Initial contribution and API
 * @author Jonathan Thoene - Expanding templates to generate mocking statements
 * @author Axel Terfloth - Make base types pluggable
 * 
 */
class GTest {

	public val static String SCTUNIT_INFERRER = "com.yakindu.sctunit.generator.c.CSCTUnitGenerator.typeInferrer"

	@Inject extension SCTUnitCNaming
	@Inject extension ICodegenTypeSystemAccess
	@Inject extension CNavigationExtensions
	@Inject extension CStatementExtensions
	@Inject extension CMockingExtensions
	@Inject extension CSCTUnitGenmodelEntries
	@Inject extension INamingService
	@Inject extension Naming
	@Inject extension FileNaming
	@Inject extension StatechartAnnotations
	@Inject
	@Named(SCTUNIT_INFERRER)
	extension ITypeSystemInferrer;
	@Inject(optional=true) ExecutionFlow flow

	@Inject protected GeneratorEntry entry
	@Inject extension CSubchartInitializer
	@Inject extension IFlowConfiguration
	@Inject extension RunCycleMethod
	@Inject extension GenerationHelper
	
	@Inject protected extension CTypes

	def generateGTest(SCTUnitClass it, IFileSystemAccess fsa, String outletFolder) {
		var gTestFileName = '''«testClassName.cc»'''.toString
		var content = generateGTest
		fsa.generateFile(gTestFileName, content)
	}

	def private generateGTest(SCTUnitClass it) {
		defineConfigurationForStatechart(statechart)
		initializeNamingService(statechart)
		'''
			«entry.licenseText»
			
			«IF isMockingRequired»
				#include <algorithm>
				#include <list>
			«ENDIF»
			#include "gtest/gtest.h"
			#include "«IF entry.identifierModuleName.nullOrEmpty»«statechart.module.h»«ELSE»«entry.identifierModuleName.h»«ENDIF»"
			
			«IF statechartHasOperations(statechart) || isTimed»#include "«IF entry.identifierModuleName.nullOrEmpty»«statechart.module.client.h»«ELSE»«entry.identifierModuleName.client.h»«ENDIF»"«ENDIF»
			«FOR submachine : getTimedSubmachines(statechart)»
			#include "«submachine.module.client.h»"
			«ENDFOR»
			#include "«sc_timer_service.h»"
			
			#define SC_UNUSED(P) (void)P
			
			static «statechart.type» «statechartNaming»;
			
			«FOR vardef : variableDefinitions»
				static «vardef.generate»
			«ENDFOR»

			«generateTestFixtureClass»
			
			static «testClassName» * «testClassObject»;
			
			«IF isMockingRequired»
				«generateOperationMockClasses»
			«ENDIF»
			
			«generateTestFixtureClassDefinition»
			
			«FOR operation : allTestOps»
			TEST_F(«it.testClassName», «IF operation.isIgnore»DISABLED_«ENDIF»«operation.name») {
				«operation.testOperationName»();
			}
			«ENDFOR»		
			
			«IF isTimed»«statechart.timerServiceFuncs»«ENDIF»
			
			«IF statechart.hasTimedSubmachines»«statechart.submachineTimerServiceFuncs»«ENDIF»
			
			«generateRequiredOperations»
		'''
	}

	def protected generateSCUnitMethodDefinitions(SCTUnitClass it) {
		'''
			«FOR op : SCTUnitOperations.filter[(isTestOperationCalledFromOthers || !isTest)]»
				«op.infer.type.targetLanguageName» «op.name»(«op.generateParams»);
			«ENDFOR»
		'''
	}


	def generateMocks(SCTUnitOperation it) {
		'''
			«FOR op : getAllOperations(allMockStatementsInOperation)»
				«op.initializeOperationMock(SCTUnitClass.isOperationForStatement(op, typeof(MockReturnStatement)))»
			«ENDFOR»
			
		'''
	}

	def protected generateMethodBody(SCTUnitOperation it) {
		'''	
			«FOR statement : body.code»
				«statement.generate»
			«ENDFOR»
		'''
	}

	def private generateParams(SCTUnitOperation it) {
		'''«FOR param : parameters SEPARATOR ', '»«param.typeSpecifier.targetLanguageName» «param.name»«»«ENDFOR»'''
	}
	
	def private timerServiceFuncs(Statechart it) {
		'''
			void «prefix»_set_timer(«type»* statechart, const «sc_eventid.fqName» evid, const «sc_time.fqName» time_ms, const «sc_bool.fqName» periodic){
				tc->set_timer(statechart, evid, time_ms, periodic);
			}
			
			void «prefix»_unset_timer(«type»* handle, const «sc_eventid.fqName» evid){
				tc->unset_timer(handle, evid);
			}
		'''
	}
	
	def private submachineTimerServiceFuncs(Statechart it) {
		'''
		«FOR submachine : getTimedSubmachines»
			void «submachine.type.toString.toFirstLower»_set_timer(«submachine.type»* statechart, const «sc_eventid.fqName» evid, const «sc_time.fqName» time_ms, const «sc_bool.fqName» periodic){
				tc->set_timer(statechart, evid, time_ms, periodic);
			}
			
			void «submachine.type.toString.toFirstLower»_unset_timer(«submachine.type»* handle, const «sc_eventid.fqName» evid){
				tc->unset_timer(handle, evid);
			}
		«ENDFOR»
		'''
	}

	def private generateTestFixtureClass(SCTUnitClass it) {
		'''	
			class «testClassName» : public ::testing::Test
			{
			public:
				/* All operations from the SCTUnit test class. */
				«FOR op : SCTUnitOperations»
					«op.infer.type.targetLanguageName» «op.testOperationName»(«op.generateParams»);
				«ENDFOR»
				void set_timer(«statechart.type»* statechart, const «sc_eventid.fqName» evid, const «sc_time.fqName» time_ms, const «sc_bool.fqName» periodic);
				void unset_timer(«statechart.type»* handle, const «sc_eventid.fqName» evid);
				«FOR submachine : getTimedSubmachines(statechart)»
					void set_timer(«submachine.type»* statechart, const «sc_eventid.fqName» evid, const «sc_time.fqName» time_ms, const «sc_bool.fqName» periodic);
					void unset_timer(«submachine.type»* handle, const «sc_eventid.fqName» evid);
				«ENDFOR»
			protected:
				sc_unit_timer_service_t *timer_service;
				«statechart.subchartMemberVariables»
				virtual void SetUp();
				virtual void TearDown();
			};
		'''
	}
	
	def private generateTestFixtureClassDefinition(SCTUnitClass it) {
		'''	
			«IF isTimed || hasTimedSubmachines(statechart)»
			static void dispatchTimeEvent(void* handle, «sc_eventid.fqName» evid)
			{
				«IF isTimed»
				if(handle == &«statechartNaming»){
					«naming.raiseTimeEventFctID(flow)»(&statechart, evid);
				}«ENDIF»
				«FOR submachine : statechart.getTimedSubmachines»
				if(handle == «statechart.submachineMap.get(submachine)»){
					«naming.raiseTimeEventFctID(submachine)»(«statechart.submachineMap.get(submachine)», evid);
				}
				«ENDFOR»
			}
			
			«ENDIF»
			void «testClassName»::SetUp()
			{
				«naming.initFctID(flow)»(&«statechartNaming»);
				«statechart.subchartInitSequence»
				«generateTimerServiceInit»
				
				«FOR op : getAllOperations(allMockStatementsInClass)»
				«op.initializeOperationMock(it.isOperationForStatement(op, typeof(MockReturnStatement)))»
				«ENDFOR»
				
				tc = this;
			}
			
			void «testClassName»::TearDown()
			{
				
				«FOR op : getAllOperations(allMockStatementsInClass).toList.reverse»
				«op.freeOperationMock»
				«ENDFOR»
				«generateTimerServiceFree»
				«statechart.subchartTeardownSequence»
			}
			
			«FOR op : SCTUnitOperations»
			«op.infer.type.targetLanguageName» «testClassName»::«op.testOperationName»(«op.generateParams»)
			{
				«op.generateMethodBody»
			}
			«ENDFOR»
			
			void «testClassName»::set_timer(«statechart.type»* statechart, const «sc_eventid.fqName» evid, const «sc_time.fqName» time_ms, const «sc_bool.fqName» periodic){
				sc_unit_timer_t timer;
				sc_unit_timer_init(&timer, time_ms, periodic, evid, statechart);
				insert_timer(tc->timer_service, timer);
			}
			
			void «testClassName»::unset_timer(«statechart.type»* handle, const «sc_eventid.fqName» evid){
				SC_UNUSED(handle);
				delete_task(tc->timer_service, find_time_event(tc->timer_service, evid));
			}
			«FOR submachine : statechart.timedSubmachines»
			
			void «testClassName»::set_timer(«submachine.type»* statechart, const «sc_eventid.fqName» evid, const «sc_time.fqName» time_ms, const «sc_bool.fqName» periodic){
				sc_unit_timer_t timer;
				sc_unit_timer_init(&timer, time_ms, periodic, evid, statechart);
				insert_timer(tc->timer_service, timer);
			}
			
			void «testClassName»::unset_timer(«submachine.type»* handle, const «sc_eventid.fqName» evid){
				SC_UNUSED(handle);
				delete_task(tc->timer_service, find_time_event(tc->timer_service, evid));
			}
			«ENDFOR»
			'''
	}
	
	def protected generateTimerServiceInit(SCTUnitClass it) {
		var eventBased = it.statechart.isEventDriven
		var cyclePeriod = it.statechart.cyclePeriod
		'''
			this->timer_service = (sc_unit_timer_service_t*)malloc(sizeof(sc_unit_timer_service_t));
			sc_unit_timer_service_init(
				this->timer_service,
				«IF isTimed || hasTimedSubmachines(statechart)»
					(sc_raise_time_event_fp) dispatchTimeEvent,
				«ELSE»
					0,
				«ENDIF»
				«IF hasRunCycleFunctionAsAPI(flow)»
				(sc_run_cycle_fp) &«naming.runCycleFctID(flow)»,
				«ELSE»
				NULL,
				«ENDIF»
				«IF eventBased»true«ELSE»false«ENDIF»,
				«cyclePeriod»,
				&«statechartNaming»
			);
		'''
	}
	
	def protected generateTimerServiceFree(SCTUnitClass it) {
		'''
			sc_unit_timer_service_free(
				timer_service
			);
		'''
	}

	def protected statechartHasOperations(Statechart it) {
		return !EcoreUtil2.getAllContentsOfType(it, typeof(OperationDefinition)).nullOrEmpty
	}

	def protected boolean isTestOperationCalledFromOthers(SCTUnitOperation it) {
		val clazz = EcoreUtil2.getContainerOfType(it, typeof(SCTUnitClass))
		var elemRefs = EcoreUtil2.getAllContentsOfType(clazz, typeof(ElementReferenceExpression))
		val allOpRefs = elemRefs.filter[ref|ref.reference instanceof SCTUnitOperation].map[ref|ref.reference].toList
		if (allOpRefs.contains(it)) {
			return true
		}
		return false;
	}
}
