/**
 * Copyright (c) 2020-2026 itemis AG - All rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * 
 */
package com.yakindu.sct.simulation.core.sexec.interpreter

import com.google.common.collect.Sets
import com.google.inject.Inject
import com.itemis.create.base.model.bindings.PropertyChangedNotification
import com.itemis.create.base.model.core.EventModel
import com.yakindu.base.base.NamedElement
import com.yakindu.base.expressions.expressions.ArgumentExpression
import com.yakindu.base.expressions.expressions.ElementReferenceExpression
import com.yakindu.base.expressions.expressions.FeatureCall
import com.yakindu.base.expressions.expressions.MetaCall
import com.yakindu.base.expressions.expressions.ThisExpression
import com.yakindu.base.expressions.interpreter.IOperationExecutor
import com.yakindu.base.expressions.interpreter.base.EventSemantics
import com.yakindu.base.expressions.interpreter.base.IInstanceFactory
import com.yakindu.base.expressions.interpreter.base.IInterpreter
import com.yakindu.base.expressions.interpreter.base.InterpreterException
import com.yakindu.base.expressions.interpreter.base.PropertySemantics
import com.yakindu.base.expressions.interpreter.base.ResolvingRuntimeInstance
import com.yakindu.base.expressions.interpreter.base.SRuntimeInterpreter.EventInstance
import com.yakindu.base.expressions.interpreter.base.ValueSemantics
import com.yakindu.base.types.ComplexType
import com.yakindu.base.types.Declaration
import com.yakindu.base.types.Direction
import com.yakindu.base.types.EnumerationType
import com.yakindu.base.types.Enumerator
import com.yakindu.base.types.Event
import com.yakindu.base.types.Expression
import com.yakindu.base.types.Operation
import com.yakindu.base.types.Part
import com.yakindu.base.types.Property
import com.yakindu.base.types.TypeBuilder
import com.yakindu.base.types.adapter.OriginTracing
import com.yakindu.base.types.typesystem.ITypeSystem
import com.yakindu.base.types.typesystem.ITypeValueProvider
import com.yakindu.sct.model.sexec.Call
import com.yakindu.sct.model.sexec.EnterState
import com.yakindu.sct.model.sexec.ExecutionFlow
import com.yakindu.sct.model.sexec.ExecutionState
import com.yakindu.sct.model.sexec.ExitState
import com.yakindu.sct.model.sexec.HistoryEntry
import com.yakindu.sct.model.sexec.Method
import com.yakindu.sct.model.sexec.SaveHistory
import com.yakindu.sct.model.sexec.Sequence
import com.yakindu.sct.model.sexec.StateSwitch
import com.yakindu.sct.model.sexec.Step
import com.yakindu.sct.model.sexec.Trace
import com.yakindu.sct.model.sexec.concepts.EnterMethod
import com.yakindu.sct.model.sexec.concepts.EventProcessing
import com.yakindu.sct.model.sexec.concepts.EventQueue
import com.yakindu.sct.model.sexec.concepts.ExitMethod
import com.yakindu.sct.model.sexec.concepts.InitializedCheck
import com.yakindu.sct.model.sexec.concepts.MicroStep
import com.yakindu.sct.model.sexec.concepts.StateMachineBehaviorConcept
import com.yakindu.sct.model.sexec.concepts.SubMachine
import com.yakindu.sct.model.sexec.concepts.SubMachine.SubmachineTypeLibrary
import com.yakindu.sct.model.sexec.concepts.SuperStep
import com.yakindu.sct.model.sexec.extensions.SExecExtensions
import com.yakindu.sct.model.sgraph.FinalState
import com.yakindu.sct.model.sgraph.RegularState
import com.yakindu.sct.model.sgraph.Statechart
import com.yakindu.sct.model.sgraph.naming.SGraphShortNameProvider
import com.yakindu.sct.model.sruntime.CompositeSlot
import com.yakindu.sct.model.sruntime.ExecutionContext
import com.yakindu.sct.model.sruntime.ExecutionEvent
import com.yakindu.sct.model.sruntime.ExecutionSlot
import com.yakindu.sct.model.sruntime.ReferenceSlot
import com.yakindu.sct.model.stext.concepts.StatechartAnnotations
import com.yakindu.sct.model.stext.stext.ActiveStateReferenceExpression
import com.yakindu.sct.model.stext.stext.OperationDefinition
import com.yakindu.sct.model.stext.stext.SubmachineReferenceExpression
import com.yakindu.sct.simulation.core.util.ExecutionContextExtensions
import java.util.ArrayDeque
import java.util.ArrayList
import java.util.Map
import java.util.Queue
import java.util.Set
import org.eclipse.emf.ecore.EObject

import static extension org.eclipse.xtext.EcoreUtil2.*

/**
 * This instance implementation cares about the execution of an ExecutionFlow which defines the type of the instance.
 * 
 * @author axel terfloth
 */
class ExecutionFlowInstance extends ResolvingRuntimeInstance implements IInterpreter.Instance {
	
	protected ExecutionFlow type
	
	protected ExecutionState[] activeStateConfiguration
	protected Map<Integer, ExecutionState> historyStateConfiguration
	protected int activeStateIndex
	
	protected Queue<EventInstance> internalEventQueue = null 
	protected Queue<EventInstance> inEventQueue = null
	
	@Inject protected extension StateMachineBehaviorConcept
	@Inject protected extension ExecutionContextExtensions
	@Inject protected extension StatechartAnnotations
	@Inject(optional=true) protected ITraceStepInterpreter traceInterpreter
	@Inject(optional=true) protected Set<IOperationExecutor> operationExecutors = Sets.newHashSet
	@Inject protected extension TypeBuilder
	@Inject protected extension ITypeValueProvider
	@Inject protected extension SExecExtensions
	@Inject protected extension PropertySemantics
	@Inject protected extension EventSemantics
	@Inject protected extension ValueSemantics
	@Inject protected extension EventModel
	@Inject protected extension EventQueue
	@Inject protected extension SubMachine
	@Inject extension SGraphShortNameProvider
	@Inject extension OriginTracing
	@Inject protected extension PropertyChangedNotification
	@Inject protected extension ITypeSystem
	
	@Inject protected extension SubmachineTypeLibrary
	
	protected IInstanceFactory factory
	
	
	def void setUp(CompositeSlot instance, IInterpreter.Context context, ExecutionFlow flow, IInstanceFactory factory) {
		super.setUp(instance, context)
		
		this.type = flow
		this.factory = factory
		
		if(flow.requiresIncomingEventQueue) {
			this.inEventQueue = new ArrayDeque<EventInstance>
		}
		if (flow.requiresInternalEventQueue) {
			this.internalEventQueue = new ArrayDeque<EventInstance>
		}
	}
	
	def _executeNamedSequence(Step it) {
		instanceSlot._executeOperation('_' + (eContainer.eContainer as NamedElement)?.name + "." + (eContainer as NamedElement)?.name + "." + name + '_', #[],[
			_delegate			
		])		
	}	
	
	override provideExecution(Object program) {
		switch (program) {

			EnterState : program.execution
			ExitState : program.execution
			StateSwitch : program.execution
			SaveHistory : program.execution
			HistoryEntry : program.execution
			Trace : program.execution
			MetaCall : program.execution
			
			ActiveStateReferenceExpression : program.execution
			
			Call : _executeNamedSequence(program.step)				

			Sequence case program.name == EventProcessing.CLEAR_EVENT : clearEvent(program.expression )			
			Sequence case program.name == EventProcessing.MOVE_EVENT : moveEvent(program.expression, program.getExpression(1))
			
			Invokation case (program.operation as Operation).name == EventProcessing.NEXT_EVENT : nextEvent(true)		
			Sequence case (program.name == EventProcessing.NEXT_EVENT) : nextEvent(false)	
				
			Sequence case program.name == EnterMethod.TRACE_ENTER : _execute[]		// TODO: extend for YET tracing
			Sequence case program.name == ExitMethod.TRACE_EXIT : _execute[
				// Clear executed elements so transition highlights are removed when exiting a submachine
				executionContextSlot.executedElements.clear
			]		// TODO: extend for YET tracing	
			Sequence case program.name == InitializedCheck.INIT_CHECK : _execute[]	// do nothing so far ...		
						
			Invokation : program.caller.doInvoke(program.operation)
			 
			Method case type.allMethods.contains(program): program.doInvoke 
						
			String case program.hasMethod : program.lookupMethod.doInvoke 
			
			case 'init': init
			case 'isActive': isActive
			case 'isFinal': isFinal
			case 'isStateActive': isStateActive
			ExecutionEvent case program.name == 'triggerWithoutEvent' : "runCycle"._exec

			ThisExpression : doProvideThis

			default : 
				throw new IllegalArgumentException("Cannot execute '" + program +"'")
		}
	}


	def void init() {
		
		instanceSlot._executeOperation( '''«(instanceSlot as ExecutionSlot).name».init''', #[], [
			activeStateConfiguration = newArrayOfSize(type.stateVector.size)
			activeStateIndex = 0
			historyStateConfiguration = newHashMap()

			_executeNamedSequence( type.staticInitSequence )
			_executeNamedSequence( type.initSequence )
			
			type.allFeatures.forEach[
				it.initMember
			]
			
			executionContextSlot.connectOutEvents
		])
	}
	
		def protected dispatch void initMember(Declaration it){
		// do nothing by default
	}
	
	def protected dispatch void initMember(Property p) {
		
		if ( p.initialValue !== null) {
	
			val slot = instanceSlot.resolve(p.name) as ExecutionSlot
	
			p.initialValue._delegate
			_value
			
			_execute('''initialize «slot.fqName»''', [
				var value = popValue
				_valueSemantics.setValue(slot, value)
			])
		}
	}
	
	
	def protected dispatch void initMember(Part p){
		
		val slot = instanceSlot.resolve(p.name) as ExecutionSlot
		
		if ( slot instanceof ReferenceSlot ) {
		
			_execute('''new part «slot.fqName»''', [
				val part = factory.newInstance(p.type) => [
					if (it instanceof ExecutionSlot) it.fqName = slot.fqName
				]
				slot.reference = part as ExecutionSlot
			])
			
		} else if (slot instanceof CompositeSlot) {
			_initMember(p as Property)
		}
	}
	
	
	def void isActive() {
		
		
		instanceSlot._executeOperation( "isActive", #[], [

			val isActive = ! activeStateConfiguration
								.filterNull
								.empty

			exitCall(isActive)			
		])
	}


	def void isStateActive() {
		
		
		instanceSlot._executeOperation( "isStateActive", #["stateName"], [

			val stateId = context.resolve(null, "stateName").get
			val stateName = switch (stateId) {
				        case null : null
				String            : stateId
				Enumerator		  : (stateId.origin as RegularState).qualifiedName.toString			
			}
			
			var isActive = switch (stateName) {
				case null : activeStateConfiguration.exists[it === null]
				String    : {
					activeStateConfiguration
					.filterNull
					.exists[
						(it.sourceElement as RegularState).qualifiedName.toString == stateName 
						||
						it.sourceElement.allContainers.filter(RegularState).exists[n | n.qualifiedName.toString == stateName]
					]	
				}
			}

			exitCall(isActive)			
		])
	}
	
	def create new ArrayList<RegularState> allStates(ExecutionFlow flow) {
		
		it.addAll((flow.sourceElement as Statechart)
					.eAllContents
					.filter(RegularState)
					.filter[!name.isNullOrEmpty]
					.toList
		)
	}
	
	def void isFinal() {
		
		
		instanceSlot._executeOperation( "isFinal", #[], [

			val activeStates = activeStateConfiguration.filterNull.toList
			
			var isFinal = 	
				if (activeStates.empty) false
				else {
					activeStates.forall[
						it.sourceElement instanceof FinalState
					]
				}
			
			exitCall(isFinal)			
		])
	}

	def protected dispatch void execution(Object it) {
		throw new IllegalArgumentException("Cannot execute '" + it + "'")
	}

	def protected dispatch void execution(EnterState enterState) {
		_execute ("enter(" + enterState.state.name + ")", [
			val execState = enterState.state
			val firstPos = execState.statevectorFirstPosition
			val lastPos = execState.statevectorLastPosition
			for ( i : firstPos .. lastPos) {
				activeStateConfiguration.set(i, execState)
			}
			activeStateConfiguration.syncActiveStates
			val positionSlot = instanceSlot.resolve(MicroStep.STATE_VECTOR_POSITION)
			if ( positionSlot !== null ) _valueSemantics.setValue(positionSlot, firstPos )		
			val changedSlot = instanceSlot.resolve(SuperStep.STATE_CHANGED)
			if ( changedSlot !== null ) _valueSemantics.setValue(changedSlot, true)		
		])
	}

	def protected dispatch void execution(ExitState exitState) {
		_execute ("exit(" + exitState.state.name + ")", [
			val execState = exitState.state
			val firstPos = execState.statevectorFirstPosition
			val lastPos = execState.statevectorLastPosition
			
			val parentExecState = execState.superScopes.filter(ExecutionState).head
			
			for ( i : firstPos .. lastPos) {
				activeStateConfiguration.set(i, parentExecState)
			}			
			val positionSlot = instanceSlot.resolve(MicroStep.STATE_VECTOR_POSITION)
			if ( positionSlot !== null ) _valueSemantics.setValue(positionSlot, lastPos )
			
			activeStateConfiguration.syncActiveStates
		])
	}

	def protected syncActiveStates(ExecutionState[] it) {
		val activeVectorStates = it.filter[ it !== null].map[sourceState(sourceElement)].toSet
		val activeStates = executionContextSlot.activeStates
		
		activeStates.clear
		activeStates.addAll(activeVectorStates)
	}

	def protected dispatch RegularState sourceState(ExecutionState execState, RegularState source) {
		return source
	}

	def protected dispatch RegularState sourceState(ExecutionState execState, SubmachineReferenceExpression source) {
		val superScope = execState.superScope
		if(superScope instanceof ExecutionState) {
			return superScope.sourceState(superScope.sourceElement)
		}
		return null
	}

	def protected dispatch void execution(StateSwitch stateSwitch) {
		_execute ("stateSwitch", [
			val historyRegion = stateSwitch.historyRegion
			if (historyRegion !== null) {
				val historyState = historyStateConfiguration.get(historyRegion.historyVector.offset)
				stateSwitch.cases.filter[it.state == historyState].head?.step?._delegate
			} else {
				stateSwitch.cases.filter[activeStateConfiguration.contains(state)].head?.step?._delegate
			} 
		])
	}

	def protected dispatch void execution(SaveHistory it) {
		_execute ("saveHistory(" + it.region.name + ")", [
			var region = it.region
			historyStateConfiguration.put(region.historyVector.offset,
				activeStateConfiguration.get(region.stateVector.offset))
		])
	}

	def dispatch void execution(HistoryEntry entry) {
		_execute("histoiryEntry", [
			if (historyStateConfiguration.get(entry.region.historyVector.offset) !== null) {
				entry.historyStep?._executeNamedSequence
			} else {
				entry.initialStep?._executeNamedSequence
			}			
		])
	}

	def dispatch void execution(ActiveStateReferenceExpression it) {
		_return("isActive(" + value.name + ")", [
			val state = it.value			
			return executionContextSlot.allActiveStates.contains(state)
		])
	}

	def dispatch void execution(Trace it) {
		if ( traceInterpreter !== null) {
			_execute('''trace «it.class.simpleName»''', [
				traceInterpreter?.evaluate(it, executionContextSlot)
			])	
		}
	}

	def dispatch void execution(MetaCall it) {
		val metaFeature = featureOrReference
		val metaFeatureName = (metaFeature instanceof NamedElement) ? metaFeature.name : "<unnamed>"
		val subject = it.owner.featureOrReference
		val subjectName = (subject instanceof NamedElement) ? subject.name : "<unnamed>"
		
		executeMetaCall(subject, metaFeature)
	}

	def dispatch void executeMetaCall(MetaCall metaCall, EObject subject, EObject metaOp) {
		throw new IllegalArgumentException('''Cannot handle meta call «subject.class.simpleName»:«metaOp.class.simpleName».''')
	}

	def dispatch void executeMetaCall(MetaCall metaCall, Event subject, Operation metaOp) {
		if (metaOp.implementation !== null) {
			// TODO: ignore for now - add as soon we have to process meta ops with implementation
			// TODO: should be delegated to EventInstance
		} else {
			if (storeEventConcept.appliesTo(metaOp)) { // TODO : add variants for out event, internal event, w/wo buffer etc.
				metaCall.owner._delegate
				_execute('''«subject.name»:store''', [
					val slot = popValue()
					if (slot instanceof ExecutionEvent) {
						if (slot.direction == Direction::IN && inEventQueue !== null) {
							inEventQueue.add(new EventInstance(slot, null)); // TODO: extend for payloads
						}
					}
				])		
			}
		}
	}

	def protected void clearEvent(Expression event) {
		event._delegate
		_execute("@clearEvent()", [
			val Object eventSlot = popValue()
			(eventSlot as ExecutionEvent).raised = false
		])		
	}
	
	
	def protected void moveEvent(Expression from, Expression to) {
		from._delegate
		to._delegate
		_execute("@moveEvent()", [
			val Object toSlot = popValue()
			val Object fromSlot = popValue()
			
			(toSlot as ExecutionEvent).raised = (fromSlot as ExecutionEvent).isRaised()
			toSlot.setValue(fromSlot)
			(fromSlot as ExecutionEvent).raised = false
		])		
	}
		
		
	def protected void nextEvent(boolean returnResult) {
		_execute("@nextEvent()", [
			var EventInstance event = internalEventQueue?.poll ?: inEventQueue?.poll
			if ( event !== null ){
				val eventBehavior = event.event.eventInstance
				if (eventBehavior !== null) {
					eventBehavior.activate(event.value)
				} else {
					event.event.raised = true
					_valueSemantics.setValue(event.event, event.getValue())
				}
				if(returnResult) _return(event.event.fqName + " is next", [true])
			} else {
				if(returnResult) _return("no event next", [false])				
			}
		])		
	}

		
	def protected executionContextSlot(){
		return instanceSlot as ExecutionContext
	}
	
	
	//=========================================================================================
	//
	// Handling invocations
	//
	
	def dispatch void invoke(Object slot, Object operation) {
		operation.doInvoke
	}
	
	def dispatch void invoke(ReferenceSlot slot, Object operation) {
		slot.reference.instance.invoke(slot.reference, operation)
	}
	
	
	def dispatch void doInvoke(Object operation) {
		throw new InterpreterException("Cannot invoke operation: " + operation)
	}
	
	def dispatch void doInvoke(Void void) {
		throw new InterpreterException("Cannot invoke null operation")
	}
	
	def dispatch void doInvoke(Operation it) {
		val impl = (it instanceof Method) ? it.body : it.implementation
		if (impl !== null)  {
			_executeOperation('''«(instanceSlot as ExecutionSlot).name».«name»''', instanceSlot, name, parameters.map[name], [
				impl._delegate
			])
		}
		else {
			executeMock
		}
	}
	
	def protected void executeImplementation(Operation it) {
		if (it?.implementation !== null)  {
			_executeOperation('''«(instanceSlot as ExecutionSlot).name».«name»''', instanceSlot, name, parameters.map[name], [
				if(it instanceof Method) body._delegate
				else implementation._delegate
			])
		}
	}
	
	
	def protected void executeMock(Operation it) {
		_executeOperation('''mock «(instanceSlot as ExecutionSlot).name».«name»''', instanceSlot, name, parameters.map[name], [
			val returnType = it.typeSpecifier.type
			if (! returnType.isVoid) {
				_return[
					if (returnType instanceof ComplexType && ! (returnType instanceof EnumerationType)) {
						factory.newInstance(it.typeSpecifier)
					}
					else {
						type.defaultValue
					}
				]
			}
		])
	}
	
	def protected invokeWithCaller(Operation it, ArgumentExpression expr){
		if(validateCallerIsOwner(expr))
			doInvoke
	}
	
	def protected validateCallerIsOwner(Operation it, ArgumentExpression expr) {
		val owner = ((expr as FeatureCall)?.owner as ArgumentExpression)?.featureOrReference as Property
		return (owner?.type?.name == (it.eContainer as NamedElement).name)
	}

	def dispatch void doInvoke(ArgumentExpression expr) {
		val feature = expr.featureOrReference
		
		switch (feature) {
			Method:
				feature.doInvoke
			Operation case 'init'==feature.name : init
			Operation case 'isActive'==feature.name: isActive
			Operation case 'isFinal'==feature.name: isFinal
			Operation case 'isStateActive'==feature.name: isStateActive
				
			Operation: {
				val op = resolveOperation(feature)
				if (op !== null) op.invokeWithCaller(expr)
				else expr.doInvoke(feature)		
			}
			default:
				throw new InterpreterException("Cannot invoke :"+ feature)			
		}
	}
	
	def protected dispatch featureOrReference(FeatureCall it) {feature}
	def protected dispatch featureOrReference(ElementReferenceExpression it) {reference}
	def protected dispatch featureOrReference(Expression it) {null}

	def dispatch void doInvoke(Object caller, Object operation) {
		throw new UnsupportedOperationException('''Don't know how «caller» calls «operation»''')				
	}

	def dispatch void doInvoke(ArgumentExpression caller, Method operation) {
		if (operation.body !== null)
			operation.doInvoke
		else
			doInvokeExecutor(caller, operation)
	}

	def dispatch void doInvoke(ArgumentExpression caller, Operation operation) {
		if (operation.implementation !== null)
			operation.doInvoke
		else
			doInvokeExecutor(caller, operation)
	}

	def dispatch void doInvoke(ArgumentExpression caller, OperationDefinition it) {
		doInvokeExecutor(caller,it)
	}
	
	def void doInvokeExecutor(ArgumentExpression caller, Operation it) {
				
		_execute ('''invoke «name»''', [ 
			val paramValues = new ArrayList<Object>
			caller.expressions.forEach[paramValues.add(popValue)]
			enterCall(name)
			paramValues.reverse
			_execute("invoke using executor",[
				var Object result
				val executor = operationExecutors.findFirst[canExecute(caller, executionContextSlot)]
				if (executor !== null) {
					result = externalCall[executor.execute(caller, executionContextSlot, paramValues)]
					if (result === null && it.type != _void) {
						result = it.type?.defaultValue?.asValue
					} 
				}
				
				if (result !== null) {
					val callResult = result
					_execute[
						exitCall(callResult)	
					]
				}
				 else {
					_throw(new InterpreterException("Cannot execute operation '" + it.name + "'."))
				}				
			])
		])
	}
	
	
	def protected boolean hasMethod(String name) {
		lookupMethod(name) !== null
	}	
	
	def lookupMethod(String name) {
		type.features.filter(Operation).filter[it.name == name].head
	}
	
	
	def protected allMethods(ExecutionFlow it) {
		val allMethods = features.filter(Method).toList
		allMethods.addAll(states.map[features.filter(Method)].flatten)
		
		allMethods
	}
	
	def protected allOperations(ExecutionFlow it) {
		val allOperations = features.filter(Operation).toList
		allOperations.addAll(states.map[features.filter(Operation)].flatten)
		
		allOperations
	}
	
	//=========================================================================================
	//
	// Raising events
	//
	
	
	override raise(Object slot, Object value) {
		if (slot instanceof ExecutionEvent) {
			_execute('''raise «slot.name»''', [
				if (slot.direction == Direction::LOCAL && internalEventQueue !== null) {
					_execute('''internalEventQueue.add(«slot.name»)''', [
						internalEventQueue.add(new EventInstance(slot, value));
					])
				} else if (slot.direction == Direction::IN && inEventQueue !== null) {
					_execute('''inEventQueue.add(«slot.name»)''', [
						inEventQueue.add(new EventInstance(slot, value));
					])
					if (type.statechart.isSubMachine) {
						val contextRef = instanceSlot.slots.findFirst[name == SubMachine.SUBMACHINE_CONTEXT_MEMBER]
						val context = if (contextRef instanceof ReferenceSlot) contextRef.reference else null
						if (context !== null) {
							_execute('''submachineContext.eventRaised()''', [
								contextRef.invoke(submachineContextInterfaceEventRaised)
							])
						}
					} else {
						"runCycle"._exec
					}
				} else {
					_valueSemantics.setValue(slot, value)
					slot.raised = true
					
					if ( slot.direction == Direction.OUT) {
						slot.connections.forEach[ c |
							c.instance.raise(c, value)
						]	
					} 
					if (type.statechart.isEventDriven && slot.direction == Direction.IN) {
						"runCycle"._exec
					}
				}				
			])	
		}	
	}
	

	//=========================================================================================
	//
	// Setting properties
	//
	
	
	def dispatch set(Object slot, Object value) {
		if (slot.property !== null) {
			slot.property.set(value)
		} else {
			_valueSemantics.setValue(slot, value)
		}
	}

	def dispatch set(ReferenceSlot slot, CompositeSlot value) {
		disconnectOutEvents(slot)
		if (slot.property !== null) {
			slot.property.set(value)
		} else {
			_valueSemantics.setValue(slot, value)
		}
		connectOutEvents(slot)		
	}

	def protected disconnectOutEvents(ReferenceSlot it) {

		it.reference?.outEvents()?.forEach[ out | 
			val outShadow = shadowEvent(out)
			if (outShadow !== null) out.connections.remove(outShadow)
		]		
	}	

	def protected connectOutEvents(ReferenceSlot it) {

		it.reference?.outEvents()?.forEach[ out | 
			val outShadow = shadowEvent(out)
			if (outShadow !== null) out.connections.add(outShadow)
		]				
	}
	
	def protected void connectOutEvents(ExecutionContext it) {
		outEvents.forEach[ out |
			val localOut = out.localOutEvent
			if (localOut !== null) out.connections.add(localOut)
		]
	}
	
	def protected dispatch outEvents(ExecutionSlot it) {
		return #[]
	}

	def protected dispatch outEvents(CompositeSlot it) {
		it.eAllContents.filter(ExecutionEvent).filter[direction == Direction.OUT].toList
	}
	
	def protected shadowEvent(ReferenceSlot ref, ExecutionEvent foreignOutEvent) {
		val shadowEventName = '''«ref.fqName.replace('.', '_')»_«foreignOutEvent.fqName.replace('.', '_')»'''
		resolveSlot(instanceSlot, shadowEventName) as ExecutionEvent
	}
	
	def protected localOutEvent(ExecutionEvent ownOutEvent) {
		if (ownOutEvent.fqName !== null && !ownOutEvent.fqName.empty) {
			val localEventName = '''local_«ownOutEvent.fqName.replace('.', '_')»'''
			return resolveSlot(instanceSlot, localEventName) as ExecutionEvent
		}
		return null
	}
	
	
	
	//=========================================================================================
	//
	// resolving elements (implementation of the IInterpreter.Resolver)
	//
			
	def protected Method resolveMethod(Declaration decl) {
		type.allMethods.findFirst[name == decl.name]
	}
	
	def protected Operation resolveOperation(Declaration decl) {
		type.allOperations.findFirst[name == decl.name]
	} 

	
}