/**
 * Copyright (c) 2024-2025 itemis AG - All rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * 
 */
package com.yakindu.sct.simulation.core.sexec.interpreter

import com.google.inject.Inject
import com.google.inject.Singleton
import com.yakindu.base.expressions.interpreter.base.IInterpreter.Instance
import com.yakindu.base.expressions.interpreter.types.RuntimeInstanceFactory
import com.yakindu.base.types.Declaration
import com.yakindu.base.types.Package
import com.yakindu.base.types.inferrer.ITypeSystemInferrer.InferenceResult
import com.yakindu.sct.model.sexec.ExecutionFlow
import com.yakindu.sct.model.sexec.extensions.SExecExtensions
import com.yakindu.sct.model.sexec.extensions.StatemachineTypes
import com.yakindu.sct.model.sgraph.ImportDeclaration
import com.yakindu.sct.model.sruntime.ExecutionContext
import com.yakindu.sct.model.sruntime.ExecutionSlot
import com.yakindu.sct.model.stext.stext.ImportScope
import org.eclipse.xtext.EcoreUtil2
import org.eclipse.xtext.naming.IQualifiedNameProvider
import java.util.List
import com.yakindu.sct.model.sruntime.CompositeSlot
import org.eclipse.emf.ecore.EObject
import com.yakindu.base.types.ComplexType
import com.yakindu.base.types.Operation
import com.yakindu.sct.model.stext.stext.InterfaceScope
import com.yakindu.sct.model.stext.stext.InternalScope

/**
 * Creates statechart instances based on an ExecutionFlow
 * 
 * @author Axel Terfloth (terfloth@itemis.de)
 */
@Singleton
class SexecInstanceFactory extends RuntimeInstanceFactory {
	
	public static final String IMPORTS_SCOPE_NAME = "imports"
	
	@Inject protected extension IQualifiedNameProvider
	@Inject protected extension StatemachineTypes
	@Inject protected extension SExecExtensions


	protected def dispatch Object createNewInstance(ExecutionFlow flow) {
		val instance = createExecutionContext => [
			name = if (heap.slots.size == 0) "$root" else "instance@" + heap.slots.size
			fqName = name
			type = flow
		]
		
		flow.createImportScopes(instance)
		contextInitializer.initialize(instance, flow)
		flow.createMetaFeatureInstanceMember(instance)
		flow.createInstanceMembers(instance)
		
		heap.slots.add(instance)
		
		instance.adaptInstance( createInstanceExecution(instance, flow) )

		instance
	}
	
	def void createMetaFeatureInstanceMember(ExecutionFlow it, CompositeSlot instanceSlot) {
		scopes.filter[!(it instanceof ImportScope)].forEach[
			val scopeSlot = switch (it) {
				InterfaceScope case name.isNullOrEmpty : instanceSlot.slotByName("default")
				InterfaceScope : instanceSlot.slotByName(name)
				InternalScope  : instanceSlot.slotByName("internal")
				default: null
			}
			if (scopeSlot !== null) 
				if (scopeSlot instanceof CompositeSlot) {
					it.members.forEach[ m |
						scopeSlot.slots += m.createMetaFeatureInstanceMember(it)
					]
				}
		]
	}
	
	def protected slotByName(CompositeSlot slot, String symbol) {
		slot.slots.findFirst[ s | s.name == symbol || s.fqName == symbol] 
	}
	
	def protected Instance createInstanceExecution(ExecutionContext instanceMemory, ExecutionFlow flow) {
		new ExecutionFlowInstance => [
			it.injectMembers
			it.setUp(instanceMemory, instanceContext, flow, this)	
		]
	} 
	
	
	protected def dispatch Object createNewInstance(ExecutionFlow flow, InferenceResult ts) {
		flow.createNewInstance
	}
	
	def protected void createImportScopes(ExecutionFlow it, ExecutionContext context) {
		context.slots += importSlot
		scopes.filter(ImportScope).forEach[ transform ]
	}
	
	def ExecutionSlot transform(ImportScope scope) {

		val composite = importSlot(scope.flow)

		// retrieve namespaces from variable names and create corresponding composite slots
		for (Declaration decl : scope.declarations.filter(ImportDeclaration).map[declaration].filter[useInImportScope]) {

			val declName = decl.name
			val declGlobalSlot = decl.provideInstance as ExecutionSlot
			val declLocalRefSlot = createReferenceSlot => [
				reference = declGlobalSlot
				name = declGlobalSlot.name
				type = declGlobalSlot.type
				origin = decl
			]

			val pkg = EcoreUtil2.getContainerOfType(decl, Package)
			if (pkg !== null) {
				val namespace = pkg.name
				val pkgHeader = pkg.fullyQualifiedName.lastSegment
				val slot = composite.slots.getSlotFor(pkgHeader)
				slot.origin = pkg
				declLocalRefSlot.fqName = namespace + "." + declName
				
				// only add imported variables/events when they have not yet been imported
				if (!slot.slots.exists[fqName == declLocalRefSlot.fqName]) {
					slot.slots += declLocalRefSlot
				}
			} else {
				composite.slots += declLocalRefSlot
			}
		}
		composite
	}
	
	def protected useInImportScope(Declaration it) {
		return useInImportScope(eContainer)
	}
	def protected dispatch useInImportScope(Declaration it, EObject parent) { false }
	def protected dispatch useInImportScope(Package it, Package parent) { false }
	def protected dispatch useInImportScope(Declaration it, Package parent) { true }
	def protected dispatch useInImportScope(Declaration it, ComplexType parent) { isStatic }
	def protected dispatch useInImportScope(Operation it, EObject parent) { false }
	def protected dispatch useInImportScope(Operation it, Package parent) { true }
	
	def protected getSlotFor(List<ExecutionSlot> slots, String name) {
		val existingSlot = slots.findFirst[it.name == name]
		if (existingSlot instanceof CompositeSlot) {
			existingSlot
		} else
			createCompositeSlot => [ newSlot |
				newSlot.name = name
				slots += newSlot
			]
	}
	
	def create it : createCompositeSlot importSlot(ExecutionFlow flow) {
		it => [
			name = IMPORTS_SCOPE_NAME
		]
	}
	
	
}