/**
 * Copyright (c) 2020 itemis AG - All rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * 
 */
package com.yakindu.sct.generator.python.files

import com.google.inject.Inject
import com.yakindu.base.base.NamedElement
import com.yakindu.base.expressions.expressions.AssignmentExpression
import com.yakindu.base.expressions.expressions.ElementReferenceExpression
import com.yakindu.base.types.TypesUtil
import com.yakindu.base.types.typesystem.ITypeValueProvider
import com.yakindu.sct.generator.core.extensions.StringHelper
import com.yakindu.sct.generator.core.filesystem.ISCTFileSystemAccess
import com.yakindu.sct.generator.core.library.ICoreLibraryHelper
import com.yakindu.sct.generator.python.FlowCode
import com.yakindu.sct.generator.python.GenmodelEntries
import com.yakindu.sct.generator.python.InitialValueProvider
import com.yakindu.sct.generator.python.PythonObservables
import com.yakindu.sct.generator.python.PythonStatechartIncludeProvider
import com.yakindu.sct.generator.python.naming.Naming
import com.yakindu.sct.generator.python.naming.PythonNamingService
import com.yakindu.sct.generator.python.submodules.InternalFunctionsGenerator
import com.yakindu.sct.generator.python.submodules.InternalTypes
import com.yakindu.sct.generator.python.submodules.ScopeGenerator
import com.yakindu.sct.generator.python.submodules.TimingFunctions
import com.yakindu.sct.model.sexec.ExecutionFlow
import com.yakindu.sct.model.sexec.Sequence
import com.yakindu.sct.model.sexec.Statement
import com.yakindu.sct.model.sexec.extensions.SExecExtensions
import com.yakindu.sct.model.sexec.extensions.ShadowEventExtensions
import com.yakindu.sct.model.sexec.extensions.StateVectorExtensions
import com.yakindu.sct.model.sgen.GeneratorEntry
import java.nio.file.Paths

import static com.yakindu.sct.generator.core.filesystem.ISCTFileSystemAccess.*
import com.yakindu.sct.model.sexec.Execution

class Statemachine {
	
	@Inject extension Naming
	@Inject extension PythonNamingService
	@Inject extension GenmodelEntries
	@Inject extension SExecExtensions
	@Inject extension StateVectorExtensions
	@Inject extension PythonObservables
	@Inject extension ShadowEventExtensions
	@Inject extension InternalTypes
	@Inject extension InternalFunctionsGenerator
	@Inject extension ITypeValueProvider
	@Inject extension InitialValueProvider
	@Inject extension FlowCode
	@Inject extension ICoreLibraryHelper
	@Inject extension ScopeGenerator
	@Inject extension TimingFunctions
	@Inject extension StringHelper
	@Inject extension PackageInit
	@Inject protected extension TypesUtil

	@Inject protected PythonStatechartIncludeProvider imports
	@Inject protected GeneratorEntry entry
	
	def generateStatemachine(ExecutionFlow flow, GeneratorEntry entry, ISCTFileSystemAccess fsa) {
		var content = content(flow, entry, fsa)
		val path = entry.basePackagePath
		fsa.generateFile(path + '/' + flow.statemachineClassName.py, content)
		
		if (!path.nullOrEmpty)
			fsa.generatePyPackageInit(path, entry)
	}
	
	protected def content(ExecutionFlow flow, GeneratorEntry entry, ISCTFileSystemAccess fsa) { 
	'''
			"""Implementation of statechart «flow.statemachineClassName».
			«entry.licenseText»
			"""
			
			«flow.imports(fsa)»
			«flow.observerClass»
			class «flow.statemachineName»:
				"""Implementation of the state machine «flow.statemachineName».
				"""
			
				«flow.statesEnum»
				
				«flow.internalTypes»
				«flow.interfaceTypes»
				«flow.constructor»
				«flow.activeFunction»
				«flow.finalFunction»
				«flow.stateActiveFunction»
				«flow.timingFunctions»
				«flow.internalScopeFunctions»
				«flow.interfaceScopeFunctions»
				«flow.functionImplementations»
		'''
	}
	
	protected def imports(ExecutionFlow it, ISCTFileSystemAccess fsa) '''
		«IF entry.libraryTargetFolderValue !== null»
			import sys, os
			sys.path.insert(0, os.path.abspath(os.path.join(os.path.dirname(__file__), '«fsa.relativeLibFolder»')))
		«ENDIF»
		«IF entry.outEventObservablesUsed && hasOutgoingEvents»
			from «entry.libraryPackage.dot(rxpyModule)» import Observable
		«ENDIF»
		«IF !shadowEvents.nullOrEmpty»
			from «entry.libraryPackage.dot(rxpyModule)» import Observer
		«ENDIF»
		«flow.submachineImports»
		
	'''
	
	
	protected def getRelativeLibFolder(ISCTFileSystemAccess fsa) {
		val base = Paths.get(entry.basePackagePath)
		var rel = ""
		if (!entry.basePackagePath.nullOrEmpty) {
			// go all the way up
			rel = (1 .. base.nameCount).map['../'].join
		}
		val defaultTarget = fsa.outputConfigurations.get(TARGET_FOLDER_OUTPUT).outputDirectory
		val libTarget = fsa.outputConfigurations.get(LIBRARY_TARGET_FOLDER_OUTPUT).outputDirectory
		if (libTarget != defaultTarget) {
			// go down to lib folder
			rel += '../' + libTarget
		}
		rel
	}
	
	protected def statesEnum(ExecutionFlow it) {
		'''
			class «stateEnumName»:
				""" State Enum
				"""
				(
					«FOR state : flow.states»
						«state.stateName»,
					«ENDFOR»
					«getNullStateName()»
				) = range(«flow.states.length + 1»)
			
		'''
	}
	
	protected def interfaceTypes(ExecutionFlow flow) '''
		«FOR scope : flow.namedInterfaceScopes»
			«scope.interfaceScopeClass(flow)»
		«ENDFOR»
	'''
	
	protected def submachineImports(ExecutionFlow flow) '''
		«FOR i : imports.getImports(flow, entry)»
			«i»
		«ENDFOR»
	'''
	
	protected def constructor(ExecutionFlow flow) '''
		def __init__(self):
			""" Declares all necessary variables including list of states, histories etc. 
			"""
			«flow.interfaceProperties»
			«flow.interfaceScopeMembers»
			«flow.internalScopeMembers»
			«flow.shadowEventProperties»
			«flow.stateProperties»
			«flow.timedProperties»
			«flow.historyProperties»
			# initializations:
			«flow.initSequence.removeStateConfFromInit.code»
			«FOR p : flow.getProperties»
				self.«p.identifier» = «p.type.defaultValue.initialValue(p.type)»
			«ENDFOR»
		
	'''
	
	protected def removeStateConfFromInit(Sequence seq){
		val stepForStateConf = newArrayList
		seq.steps.forEach[ step |
			if(step instanceof Execution){
				if(step.statement instanceof AssignmentExpression){
					var stepsAsignmentExpr = step.statement as AssignmentExpression
					if(stepsAsignmentExpr.varRef instanceof ElementReferenceExpression){
						var stepVarref = stepsAsignmentExpr.varRef as ElementReferenceExpression
						if(stepVarref.reference instanceof NamedElement && (stepVarref.reference as NamedElement).name == "stateVector")
							stepForStateConf += step
					}
				}
			}
		]
		stepForStateConf.forEach[ step |
			seq.steps.remove(step)
		]		
		seq
	}
	
	protected def interfaceProperties(ExecutionFlow flow) '''
		«FOR scope : flow.namedInterfaceScopes»
			self.«scope.interfaceVariableName» = «flow.statemachineName».«scope.interfaceTypeName()»(self)
		«ENDFOR»
		
	'''
	
	protected def shadowEventProperties(ExecutionFlow flow) '''
		«FOR event : flow.shadowEvents»
			self.«event.identifier» = None
			self.«event.asObserver» = «event.asObserverClass»(self)
			«IF event.hasValue»
				self.«event.valueIdentifier» = None
			«ENDIF»
			
		«ENDFOR»
	'''
	
	protected def stateProperties(ExecutionFlow flow) '''
		# enumeration of all states:
		self.«stateInstance» = «flow.statemachineName».«stateEnumName»
		self.__state_conf_vector_changed = None
		self.«stateVector» = [None] * «flow.stateVector.size»
		for «stateIndex» in range(«flow.stateVector.size»):
			self.«stateVector»[«stateIndex»] = self.«stateEnumName».«nullStateName»
		
	'''
	
	protected def timedProperties(ExecutionFlow flow) '''
		«IF flow.timed»
			# for timed statechart:
			self.«timerService» = None
			self.«timeEvents» = [None] * «flow.timeEvents.size»
			
		«ENDIF»
	'''
	
	protected def historyProperties(ExecutionFlow flow) '''
		«IF flow.hasHistory»
			# history vector:
			self.«historyVector» = [None] * «flow.historyVector.size»
			for «historyIndex» in range(«flow.historyVector.size»):
				self.«historyVector»[«historyIndex»] = self.«stateEnumName».«nullStateName»
			
		«ENDIF»
	'''
	
	protected def internalScopeMembers(ExecutionFlow flow) '''
		«flow.internalScope?.properties»
	'''
	
	protected def interfaceScopeMembers(ExecutionFlow flow) '''
		«flow.defaultScope?.properties»
	'''
	
	protected def isStateActiveFunction(ExecutionFlow flow) '''
		def «isStateActive»(self, state):
			"""Checks if the state is currently active.
			"""
			s = state
			«FOR state : flow.states»
				if s == self.«stateInstance».«state.stateName.asEscapedIdentifier»:
					return «IF state.leaf»self.«stateVector»[«state.stateVector.offset»] == self.«stateInstance».«state.stateName.asEscapedIdentifier»«ELSE»(self.«stateVector»[«state.stateVector.offset»] >= self.«stateInstance».«state.stateName.asEscapedIdentifier»)\
						and (self.«stateVector»[«state.stateVector.offset»] <= self.«stateInstance».«state.subStates.lastOrNull.stateName.asEscapedIdentifier»)«ENDIF»
				«ENDFOR»
			return False
			
	'''
	
	protected def isActiveFunction(ExecutionFlow flow) '''
		def «isActive»(self):
			"""Checks if the state machine is active.
			"""
			return «FOR i : 0 ..< flow.stateVector.size SEPARATOR ' or '»self.«stateVector»[«i»] is not self.«stateInstance».«nullStateName»«ENDFOR»
		
	'''

	protected def isFinalFunction(ExecutionFlow flow) {
		val finalStateImpactVector = flow.finalStateImpactVector
		'''
			def «isFinal»(self):
				"""Checks if the statemachine is final.«IF !finalStateImpactVector.isCompletelyCovered»
				Always returns 'false' since this state machine can never become final.«ENDIF»
				"""
		''' +

		// only if the impact vector is completely covered by final states the state machine
		// can become final

			{if (finalStateImpactVector.isCompletelyCovered) {
			'''	return «FOR i : 0 ..<finalStateImpactVector.size SEPARATOR ' and '»(«FOR fs : finalStateImpactVector.get(i) SEPARATOR ' or '»self.«stateVector»[«i»] == «
								IF fs.stateVector.offset == i
									»self.«stateInstance».«fs.stateName.asEscapedIdentifier»«
								ELSE
									»self.«stateInstance».«nullStateName»«
								ENDIF»«
							ENDFOR»)«ENDFOR»
							
		'''} else {
		'''	return False
		
		'''} }
	}
	
	protected def internalScopeFunctions(ExecutionFlow flow) '''
		«FOR scope : flow.internalScopes»
			«scope.methods»
		«ENDFOR»
	'''
	
	protected def interfaceScopeFunctions(ExecutionFlow flow) '''
		«flow.defaultScope?.methods»
	'''
	
}
