/**
 * Copyright (c) 2022 itemis AG - All rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * 
 * Contributors:
 * 	Andreas Muelder - itemis AG
 * 
 */
package com.yakindu.sctunit.generator.python.extensions

import com.google.inject.Inject
import com.yakindu.base.expressions.expressions.ArgumentExpression
import com.yakindu.base.expressions.expressions.AssignmentExpression
import com.yakindu.base.expressions.expressions.BoolLiteral
import com.yakindu.base.expressions.expressions.ConditionalExpression
import com.yakindu.base.expressions.expressions.DoubleLiteral
import com.yakindu.base.expressions.expressions.ElementReferenceExpression
import com.yakindu.base.expressions.expressions.EventRaisingExpression
import com.yakindu.base.expressions.expressions.EventValueReferenceExpression
import com.yakindu.base.expressions.expressions.FeatureCall
import com.yakindu.base.expressions.expressions.FloatLiteral
import com.yakindu.base.expressions.expressions.HexLiteral
import com.yakindu.base.expressions.expressions.IntLiteral
import com.yakindu.base.expressions.expressions.LogicalAndExpression
import com.yakindu.base.expressions.expressions.LogicalNotExpression
import com.yakindu.base.expressions.expressions.LogicalOrExpression
import com.yakindu.base.expressions.expressions.LogicalRelationExpression
import com.yakindu.base.expressions.expressions.NullLiteral
import com.yakindu.base.expressions.expressions.PostFixOperator
import com.yakindu.base.expressions.expressions.PostFixUnaryExpression
import com.yakindu.base.expressions.expressions.PrimitiveValueExpression
import com.yakindu.base.expressions.expressions.RelationalOperator
import com.yakindu.base.expressions.expressions.StringLiteral
import com.yakindu.base.expressions.expressions.UnaryExpression
import com.yakindu.base.expressions.util.ExpressionExtensions
import com.yakindu.base.types.Declaration
import com.yakindu.base.types.EnumerationType
import com.yakindu.base.types.Enumerator
import com.yakindu.base.types.Expression
import com.yakindu.base.types.Operation
import com.yakindu.base.types.Property
import com.yakindu.sct.generator.python.PythonMultiStatemachineFunctionProvider
import com.yakindu.sct.generator.python.naming.Naming
import com.yakindu.sct.model.sgraph.util.StatechartUtil
import com.yakindu.sct.model.stext.stext.ActiveStateReferenceExpression
import com.yakindu.sct.model.stext.stext.EventDefinition
import com.yakindu.sct.model.stext.stext.InterfaceScope
import com.yakindu.sct.model.stext.stext.VariableDefinition
import com.yakindu.sctunit.generator.base.extensions.BaseExpressionExtensions
import com.yakindu.sctunit.generator.base.extensions.BaseNavigationExtensions
import com.yakindu.sctunit.generator.base.extensions.BaseStatementExtensions
import com.yakindu.sctunit.sCTUnit.EnterExpression
import com.yakindu.sctunit.sCTUnit.ExitExpression
import com.yakindu.sctunit.sCTUnit.IfStatement
import com.yakindu.sctunit.sCTUnit.ProceedExpression
import com.yakindu.sctunit.sCTUnit.ReturnStatement
import com.yakindu.sctunit.sCTUnit.SCTUnitOperation
import com.yakindu.sctunit.sCTUnit.StatechartActiveExpression
import com.yakindu.sctunit.sCTUnit.StatechartFinalExpression
import com.yakindu.sctunit.sCTUnit.TriggerWithoutEventExpression
import org.eclipse.emf.ecore.EObject
import org.eclipse.xtext.util.Strings

/**
 * 
 * @author andreas muelder - Initial contribution and API
 * 
 */
class PythonExpressionExtensions extends BaseExpressionExtensions {

	@Inject extension BaseNavigationExtensions
	@Inject extension BaseStatementExtensions stms
	@Inject extension NamingExtensions
	@Inject extension StatechartUtil
	@Inject extension ExpressionExtensions
	@Inject extension PythonMultiStatemachineFunctionProvider
	@Inject extension Naming

	override dispatch CharSequence code(SCTUnitOperation operation) {
		return '''mcfly«operation.name»()'''
	}

	override dispatch CharSequence code(UnaryExpression it) {
		operator.literal + operand.code
	}

	override dispatch CharSequence code(PostFixUnaryExpression it) {
		'''«operand.code» = «operand.code»«operator.unaryOperator»1'''
	}

	def protected CharSequence unaryOperator(PostFixOperator operator) {
		switch operator {
			case PostFixOperator.DECREMENT: return '''-'''
			case PostFixOperator.INCREMENT: return '''+'''
			default: return '''#«operator» not implemented'''
		}
	}

	override dispatch CharSequence code(ElementReferenceExpression expression) {
		/*local (internal) defined test operations*/
		if (expression.isExpressionVoidOperation) {
			return '''self.«expression.referenceName»(«FOR expr : expression.expressions SEPARATOR ', '»«expr.code»«ENDFOR»)'''
		}
		if (!expression.reference.isLocal) {
			return expression.generateNotLocalElemRefExpr
		}
		/*variables & parameter */
		'''«expression.referenceName»'''
	}

	override dispatch String referenceName(ElementReferenceExpression it) {
		'''«IF reference.isTest»test_«ENDIF»«reference.identifier»'''
	}

	override dispatch CharSequence code(Expression expression) {
		'''# «expression» not implemented - sorry :('''
	}

	override dispatch CharSequence code(ConditionalExpression expression) {
		'''«expression.trueCase.code» if «expression.condition.code»«IF expression.falseCase !== null» else «expression.falseCase.code»«ENDIF»'''
	}

	override dispatch CharSequence code(LogicalAndExpression expression) {
		'''«expression.leftOperand.code» and «expression.rightOperand.code»'''
	}

	override dispatch CharSequence code(LogicalOrExpression expression) {
		'''«expression.leftOperand.code» or «expression.rightOperand.code»'''
	}

	override dispatch CharSequence code(LogicalNotExpression expression) {
		'''not «expression.operand.code»'''
	}

	override dispatch CharSequence code(LogicalRelationExpression expression) {
		if (expression.leftOperand.infer.type.isStringType && expression.operator == RelationalOperator::EQUALS) {
			'''«expression.leftOperand.code» == «expression.rightOperand.generateString»'''
		} else
			'''«expression.leftOperand.code» «expression.operator.literal» «expression.rightOperand.code»'''
	}

	def dispatch CharSequence code(IfStatement expression) {
		'''
			if «expression.condition.code»:
				«IF expression.then.code.nullOrEmpty»
					pass
				«ENDIF»
				«FOR trueExpression : expression.then.code»
					«stms.generate(trueExpression)»
				«ENDFOR»
			«IF expression.^else !== null»
				«IF expression.then.code.exists[it instanceof ReturnStatement]»
					«FOR falseExpression : expression.^else.code»
						«stms.generate(falseExpression)»
					«ENDFOR»
				«ELSE»
					else:
						«IF expression.^else.code.nullOrEmpty»
							pass
						«ENDIF»
						«FOR falseExpression : expression.^else.code»
							«stms.generate(falseExpression)»
						«ENDFOR»
				«ENDIF»
			«ENDIF»
		'''
	}

	override dispatch CharSequence code(PrimitiveValueExpression expression) {
		return '''«expression.value.valueLiteral»'''
	}

	def dispatch valueLiteral(IntLiteral literal) {
		return literal.value
	}

	def dispatch valueLiteral(HexLiteral literal) {
		return literal.value
	}

	def dispatch valueLiteral(BoolLiteral bool) {
		return bool.value.toString.toFirstUpper
	}

	def dispatch valueLiteral(FloatLiteral literal) {
		return literal.value
	}

	def dispatch valueLiteral(DoubleLiteral literal) {
		return literal.value
	}

	def dispatch valueLiteral(StringLiteral it) '''"«Strings.convertToJavaString(value.escaped)»"'''


	def dispatch valueLiteral(NullLiteral literal) {
		'''None'''
	}

	def dispatch CharSequence code(EventValueReferenceExpression expression) {
		val fc = expression.value
		if (fc instanceof FeatureCall) {
			if (fc.feature.eContainer.multiSM) {
				return '''«fc.access».«fc.feature.valueAccess»_value'''
			}
			'''«fc.code».«fc.feature.valueAccess»_value'''
		} else {
			// Check if the expression is defined in an unnamed interface scope
			'''self.statemachine.«expression.value.referenceName»_value'''
		}
	}

	override dispatch CharSequence code(EventRaisingExpression expression) {
		val fc = expression.event
		if (fc instanceof FeatureCall) {
			if (fc.feature.eContainer.multiSM) {
				return '''«fc.access».«fc.feature.code»(«IF expression.value !== null»«expression.value.code»«ENDIF»)'''
			}
			return '''«(expression.event as FeatureCall).owner.access».raise_«((expression.event as FeatureCall).feature as EventDefinition).identifier»(«IF expression.value !== null»«expression.value.code»«ENDIF»)'''

		} else {
			'''self.statemachine.raise_«expression.event.featureOrReference.identifier»(«IF expression.value !== null»«expression.value.code»«ENDIF»)'''
		}
	}

	override dispatch CharSequence code(ActiveStateReferenceExpression expression) {
		'''self.statemachine.is_state_active(«expression.SCTUnitClass?.statechart.statemachineName».«stateEnumName».«expression.value.stateName»)'''
	}

	override generateNotLocalElemRefExpr(ElementReferenceExpression it) {
		return '''self.statemachine.«reference.getterName»'''
	}

	override CharSequence generateNotLocalAssignment(AssignmentExpression it) {
		if (operator.name().equalsIgnoreCase("assign")) {
			if (expression instanceof PostFixUnaryExpression) {
				return '''
				«varRef.code» = «(expression as PostFixUnaryExpression).operand.code»
				«expression.code»'''
			}
			val varRef = varRef
			if (varRef instanceof FeatureCall) {
				if (varRef.feature.eContainer.multiSM) {
					return '''«varRef.code» = «expression.code»'''
				}
			}
			return '''«varRef.context»«varRef.valueSetter» = «expression.code»'''

		} else
			return '''self.«varRef.valueSetter»(«varRef.code» «operator.toString» «expression.code»)'''
	}

	def dispatch CharSequence context(ElementReferenceExpression it) {
		if (reference.isStatechartLocal) {
			val scope = reference.eContainer
			if (scope instanceof InterfaceScope && !(scope as InterfaceScope).name.nullOrEmpty) {
				return '''statemachine.«scope.valueGetter»().'''
			}
		}
		return ''''''
	}

	def dispatch CharSequence context(Expression it) '''statemachine.'''

	def dispatch CharSequence context(FeatureCall it) {
		if (feature instanceof EnumerationType) {
			'''self.'''
		} else {
			'''self.statemachine.'''
		}
	}

	def name(EventDefinition exp) {
		exp.name
	}

	override dispatch CharSequence code(StatechartActiveExpression exp) {
		'''self.statemachine.is_active()'''
	}

	override dispatch CharSequence code(StatechartFinalExpression exp) {
		'''self.statemachine.is_final()'''
	}

	override dispatch code(Expression owner, EObject feature) {
		'''
		"""
		Unknown FeatureCall for «owner.eContainer.eClass.name»
		for feature «feature.eClass.name»
		"""'''
	}

	def dispatch code(ArgumentExpression it, Enumerator target) {
		if (target.eContainer.isOriginStatechart) {
			return '''«(featureOrReference as Declaration).enumAccess».«stateEnumName».«target.stateName»'''
		}
		'''«target.code»'''
	}

	def dispatch code(Expression owner, Operation feature) {
		'''«owner.access(owner.featureOrReference)».«feature.code»(«owner.eContainer.codeArguments»)'''
	}
	
	override dispatch code(ElementReferenceExpression owner, Operation feature) {
		'''«owner.access(owner.featureOrReference)».«feature.code»(«owner.eContainer.codeArguments»)'''
	}


	def dispatch codeArguments(EObject it) {
		""
	}

	def dispatch codeArguments(ArgumentExpression it) {
		'''«FOR arg : it.expressions SEPARATOR ","»«arg.code»«ENDFOR»'''
	}

	def dispatch code(Operation it) {
		functionId
	}

	def dispatch code(EventDefinition it) {
		if (eContainer.multiSM) {
			return eventRaisingAccess
		}
		it.name.toLowerCase
	}

	def dispatch code(Declaration it) {
		it.identifier
	}

	def dispatch code(Expression owner, Property feature) {
		if (feature.eContainer.isMultiSM) {
			return '''«owner.access(owner.featureOrReference)».«feature.valueAccess»'''
		}
		'''«owner.access».«feature.getterName»'''
	}

	def dispatch code(Expression owner, EventDefinition feature) {
		if (feature.eContainer.isMultiSM) {
			return '''«owner.access».«feature.valueAccess»'''
		}
		return '''«owner.access».«feature.identifier»'''
	}
	
	def dispatch CharSequence access(Expression it)'''self.statemachine'''

	def dispatch CharSequence access(ElementReferenceExpression it) {
		val ref = reference
		if (ref instanceof InterfaceScope){
			if (!ref.name.nullOrEmpty)
				return '''self.statemachine.«ref.interfaceVariableName»'''
			else
				return '''self.statemachine.'''
		}
		'''self.statemachine.«reference.valueAccess»'''
	}

	def dispatch access(FeatureCall it) {
		owner.access(owner.featureOrReference)
	}

	def dispatch CharSequence access(Expression owner, EObject feature) {
		'''"""
		Unknown access for feature '«feature.getClass.name»'
		"""'''
	}

	def dispatch CharSequence access(Expression it, Property target) {
		if (it instanceof FeatureCall) {
			return '''«access»'''
		}
		return ''''''
	}

	def dispatch CharSequence access(Expression it, VariableDefinition reference) {
		if (it instanceof FeatureCall) {
			if (reference.eContainer.multiSM) {
				return '''«access».«reference.valueAccess»'''
			}
		}
		'''«reference.valueSetter»'''
	}

	override proceedCycles(ProceedExpression it) {
		'''
			self.timer_service.cycle_leap(«value.code»)
		'''
	}

	override proceedTime(ProceedExpression it) {
		'''
			self.timer_service.time_leap(«value.toMilliseconds(unit).code»)
		'''
	}

	def dispatch code(EnterExpression stm) {
		'''self.statemachine.enter()'''
	}

	def dispatch code(ExitExpression stm) {
		'''self.statemachine.exit()'''
	}
	
	def dispatch code(TriggerWithoutEventExpression stm) {
		'''self.statemachine.trigger_without_event()'''
	}
	
	override dispatch code(IntLiteral literal) {
		return '''«literal.value»'''
	}
}
