/**
 * Copyright (c) 2022-2026 itemis AG - All rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * 
 */
package com.yakindu.sct.model.sexec.concepts

import com.google.inject.Inject
import com.yakindu.base.expressions.ExpressionBuilder
import com.yakindu.base.expressions.expressions.ExpressionsFactory
import com.yakindu.base.types.Expression
import com.yakindu.base.types.Property
import com.yakindu.base.types.TypeBuilder
import com.yakindu.sct.model.sexec.ExecutionFlow
import com.yakindu.sct.model.sexec.ExecutionState
import com.yakindu.sct.model.sexec.Sequence
import com.yakindu.sct.model.sexec.Step
import com.yakindu.sct.model.sexec.extensions.SExecExtensions
import com.yakindu.sct.model.sexec.extensions.SexecBuilder
import com.yakindu.sct.model.sexec.transformation.StextExtensions
import com.yakindu.sct.model.sgraph.FinalState
import com.yakindu.sct.model.sgraph.RegularState
import com.yakindu.sct.model.sgraph.State
import com.yakindu.sct.model.sgraph.Synchronization
import com.yakindu.sct.model.sgraph.Transition
import com.yakindu.sct.model.stext.concepts.CompletionTransition
import java.util.ArrayList

/**
 * This class defines the general concept of completion events. 
 * 
 * @author lkovacs
 */
class CompletionEvent {

	@Inject protected extension SExecExtensions
	@Inject protected extension StextExtensions
	@Inject protected extension TypeBuilder
	@Inject protected extension SexecBuilder sexec
	@Inject protected extension CompletionTransition
	@Inject extension ExpressionBuilder
	@Inject extension HaltBehavior

	extension ExpressionsFactory = ExpressionsFactory.eINSTANCE

	public static val COMPLETED = "completed"
	public static val DO_COMPLETION = "doCompletion"

	def defineFeatures(ExecutionFlow it) {
		if (hasCompletionTransition) {
			it.features += _variable(COMPLETED, _boolean) => [
				_synthetic
				initialValue = _false
			]
			it.features += _variable(DO_COMPLETION, _boolean) => [
				_synthetic
				initialValue = _false
			]
		}
	}

	def _completionLoop(ExecutionFlow it, Step body) {
		if (hasCompletionTransition)
			_sequence(
				doCompletionProperty._assign(_false),
				_do(
					(_sequence(
						sexec._if(completedProperty._ref)._then(
							doCompletionProperty._assign(_true)
						),
						completedProperty._assign(_false),
						body,
						doCompletionProperty._assign(_false)
					))
				)._while(
					completedProperty._ref._andIsNotHalted(it)
				)
			)
		else
			body
	}

	def completionEntry(ExecutionFlow it, RegularState state) {
		completedProperty._assign(_true)
	}

	def createCompletionReactionSequence(ExecutionState state, Sequence localStep) {

		val completionTransitionEffect = state.reactions.fold(_sequence(), [ s, r |
			if (r.sourceElement !== null && r.sourceElement.equals(
				state.reactions.filter[transition].map[sourceElement as Transition].completionTransition
			))
				s.steps.add(r.effect)
			s
		])

		_sequence(
			sexec._if(state._completionReactionCond)._then(
				completionTransitionEffect
			)._else(
				localStep
			)
		)
	}

	def _completionReactionCond(ExecutionState state) {

		if (state.isLeaf) {
			if ((state.sourceElement as State).outgoingTransitions.completionTransition.isJoin) {
				val requiredActiveStates = ((state.sourceElement as State)
					.outgoingTransitions.completionTransition.target as Synchronization)
					.incomingTransitions.map[source].filter(RegularState)
					
					requiredActiveStates.map[active].checkOrthogonalStatesAreActive(state)
			} else
				state.flow.doCompletionProperty._ref
		} else
			state.collectLeafStates(new ArrayList<ExecutionState>).filter [
				sourceElement instanceof FinalState && directlyContainedLeafState(state)
			].map[sourceElement as RegularState]
			.map[active]
			.checkOrthogonalStatesAreActive(state)

	}

	def protected checkOrthogonalStatesAreActive(Iterable<Expression> it, ExecutionState state) {
		fold(createLogicalAndExpression => [
			leftOperand = state.flow.doCompletionProperty._ref
		], [ a, e |
			if(a.leftOperand !== null && a.rightOperand === null) a.rightOperand = a.leftOperand
			if(a.leftOperand !== null && a.rightOperand !== null) a.rightOperand = a.leftOperand._and(a.rightOperand)
			a.leftOperand = e
			a
		])
	}

	def Property completedProperty(ExecutionFlow it) {
		features.filter(typeof(Property)).filter(m|m.name == COMPLETED).head
	}

	def Property doCompletionProperty(ExecutionFlow it) {
		features.filter(typeof(Property)).filter(m|m.name == DO_COMPLETION).head
	}

	def boolean hasCompletionTransition(ExecutionFlow it) {
		statechart.hasCompletionTransition
	}

	def protected isJoin(Transition it) {
		// Transition to Join
		if ((target instanceof Synchronization) && target.incomingTransitions.size > 1)
			return true
		return false
	}
}
