/**
 * Copyright (c) 2020 itemis AG - All rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * 
 */
package com.yakindu.sct.model.sexec.concepts

import com.google.inject.Inject
import com.yakindu.base.expressions.ExpressionBuilder
import com.yakindu.base.types.ComplexType
import com.yakindu.base.types.Declaration
import com.yakindu.base.types.Direction
import com.yakindu.base.types.Event
import com.yakindu.base.types.Expression
import com.yakindu.base.types.Property
import com.yakindu.base.types.Type
import com.yakindu.base.types.TypeBuilder
import com.yakindu.base.types.adapter.OriginTracing
import com.yakindu.sct.model.sexec.ExecutionFlow
import com.yakindu.sct.model.sexec.TimeEvent
import com.yakindu.sct.model.sexec.naming.INamingService
import com.yakindu.sct.model.sexec.transformation.config.IFlowConfiguration
import com.yakindu.sct.model.sgraph.Scope
import com.yakindu.sct.model.sgraph.Statechart
import com.yakindu.sct.model.stext.stext.InterfaceScope
import com.yakindu.sct.model.stext.stext.InternalScope
import java.util.ArrayList
import java.util.List

/**
 * This class defines the concept of an event buffer which can be applied to cycle based state machines.
 * It defines additional types and members which implement a double buffering of events. 
 * 
 * @author aterfloth
 */
class EventBuffer {
	
	public static String EVENT_BUFFER_ANNOTATION = "__event_buffer__"
	
	@Inject extension TypeBuilder
	@Inject extension BufferEvent	
	@Inject extension INamingService
	@Inject extension OriginTracing
	@Inject extension ExpressionBuilder
	@Inject extension ShadowMemberScope
	
	@Inject protected extension IFlowConfiguration config
	
	def defineEventBuffer(ExecutionFlow flow, Statechart sc){
		
		if (! (applyIncomingEventBuffer || applyInternalEventBuffer)) return 
		
		val bufferType = _complexType() => [ bt |
			flow.scopes.filter[hasBufferedEvents].forEach[ scope | 
				val scopeBufferType = _complexType() => [ scopeType |
					scope.declarations
						.filter(Event)
						.filter[ e | e.isBuffered]
						.forEach[ e | e.createBufferEvent => [
							scopeType.features.add(it)
							it.traceOrigin(e)
						]]
					scopeType._annotate(EVENT_BUFFER_ANNOTATION)
				]
				
				scopeBufferType.traceOrigin(scope)
				scopeBufferType.name = sc.name + scope.featureName.toFirstUpper + 'EvBuf'
				if (!scopeBufferType.features.empty) {
					flow.features += scopeBufferType
					bt.features += _part(scope.featureName, scopeBufferType)
				}
			]
			
			bt.name = sc.name + 'EvBuf'
			bt._annotate(EVENT_BUFFER_ANNOTATION)
			bt.traceOrigin(flow)
		]
		
		if (!bufferType.features.empty) {
			flow.features += bufferType
			flow.features += _synthetic(_part("current", bufferType))
		}
	}
	
	def protected dispatch featureName(Scope it) {
		'timeEvents'	
	}

	def protected dispatch featureName(InternalScope it) {
		if ( isShadowMemberScope ) "shadow"
		else "internal"	
	}
	
	def protected dispatch featureName(InterfaceScope it) {
		'iface' + (if(name.nullOrEmpty) '' else name).asIdentifier.toFirstUpper
	}
		
	def hasBufferedEvents(Scope it) {
		it.declarations.filter(Event).exists[ e | e.direction != Direction::OUT]	
	}
		
	def eventBuffer(ExecutionFlow it) {
		eventBuffers.head
	}
	
	def hasEventBuffer (ExecutionFlow it) {
		eventBuffer !== null
	}
	
	def eventBuffers(ComplexType it) {
		features
			.filter(Property)
			.filter[ p | p.type instanceof ComplexType && p.type.isEventBuffer]
	}
	
	def ArrayList<ComplexType> eventBuffersTypes(ComplexType it, ArrayList<ComplexType> allEventBufferTypes) {
		val types = features
			.filter(Property)
			.filter[ p | p.type instanceof ComplexType && p.type.isEventBuffer]
			.map[p | p.type]
			.filter(ComplexType)
		types.forEach[evBuf |
			if(!evBuf.features.nullOrEmpty) 
				evBuf.eventBuffersTypes(allEventBufferTypes)
		]
		allEventBufferTypes += types
		allEventBufferTypes
	}

	def List<List<Declaration>> bufferEventPaths(Property it) {

		val l = newArrayList
		l.addAll(
			features
				.filter(Event)
				.filter[ e | e.isBufferEvent]
				.map[ e | newArrayList(it, e as Declaration) as List<Declaration>]
		)
		features
			.filter(Property)
			.filter[p | p.type.isEventBuffer]
			.map[ p | p.bufferEventPaths]
			.forEach[ bel | 
				bel.forEach[ be | be.add(0,it)] 
				l.addAll(bel)
			]

		return l			
	}
	
	def isBuffered(Event e) {
		(applyIncomingEventBuffer && (e.direction == Direction.IN || e instanceof TimeEvent)) || 
		(applyInternalEventBuffer && e.direction == Direction.LOCAL &&  !(e instanceof TimeEvent)) 
	}
	
	
	def isEventBuffer(Type it){
		getAnnotationOfType(EVENT_BUFFER_ANNOTATION) !== null		
	}
	
	
	def Event originEvent(Event it) {
		originTraces.filter(Event).head
	}

	def List<Expression> asExpressions(List<List<Declaration>> declPaths) {
		declPaths.map[ path | 
			path.fold(null as Expression, [ 
				expr, decl | if (expr === null) decl._ref else expr._dot(decl)
			])
		]	
	}
	
	def features(Property it) {
		if (it.type instanceof ComplexType) (type as ComplexType).features
		else #[]
	}
	
	def features(Type it) {
		if (it instanceof ComplexType) (it as ComplexType).features
		else #[]
	}
}