/**
 * Copyright (c) 2020 itemis AG - All rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * 
 */
package com.yakindu.sct.model.sexec.concepts

import com.google.inject.Inject
import com.itemis.create.base.generator.core.concepts.Documentation
import com.yakindu.base.expressions.ExpressionBuilder
import com.yakindu.base.expressions.expressions.ElementReferenceExpression
import com.yakindu.base.expressions.expressions.EventRaisingExpression
import com.yakindu.base.expressions.expressions.EventValueReferenceExpression
import com.yakindu.base.expressions.expressions.ExpressionsFactory
import com.yakindu.base.expressions.expressions.FeatureCall
import com.yakindu.base.expressions.util.ExpressionExtensions
import com.yakindu.base.types.Direction
import com.yakindu.base.types.Event
import com.yakindu.base.types.Expression
import com.yakindu.base.types.Property
import com.yakindu.base.types.TypeBuilder
import com.yakindu.base.types.typesystem.EventValueMetaFeature
import com.yakindu.sct.model.sexec.ExecutionFlow
import com.yakindu.sct.model.sexec.Method
import com.yakindu.sct.model.sexec.Sequence
import com.yakindu.sct.model.sexec.Step
import com.yakindu.sct.model.sexec.TimeEvent
import com.yakindu.sct.model.sexec.extensions.SExecExtensions
import com.yakindu.sct.model.sexec.extensions.SexecBuilder
import com.yakindu.sct.model.sexec.transformation.config.IFlowConfiguration
import com.yakindu.sct.model.stext.stext.EventDefinition
import java.util.HashMap
import java.util.List
import org.eclipse.emf.ecore.EObject
import org.eclipse.emf.ecore.EReference
import org.eclipse.emf.ecore.util.EcoreUtil

/**
 * This class defines the concept of event processing. It defines how events are processed for
 * cycle based and event driven execution. 
 * 
 * @author aterfloth
 */
class EventProcessing {


	@Inject protected extension TypeBuilder
	@Inject protected extension ExpressionBuilder
	extension ExpressionsFactory = ExpressionsFactory.eINSTANCE
	@Inject protected extension SexecBuilder
	@Inject protected extension Documentation
	
	@Inject protected extension SExecExtensions
	@Inject protected extension ExpressionExtensions
	@Inject extension StateMachineBehaviorConcept
	@Inject protected extension EventQueue
	@Inject protected extension EventBuffer
	@Inject protected extension EventValueMetaFeature
	@Inject protected extension BufferEvent
	@Inject protected extension CompletionEvent
	@Inject extension HaltBehavior
	
	@Inject protected extension IFlowConfiguration config

	public static val CLEAR_EVENT = StateMachineBehaviorConcept.CONCEPT_NAME_PREFIX + "clearEvent"
	public static val MOVE_EVENT = StateMachineBehaviorConcept.CONCEPT_NAME_PREFIX + "moveEvent"
	public static val NEXT_EVENT = StateMachineBehaviorConcept.CONCEPT_NAME_PREFIX + "nextEvent"
	public static val QUEUE_EVENT = StateMachineBehaviorConcept.CONCEPT_NAME_PREFIX + "queueEvent"
	public static val CLEAR_OUT_EVENTS = "clearOutEvents"
	public static val CLEAR_IN_EVENTS = "clearInEvents"
	public static val CLEAR_CURRENT_EVENTS = "clearCurrentEvents"
	public static val CLEAR_INTERNAL_EVENTS = "clearInternalEvents"
	public static val SWAP_IN_EVENTS = "swapInEvents"
	public static val SWAP_INTERNAL_EVENTS = "swapInternalEvents"
	public static val INTERNAL_EVENTS = #[CLEAR_OUT_EVENTS, CLEAR_IN_EVENTS, CLEAR_INTERNAL_EVENTS, SWAP_IN_EVENTS, SWAP_INTERNAL_EVENTS]


	def defineFeatures (ExecutionFlow it) {
		if (hasOutgoingEvents && applyOutgoingEventBuffer) defineClearOutEvents
		if (hasIncomingEvents) {
			if (applyIncomingEventBuffer) defineSwapInEvents
			if (cycleBased && hasCompletionTransition && applyIncomingEventBuffer) defineClearCurrentEvents
			defineClearInEvents
		}
		if (hasLocalEvents) {
			if (applyInternalEventBuffer) defineSwapInternalEvents
			defineClearInternalEvents
		}
	}


	def defineClearOutEvents(ExecutionFlow it) {
		it._method(CLEAR_OUT_EVENTS) => [ m | 
			m._type(_void)
			m._body(
				allOutEvents.map[ o |
					o._clear
				]
			)
		]
	}
	
	def protected Iterable<EObject> allOutEvents(ExecutionFlow it) {
		#[it.outgoingEvents].flatten
	}
	
	
	def defineClearInEvents(ExecutionFlow it) {
		it._method(CLEAR_IN_EVENTS) => [ m | 
			m._type(_void)
			
			m._body(
				allInEvents.map[ i |
					i._clear
				]
			)
		]
	}
	
	def protected Iterable<EObject> allInEvents(ExecutionFlow it) {
		#[it.incomingEvents, it.timeEvents].flatten
	}
	
	
	def defineClearInternalEvents(ExecutionFlow it) {
		it._method(CLEAR_INTERNAL_EVENTS) => [ m | 
			m._type(_void)
			
			m._body(
				localEvents.map[ i |
					i._clear
				]
			)
		]
	}
	
	def defineClearInternalEventsOp(ExecutionFlow it) {
		_op(CLEAR_INTERNAL_EVENTS, _void) => [ m | 
			m.implementation =_block => [ b |
				localEvents.forEach[exp |
					b.expressions += _assignment(exp._ref,_false)
				] 
			]
		]
	}


	def defineSwapInEvents(ExecutionFlow it) {
		it._method(SWAP_IN_EVENTS) => [ m | 
			m._type(_void)
			
			m._body(
				it.bufferEventExpressions.incoming
					.map[ e |
						e.event.originEvent._move(e)
					]
			)
		]
	}
	
	def defineClearCurrentEvents(ExecutionFlow it) {
		it._method(CLEAR_CURRENT_EVENTS) => [ m | 
			m._type(_void)
			
			m._body(
				#[it.bufferEventExpressions.incoming, it.bufferEventExpressions.internal].flatten.map [ exp |
					exp._clear
				]
			)
		]
	}
	
	//TODO: This should reside outside of Sexec
	def defineSwapInEventsOp(ExecutionFlow it) {
		_op(SWAP_IN_EVENTS, _void) => [ m |			
			m.implementation =_block => [ b |
				it.bufferEventExpressions.incoming.forEach[exp |
					b.expressions += _assignment(exp,exp.event.originEvent._ref)
					b.expressions += _assignment(exp.event.originEvent._ref,_false)
				] 
			]			
		]
	}
	
	//TODO: This should reside outside of Sexec
	def defineSwapInternalEventsOp(ExecutionFlow it) {
		_op(SWAP_INTERNAL_EVENTS, _void) => [ m | 
			
			m.implementation = _block => [ b |
				it.bufferEventExpressions.incoming.forEach[ exp |
					b.expressions += _assignment(exp,_false)
				]
				it.bufferEventExpressions.internal.forEach[ exp |
					b.expressions += _assignment(exp,exp.event.originEvent._ref)
					b.expressions += _assignment(exp.event.originEvent._ref,_false)
				]
			]
		]
	}
	
	//TODO: This should reside outside of Sexec
	def defineClearCurrentEventsOp(ExecutionFlow it) {
		_op(CLEAR_CURRENT_EVENTS, _void) => [ m | 
			
			m.implementation = _block => [ b |
				it.bufferEventExpressions.incoming.forEach[ exp |
					b.expressions += _assignment(exp,_false)
				]
				it.bufferEventExpressions.internal.forEach[ exp |
					b.expressions += _assignment(exp,_false)
				]
			]
		]
	}

	def defineSwapInternalEvents(ExecutionFlow it) {
		it._method(SWAP_INTERNAL_EVENTS) => [ m | 
			m._type(_void)
			
			m._body(
				_sequence(
					it.bufferEventExpressions.incoming
					.map[ e | 
						e._clear()
					]						
				) => [ comment = "When processing internal events all incoming events are processed and must be cleared from current buffer."],
				_sequence(
					it.bufferEventExpressions.internal
					.map[ e | 
						e.event.originEvent._move(e)
					]	
				) => [ comment = "Swap all internal events."]
			)
		]
	}
	

	def Step _eventProcessing(ExecutionFlow it, Step body) {
		
		_sequence(
			_clearOutEvents,
			_activateInitialEvents,
			_eventLoop(
				_completionLoop(_sequence(body, _clearCurrentEvents))
			)
		)
	}
	
	
	protected def Step _eventLoop(ExecutionFlow it, Step body) { 
		if ( requiresEventLoop ) {
			_sequence(
				_do(_sequence(
					body,
					_activateNextEvents
				))._while(_hasCurrentEvents._andIsNotHalted(it)) => [name = "doWhileEvent"]
			)
		} else {
			body
		}
	} 
	
	
	def requiresEventLoop(ExecutionFlow it) {
		(isCycleBased && hasLocalEvents && applyInternalEventBuffer) 
		|| requiresEventQueue	
	}
	
	
	protected def Step _activateInitialEvents(ExecutionFlow it) {
		if (isCycleBased) {
			_swapIncomingEvents
		} else if (requiresEventQueue) {
			_conceptSequence(NEXT_EVENT)
		} else {
			_empty
		}
	}
	
	protected def Step _activateNextEvents(ExecutionFlow it) {
		if (isCycleBased) {
			_swapInternalEvents
		}else {
			_empty
		}
		
	}

	protected def Expression _hasCurrentEvents(ExecutionFlow it) {
		if (isCycleBased) {
			bufferEventExpressions.internal.reduce[r1, r2| r1._or(r2)]
		} else if (requiresEventQueue) {
			_conceptOperation(NEXT_EVENT,_eventList)._ref
		}
		 else 
			_eventList	
	}
	
	protected def Expression _eventList(ExecutionFlow it){
		#[incomingEvents,localEvents,timeEvents,submachineEvents]
				.flatten
				.map[ ev | ev._ref as Expression ]
				.reduce[ e1, e2 | e1._or(e2) ]
	}
	
	protected def Step _clearCurrentEvents(ExecutionFlow it) {
		_sequence(
			_clearCurrentInEvents,
			_clearInEvents._when(!(isCycleBased && applyIncomingEventBuffer)),
			_clearInternalEvents._when(!(isCycleBased && applyInternalEventBuffer))			
		)
	}
	

	def Step _clearOutEvents(ExecutionFlow it) {
		if ( clearOutEvents !== null ) clearOutEvents._call._statement 
		else _empty	
	}


	def Step _clearCurrentInEvents(ExecutionFlow it) {
		if ( clearCurrentInEvents !== null) clearCurrentInEvents._call._statement 
		else _empty	
	}
	
	def Step _clearInEvents(ExecutionFlow it) {
		if ( clearInEvents !== null) clearInEvents._call._statement 
		else _empty	
	}

	def Step _clearInternalEvents(ExecutionFlow it) {
		if ( clearInternalEvents !== null) clearInternalEvents._call._statement 
		else _empty	
	}

	def Step _swapIncomingEvents(ExecutionFlow it) {
		if (swapIncomingEvents !== null)
			swapIncomingEvents._call._statement
		else _empty
	}

	def Step _swapInternalEvents(ExecutionFlow it) {
		if (swapInternalEvents !== null)
			swapInternalEvents._call._statement
		else _empty
	}
	
	def Step _clear(EObject it){
		_conceptSequence(CLEAR_EVENT, it)	
	}
	
	def Step _move(EObject source, EObject target){
		_conceptSequence(MOVE_EVENT, source, target)	
	}
	
	
	def transformEventAccess(ExecutionFlow flow) {
		flow.transformEventValueAcces
		if ( ! flow.hasEventBuffer ) return 
		
		val bufferedEvents = #[flow.incomingEvents, flow.timeEvents, flow.localEvents].flatten.filter[isBuffered].toSet

		val allEventReferences = flow.eAllContents
										.filter(Expression)
										.filter[ e | 
											bufferedEvents.contains(e.referenceOrFeature)
											&& !(e.eContainer instanceof FeatureCall)
											&& !(e.eContainer instanceof EventRaisingExpression)
										]
										
		val HashMap<EObject, Expression> allEventAccessExpression = newHashMap
		flow.bufferEventExpressions.forEach[ e | if (e.referenceOrFeature !== null) allEventAccessExpression.put(e.referenceOrFeature, e)]
													
		allEventReferences.forEach[ expression | 
			val event = expression.referenceOrFeature as Event
			val bufferEvent = event.createBufferEvent
			val bufferEventExpression = EcoreUtil.copy(allEventAccessExpression.get(bufferEvent))
			
			if ( expression.eContainer instanceof EventValueReferenceExpression ) {
				expression.eContainer.eContainer.substitute(expression.eContainer, bufferEventExpression._meta(bufferEvent.valueFeature))
			} else {
				expression.eContainer.substitute(expression, bufferEventExpression)			
			}
		]
				
	}
	
	def transformEventValueAcces(EObject object){
		object.eAllContents.filter(FeatureCall).filter(f | f.feature !== null && f.feature.isEventValueProperty).forEach[ expression |
			val evRefExp = createEventValueReferenceExpression
			expression.eContainer.substitute(expression, evRefExp)		
			evRefExp.value = expression.owner
		]		
	}
	
	/* Substitutes the EObject oldValue which is contained in EObject parent by EObject newValue.
	 * If oldValue is not contained in parent then nothing happens. 
	 */
	protected def substitute(EObject parent, EObject oldValue, EObject newValue) {
		if (parent !== null ) {
			val ref = parent.eClass.getEAllStructuralFeatures
				.filter(EReference)
				.filter[ f | 
					f.isChangeable 
					&& !f.isDerived 
					&& f.isContainment
					&& parent.eIsSet(f)
				]
				.findFirst[ f | 
					if (f.isMany) {
						(parent.eGet(f) as List<EObject>).contains(oldValue)
					} else {
						parent.eGet(f) === oldValue
					}
				]
			
			EcoreUtil.replace(parent, ref, oldValue, newValue);
		}	
	}
		
	
	protected def dispatch EObject referenceOrFeature(Expression e) { return null }
	protected def dispatch EObject referenceOrFeature(ElementReferenceExpression e) { return e.reference }
	protected def dispatch EObject referenceOrFeature(FeatureCall e) { return e.feature }
	
	
	
	def dispatch Event event(Sequence it){
		it.getParameter as Event
	}
	
	def protected List<Expression> bufferEventExpressions(ExecutionFlow it) {
		it
			.eventBuffer
			.bufferEventPaths
			.asExpressions		
	}
	
	def protected internal(List<Expression> it) {
		filter[ e | 
			val o = e.event.originEvent;
			o instanceof EventDefinition && o.direction == Direction.LOCAL
		]
	}
	
	def protected incoming(List<Expression> it) {
		filter[ e | 
			val o = e.event.originEvent;
			(o instanceof EventDefinition && o.direction == Direction.IN) || (o instanceof TimeEvent) 
		]
	}
	
	def dispatch Event event(ElementReferenceExpression it){
		it.reference as Event
	}
	
	def Property valueFeature(Event it) {
		it.metaFeatures.filter(Property).filter[ p | p.name == "value"].head
	} 
	
	def dispatch Event event(FeatureCall it){
		it.feature as Event
	}
	
	def internalEventMethod(Method it){
		INTERNAL_EVENTS.exists[e | name == e]
	}

	def Method clearOutEvents(ExecutionFlow it) {
		features.filter( typeof(Method) ).filter( m | m.name == CLEAR_OUT_EVENTS).head
	}
	
	def Method clearCurrentInEvents(ExecutionFlow it) {
		features.filter( typeof(Method) ).filter( m | m.name == CLEAR_CURRENT_EVENTS).head
	}
	
	def Method clearInEvents(ExecutionFlow it) {
		features.filter( typeof(Method) ).filter( m | m.name == CLEAR_IN_EVENTS).head
	}
	
	def Method clearInternalEvents(ExecutionFlow it) {
		features.filter( typeof(Method) ).filter( m | m.name == CLEAR_INTERNAL_EVENTS).head
	}
	
	def Method swapInternalEvents(ExecutionFlow it) {
		features.filter( typeof(Method) ).filter( m | m.name == SWAP_INTERNAL_EVENTS).head
	}

	def Method swapIncomingEvents(ExecutionFlow it) {
		features.filter( typeof(Method) ).filter( m | m.name == SWAP_IN_EVENTS).head
	}
	
}