/**
 * Copyright (c) 2025 itemis AG - All rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * 
 */
package com.yakindu.sct.model.sexec.concepts

import com.google.inject.Inject
import com.google.inject.Singleton
import com.itemis.create.base.generator.core.codepattern.BaseExpressionCode
import com.yakindu.base.base.NamedElement
import com.yakindu.base.expressions.ExpressionBuilder
import com.yakindu.base.expressions.expressions.AssignmentExpression
import com.yakindu.base.expressions.expressions.BinaryExpression
import com.yakindu.base.expressions.expressions.BlockExpression
import com.yakindu.base.expressions.expressions.DeclarationExpression
import com.yakindu.base.expressions.expressions.ElementReferenceExpression
import com.yakindu.base.expressions.expressions.TimeEventSpec
import com.yakindu.base.types.Expression
import com.yakindu.base.types.Property
import com.yakindu.base.types.TypeBuilder
import com.yakindu.base.types.adapter.OriginTracing
import com.yakindu.sct.model.sexec.ExecutionFlow
import com.yakindu.sct.model.sexec.TimeEvent
import com.yakindu.sct.model.sexec.extensions.SExecExtensions
import com.yakindu.sct.model.sexec.transformation.MapTimeEventValue
import com.yakindu.sct.model.sexec.transformation.SexecElementMapping
import com.yakindu.sct.model.sgraph.Statechart
import com.yakindu.sct.model.stext.concepts.LoggerTypeLibrary
import com.yakindu.sct.model.stext.concepts.StatechartAnnotations
import com.yakindu.sct.model.stext.stext.VariableDefinition
import java.util.List
import org.eclipse.emf.ecore.EObject
import org.eclipse.emf.ecore.util.EcoreUtil
import org.eclipse.xtext.EcoreUtil2
import com.yakindu.base.expressions.expressions.UnaryExpression
import com.yakindu.base.expressions.expressions.ArgumentExpression

/**
 * This class is responsible to produce the Debug logs for every statechart expression in interest.
 * It relies only on the base types and expressions.
 * 
 * @author laszlo kovacs
 */
//TODO: This class should be independent of SExec 
@Singleton
class ExecutionDebugging {
	
	public static val DEBUG_LOG = "debugLog"
	
	@Inject extension TypeBuilder
	@Inject extension ExpressionBuilder
	@Inject extension StatechartAnnotations
	@Inject extension OriginTracing
	@Inject extension BaseExpressionCode
	@Inject protected extension LoggerTypeLibrary
	@Inject protected extension MapTimeEventValue
	
	@Inject protected extension SExecExtensions
	@Inject protected extension SexecElementMapping
	
	def addLoggerFeatures(ExecutionFlow flow){
		flow.createDebugLog
	}
	
	def getDebugLog(ExecutionFlow it) {
		flow.defaultScope.members.filter(Property).findFirst[DEBUG_LOG == name]
	}
	
	def protected create it : _variable(DEBUG_LOG) createDebugLog(ExecutionFlow flow) {
		_public
		typeSpecifier = loggerType._typeSpecifier
		flow.getDefaultScope.members += it
	}
	
	def String formatSpecifierValue(EObject it) {
		switch it{
			case _integer:
				return "%d"
			case _real:
				return "%f"
			case _boolean:
				return "%s"
			case _string:
				return "%s"
			default:
				return "%d"
		}
	}
	
	def debugLogInstance(EObject it) {
		statechart.create.debugLog
	}
	
	def dispatch BlockExpression logFor(Property logger, Expression exp){
		_block()
	}
	
	def dispatch BlockExpression logFor(Property logger, TimeEvent exp){
		val allLog = _block()
		allLog.expressions += logger._ref._dot(loggerWrite,(exp.origin as EObject).code.toString._string)
		if((exp.origin as TimeEventSpec).value instanceof ElementReferenceExpression){
			val elRef = ((exp.origin as TimeEventSpec).value as ElementReferenceExpression).reference as NamedElement
			allLog.expressions += logger._ref._dot(loggerWrite,'''  < «elRef.name» = «exp.scheduleTimeValue.type.formatSpecifierValue»'''.toString._string, exp.scheduleTimeValue._ref)
		}
		allLog
	}
	
	def BlockExpression preLog(AssignmentExpression exp){
		val preLog = _block()
		preLog.expressions += exp.debugLogInstance._ref._dot(loggerWrite, exp.code.toString._string)
		exp.eAllContents.filter(ElementReferenceExpression).filter[it !== exp.varRef].filter[reference instanceof VariableDefinition].forEach[ vdRef |
			val vd = vdRef.reference as VariableDefinition
			val vdLocal = _declare(_variable("r_" + vd.name + "_local", vd.type, vd._ref))
			EcoreUtil.replace(vdRef,vdLocal.declaration._ref)
			if(!preLog.expressions.filter(DeclarationExpression).exists[e | e.declaration.name == vdLocal.declaration.name]){
				preLog.expressions += vdLocal
				preLog.expressions += exp.debugLogInstance._ref._dot(loggerWrite,'''  > «vd.name» = «vd.type.formatSpecifierValue»'''.toString._string, vdLocal.declaration._ref)
			}	
		]
		preLog
	}
	
	def BlockExpression afterLog(AssignmentExpression exp){
		val afterLog = _block()
		exp.eAllContents.filter(ElementReferenceExpression).filter[it === exp.varRef].filter[reference instanceof VariableDefinition].forEach[ vdRef |
			val vd = vdRef.reference as VariableDefinition
			val vdLocal = _declare(_variable("l_" + vd.name + "_local", vd.type, vd._ref))
			afterLog.expressions += vdLocal
			afterLog.expressions += exp.debugLogInstance._ref._dot(loggerWrite,'''  < «vd.name» = «vd.type.formatSpecifierValue»'''.toString._string, vdLocal.declaration._ref)
		]
		afterLog
	}
	
	def dispatch BlockExpression logFor(Property logger, BinaryExpression exp){
		logger.createLogForExpression(exp)
	}
	
	def dispatch BlockExpression logFor(Property logger, UnaryExpression exp){
		logger.createLogForExpression(exp)
	}
	
	def dispatch BlockExpression logFor(Property logger, ArgumentExpression exp){
		logger.createLogForExpression(exp)
	}
	
	def protected  BlockExpression createLogForExpression(Property logger, Expression exp){
		val allLog = _block()
		allLog.expressions += logger._ref._dot(loggerWrite,'''«exp.code» : «_boolean.formatSpecifierValue»'''.toString._string, true.toString._string)
		exp.eAllContents.filter(ElementReferenceExpression).map[reference].filter(VariableDefinition).toSet.forEach[ vd |
			val vdLocal = _variable(vd.name + "_local", vd.type, vd._ref)
			allLog.expressions += logger._ref._dot(loggerWrite,'''  > «vd.name» = «vd.type.formatSpecifierValue»'''.toString._string, vdLocal._ref)
		]
		allLog
	}
	
	
	
	def reduceLocalVariables(List<Expression> allActionBlocks){
		val reducedExpressions = newArrayList
		val seenNames = newHashSet
	    val noDuplDecl = allActionBlocks
	        .filter(DeclarationExpression)
	        .filter[ decl |
	            if (!seenNames.contains(decl.declaration.name)) {
	                seenNames.add(decl.declaration.name)
	                true
	            } else
	                false
	        ].toList
		reducedExpressions.addAll(noDuplDecl)
		allActionBlocks.forEach[ exp |
			if(exp instanceof DeclarationExpression){
				var decl =  (exp as DeclarationExpression).declaration as Property
				reducedExpressions.add(decl._ref._assign(decl.initialValue))
			} else
				reducedExpressions.add(exp)
		]
		reducedExpressions
	}

	def dispatch boolean appliesDebugExecution(ExecutionFlow it) {
		(it.sourceElement as Statechart).appliesDebugExecution
	}
	
	def dispatch boolean appliesDebugExecution(Statechart it) {
		isDebugExecution
	}
	
	def dispatch boolean appliesDebugExecution(EObject it) {
		EcoreUtil2.getContainerOfType(it, Statechart).appliesDebugExecution
	}
	
	def dispatch boolean appliesDebugExecution(Void it) {
		false
	}
	
}