/**
 * Copyright (c) 2021 itemis AG - All rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * 
 */
package com.yakindu.sct.model.sexec.concepts

import com.google.inject.Inject
import com.google.inject.Singleton
import com.yakindu.base.expressions.ExpressionBuilder
import com.yakindu.base.types.Property
import com.yakindu.base.types.TypeBuilder
import com.yakindu.sct.model.sexec.ExecutionFlow
import com.yakindu.sct.model.sexec.Method
import com.yakindu.sct.model.sexec.extensions.SexecBuilder
import com.yakindu.sct.model.sexec.extensions.StateVectorExtensions
import com.yakindu.sct.model.sexec.transformation.ReactMethod

/**
 * This class defines the general concept of a micro step execution. 
 * A micro step is the smallest atomic run to completion step. 
 * It may be composed to superstep or event triggered multi step execution.  
 * 
 * @author aterfloth
 */
@Singleton class MicroStep {

	@Inject protected extension CompletionEvent
	@Inject protected extension TypeBuilder
	@Inject protected extension ExpressionBuilder
	@Inject protected extension SexecBuilder sexec
	@Inject protected extension StateVectorExtensions
	@Inject protected extension ReactMethod

	public static val STATE_VECTOR_POSITION = "stateConfVectorPosition"
	public static val MICRO_STEP = "microStep"

	def defineFeatures(ExecutionFlow it) {
		if (needsStateVectorPosition) {
			it.features += _synthetic(_variable(STATE_VECTOR_POSITION, _integer)) 
		}

		defineMicroStep
	}

	def Method microStep(ExecutionFlow it) {
		features.filter(typeof(Method)).filter(m|m.name == MICRO_STEP).head
	}

	def create m : _method(MICRO_STEP) defineMicroStep(ExecutionFlow flow) {

		val transitioned = _variable("transitioned", _integer)

		m._type(_void)
		m._body
		flow.features += m

		if (flow.needsStateVectorPosition) {
			m.body += _local(transitioned)._with((-1)._integer) => [
				comment = "transitioned is used to track the transitions performed by calling the stater react behavior"
			]
			m.body += flow.stateVectorPosition._assign(0._integer)
		}

		for (var i = 0; i < flow.stateVector.size; i++) {
			val j = i

			val reactingLeafStatesOnPosition = flow.states.filter[s|s.isLeaf && s.stateVector.first == j].filter [ s |
				s.reactMethod !== null
			]

			if (! reactingLeafStatesOnPosition.empty) {
				val stateSwitch = _switch => [
					it.stateConfigurationIdx = j
					reactingLeafStatesOnPosition.forEach [ state |
						_case(
							state,
							if (flow.needsStateVectorPosition && j < flow.stateVector.size - 1)
								transitioned._assign(state.callReact(transitioned._ref))
							else if (flow.needsStateVectorPosition)
								state.callReact(transitioned._ref)._statement
							else if (state.reactMethod !== null)
								state.callReact((-1)._integer)._statement
						)
					]
				]

				m.body.steps += if (i != 0) {
					sexec._if(flow.stateVectorPosition._ref._smaller(i._integer))._then(stateSwitch)
				} else {
					stateSwitch
				}

				m.body.steps += _sequence
			}
		}
	}

	def Property stateVectorPosition(ExecutionFlow it) {
		features.filter(Property).filter(m|m.name == STATE_VECTOR_POSITION).head
	}

	def needsStateVectorPosition(ExecutionFlow it) {
		it.stateVector.size > 1
	}
}
