/**
 * Copyright (c) 2020 itemis AG - All rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * 
 */
package com.yakindu.sct.model.sexec.extensions

import com.google.inject.Inject
import com.yakindu.base.types.TypesUtil
import com.yakindu.sct.model.sexec.ExecutionFlow
import com.yakindu.sct.model.sexec.ExecutionRegion
import com.yakindu.sct.model.sexec.ExecutionScope
import com.yakindu.sct.model.sexec.ExecutionState
import com.yakindu.sct.model.sexec.SexecFactory
import com.yakindu.sct.model.sexec.StateVector
import com.yakindu.sct.model.sgraph.FinalState
import java.util.ArrayList
import java.util.List
import com.yakindu.sct.model.stext.stext.SubmachineReferenceExpression

/**
 * 
 * @author Axel Terfloth
 */
class StateVectorExtensions {
	
	@Inject protected extension TypesUtil
	
	def factory() { SexecFactory::eINSTANCE }
	
	def last(StateVector sv) {
		sv.offset + sv.size -1
	}
	
	def first(StateVector sv) {
		sv.offset
	}
	
	/**
	 * Returns a states impact state vector. 
	 * 
	 * A regular states state vector defines the range of the state configuration vector where the state or its's sub states are stored if they are active.
	 * Nevertheless a state may have a larger impact on a state configuration vector if sibling states in the same region cover a larger range. 
	 * In these cases these additional positions are 'nulled' if the state is active. So a states impact vector is its regions state vector. 
	 * This impact vector is also used to determine when parent behavior must be executed.
	 */
	def dispatch StateVector impactVector(ExecutionState it) { 
		if (superScope instanceof ExecutionState && sourceElement instanceof SubmachineReferenceExpression)
			stateVector 
		else 
			superScope.impactVector
	}


	/** 
	 * The impact vector of any other execution scope is the native stateVector of that scope.
	 */
	def dispatch StateVector  impactVector(ExecutionRegion it) { 
		if ((it.superScope.subScopes.lastOrNull === it) && (it.superScope.impactVector.last > it.stateVector.last))
			return factory.createStateVector => [ sv | 
				sv.offset = it.stateVector.offset
				sv.size = it.superScope.impactVector.last - sv.offset + 1
			] 
		else 
			return stateVector
	}


	/** 
	 * The impact vector of any other execution scope is the native stateVector of that scope.
	 */
	def dispatch StateVector impactVector(ExecutionScope it) { stateVector }
		
	/**
	 * This function calculates the final state impact vector.
	 * This vector is an array of the size of the state machines state vector. 
	 * Each array slot holds a List of ExecutionState instances that map to FinalState elements
	 * in the orginal model. The assignment of a final state to a slot means that if this state is 
	 * active it has impact on the vector slot. Here two cases have to be distinguished:
	 * 
	 * 1. The final state is directly assigned to the slot (the states vector offset). If the state is active
	 *    then the slot will contain a reference to this state
	 * 2. If the parent region contains orthogonal states then all other vector positions covered by the region
	 *    are empty (refer to no state) if the state is active so there is an indirect impact on the vector. 
	 */
	def List<ExecutionState>[] finalStateImpactVector(ExecutionFlow flow) {

		// we use a vector of list of states to mark which final state has impact on the different slots.
		val List<ExecutionState>[] vector = newArrayOfSize(flow.stateVector.size)
		
		// process all final states and mark all vector positions with the states 
		flow.states.filter[s | s.sourceElement instanceof FinalState].forEach[ fs |
			// the state vector of the final states region is relevant since the final state has impact on its 
			// own vector position and all other positions covered by the region.
			// > if a final state is active then all other positions must contain the null state.
			var regionVector = fs.superScope.stateVector
			for ( i : regionVector.offset ..< regionVector.offset + regionVector.size) {
				// create a list for the slot if it does not already exist.
				if (vector.get(i) === null) {
					vector.set(i, new ArrayList<ExecutionState>())
				}
				// and add the final state to it if it has meaningful addition to the check.
				// that means that the offset is relevant for the given statevector position
				// or if the current layer of statevector does not contain check for null_state yet
				if(fs.stateVector.offset === i || !vector.get(i).exists[state | state.stateVector.offset !== i])
					vector.get(i).add(fs);  
				// so all vector positions are marked with all those final states that have impact on the slot.
			}
		]			
				
		return vector
	}
	
	
	/** 
	 * Checks if all state vector positions are covered by states. 
	 */
	def boolean isCompletelyCovered(List<ExecutionState>[] finalStateImpactVector) {
		finalStateImpactVector.forall[ l | (l !== null) && (!l.isEmpty) ]
	}
	
	
	
}