/**
 * Copyright (c) 2020 itemis AG - All rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * 
 */
package com.yakindu.sct.model.sexec.transformation

import com.google.inject.Inject
import com.yakindu.base.expressions.ExpressionBuilder
import com.yakindu.base.expressions.expressions.ElementReferenceExpression
import com.yakindu.base.types.Expression
import com.yakindu.base.types.Property
import com.yakindu.base.types.TypeBuilder
import com.yakindu.sct.model.sexec.ExecutionFlow
import com.yakindu.sct.model.sexec.ExecutionNode
import com.yakindu.sct.model.sexec.ExecutionState
import com.yakindu.sct.model.sexec.Method
import com.yakindu.sct.model.sexec.Sequence
import com.yakindu.sct.model.sexec.Step
import com.yakindu.sct.model.sexec.concepts.CompletionEvent
import com.yakindu.sct.model.sexec.extensions.SExecExtensions
import com.yakindu.sct.model.sexec.extensions.SexecBuilder
import com.yakindu.sct.model.sexec.extensions.ShadowEventExtensions
import com.yakindu.sct.model.sexec.extensions.StateVectorExtensions
import com.yakindu.sct.model.sgraph.FinalState
import com.yakindu.sct.model.sgraph.RegularState
import com.yakindu.sct.model.sgraph.Statechart
import com.yakindu.sct.model.sgraph.util.SgraphExtensions
import com.yakindu.sct.model.stext.concepts.CompletionTransition
import com.yakindu.sct.model.stext.concepts.RegularTransition
import com.yakindu.sct.model.stext.concepts.StatechartAnnotations
import com.yakindu.sct.model.sgraph.State
import com.yakindu.sct.model.stext.stext.SubmachineReferenceExpression
import org.eclipse.xtext.EcoreUtil2
import com.yakindu.sct.model.sexec.concepts.SubMachine.SubmachineTypeLibrary

/**
 * React method is an artifact concepts that is created for each state machine state and the statechart
 * itself. It defines the react behavior for each reactive element of the state machine. 
 * 
 * @author Axel Terfloth - terfloth@itemis.de
 *  
 */
class ReactMethod {

	@Inject extension SexecElementMapping mapping
	@Inject extension SexecFactoryExtensions __sexec
	@Inject extension SExecExtensions
	@Inject extension SgraphExtensions sgraph
	@Inject extension StateVectorExtensions
	@Inject extension StatechartAnnotations

	@Inject extension TypeBuilder typeBuilder
	@Inject extension ExpressionBuilder expr
	@Inject extension SexecBuilder sexec

	@Inject extension CompletionEvent
	@Inject extension CompletionTransition
	@Inject extension RegularTransition
	@Inject extension StructureMapping
	@Inject extension ShadowEventExtensions
	@Inject extension SubmachineTypeLibrary

	/**
	 * Declares the react methods for all ExecutionNode objects. This are the ExecutionStates and the ExecutionFlow itself.
	 */
	def declareReactMethods(ExecutionFlow it) {
		flow.declareReactMethod
		states.forEach[s|s.declareReactMethod]
	}

	/**
	 * Define the react methods for all ExecutionNode objects. This are the ExecutionStates and the ExecutionFlow itself.
	 */
	def defineReactMethods(ExecutionFlow it) {
		if(flow.reactMethod !== null) flow.defineReactMethod
		states.forEach[s|s.defineReactMethod]
	}

	def hasShadowEvents(ExecutionFlow it) {
		!shadowEvents.nullOrEmpty
	}

	def hasSubmachineOutEventCalls(ExecutionFlow it) {
		val sct = it.sourceElement as Statechart
		return !sct.submachineOutEventCalls.nullOrEmpty
	}

	def dispatch ExecutionNode declareReactMethod(RegularState state) {
		state.create.declareReactMethod
	}

	def dispatch ExecutionNode declareReactMethod(Statechart statechart) {
		statechart.flow.declareReactMethod
	}

	def dispatch ExecutionNode declareReactMethod(ExecutionNode node) {
		node => [
			features.add(sexecFactory.createMethod => [ m |
				m.name = "react"
				m._type(_integer)
				m._param("transitioned_before", _integer)
			])
		]
	}

	def dispatch ExecutionNode declareReactMethod(ExecutionFlow node) {
		node => [
			features.add(sexecFactory.createMethod => [ m |
				m.name = "react"
				m._type(_integer)
				m._param("transitioned_before", _integer)
			])
		]
	}

	def defineReactMethod(ExecutionFlow it) {
		val tryTransitionParam = reactMethod.param('transitioned_before')

		reactMethod => [
			body = _sequence(
				flow.createLocalReactionSequence,
				_return(tryTransitionParam._ref)._statement
			) => [comment = "State machine reactions."]
		]
	}

	def defineReactMethod(ExecutionState execState) {
		val source = execState.sourceElement 
		
		return execState.defineReactMethod(source)
	}


	def dispatch defineReactMethod(ExecutionState execState, RegularState state) {

		val reactMethod = execState.reactMethod
		val childFirst = state.getStatechart.isChildFirstExecution

		val parentNode = if(state.parentState !== null) state.parentState.create else execState.flow
		val processParent = parentNode !== null &&
			(	    ( childFirst && parentNode.impactVector.last == execState.impactVector.last) ||
				(!childFirst && parentNode.stateVector.offset == execState.stateVector.offset)
							   )

		if (state.getStatechart.interleaveLocalReactions) {

			val tryTransitionParam = reactMethod.param('transitioned_before')
			val didTransitionVariable = _variable("transitioned_after", _integer)
			
			val transitionSeq = 
				if ((processParent && !childFirst) || (execState.sourceElement instanceof FinalState && (processParent && childFirst ))) 
					parentNode.callReact(_ref(tryTransitionParam))
				else
					_ref(tryTransitionParam)
			
			val reactSeq = execState.createReactionSequence(_empty, didTransitionVariable)
			
			val localReact =
			 if (execState.localReactions.size > 0 || (processParent && childFirst ))
				_sequence(execState.createLocalReactionSequence => [
					if (processParent && childFirst)
						_step(didTransitionVariable._assign(parentNode.callReact(_ref(tryTransitionParam))) => [comment = "invoke parent reactions" ])
				])
			else _empty			
			
			val localSeq =
				if(state.hasRegularTransitions && !localReact.isEmpty &&
					((reactSeq.eAllContents.filter(ElementReferenceExpression).filter[reference === didTransitionVariable].size > 0) ||
						(transitionSeq.reference instanceof Method)
					)
				)
					sexec._if(didTransitionVariable._ref._equals(tryTransitionParam._ref))
						._then( localReact => [comment = "then execute local reactions."]) 
						=> [comment = "If no transition was taken"]
				else 
					localReact => [comment = "Always execute local reactions."]
					
			val reactionBody = 
				_sequence(
					if(!reactSeq.isEmpty) 
						_sequence(sexec._if(didTransitionVariable._ref._smaller(execState.stateVector.first._integer))._then(reactSeq), localSeq)
					else localSeq
				)

			val stateReactions = 
			if(((reactionBody !== null && !reactionBody.isEmpty) || state.hasCompletionTransition) && !(execState.sourceElement instanceof FinalState && (processParent && childFirst ))){
				_sequence(
					_local(didTransitionVariable)._with(
						transitionSeq
					),
					if (execState.flow.hasCompletionTransition) {
						if (state.hasCompletionTransition)
							execState.createCompletionReactionSequence(_sequence(
								reactionBody
							))
						else if (reactionBody !== null && ! reactionBody.isEmpty && !(execState.sourceElement instanceof FinalState)) {
							sexec._if(_not(execState.flow.doCompletionProperty._ref)) => [
								_then(reactionBody)
							]
						}
						else
							_empty
	
					} else
						reactionBody,
					_return(didTransitionVariable._ref)._statement
				)
			} else _sequence(_return(transitionSeq)._statement)

			reactMethod.body = stateReactions

		} else {

			throw new RuntimeException("Non interleaved local reactions not supported");
		}

		reactMethod.body.comment = 'The reactions of state ' + state.name + '.'

		return reactMethod
	}
	
	def dispatch defineReactMethod(ExecutionState execState, SubmachineReferenceExpression subMachineRef) {
		val reactMethod = execState.reactMethod
		val childFirst = subMachineRef.getStatechart.isChildFirstExecution
		
		val state = EcoreUtil2.getContainerOfType(subMachineRef, State)
		val machineRef = subMachineRef.submachine.copy
		
		val parentNode = if(state !== null) state.create else execState.flow
		val processParent = parentNode !== null &&
			(	
				( childFirst && parentNode.impactVector.last == execState.impactVector.last) 
				|| (!childFirst && parentNode.stateVector.offset == execState.stateVector.offset)
			)
		
		val tryTransitionParam = reactMethod.param('transitioned_before')
		val didTransitionVariable = _variable("transitioned_after", _integer)
		val parentCall = _sequence(didTransitionVariable._assign(parentNode.callReact(_ref(didTransitionVariable))) => [comment = "invoke parent reactions" ])
		
		reactMethod.body = _sequence(_verbatim("// foo")._statement)
		reactMethod.body += _local(didTransitionVariable)._with(tryTransitionParam._ref)
		if (processParent && !childFirst) {
			reactMethod.body += parentCall
		}
		reactMethod.body += sexec._if(machineRef._call(submachineInterfaceRunSubmachineCycle)._equals(_true))
									._then(didTransitionVariable._assign(execState.stateVector.last._integer))
		if (processParent && childFirst) {
			reactMethod.body += parentCall
		}
		
		reactMethod.body += _return(didTransitionVariable._ref)._statement
		reactMethod.body.comment = 'The reactions of sub machine ' + execState.name + '.'
		
		return reactMethod
	}
	

	def dispatch callReact(ExecutionState state, Expression p) { _call(state.reactMethod)._with(p) }

	def dispatch callReact(ExecutionFlow flow, Expression p) { _call(flow.reactMethod)._with(p) }

	def Method reactMethod(ExecutionNode it) {
		features.filter(typeof(Method)).filter(m|m.name == "react").head
	}

	def Sequence createLocalReactionSequence(ExecutionNode state) {

		_sequence(
			state.reactions.filter(r|! r.transition).map [ lr |
				_if(lr.check.newRef)._then(lr.effect.newCall)
			]
		)
	}

	def Sequence createReactionSequence(ExecutionState state, Step localStep, Property transitionImpactVar) {
		val cycle = _sequence
		cycle.name = "react"

		var localSteps = _sequence
		if(localStep !== null) localSteps.steps += localStep
		if(localSteps.steps.empty) localSteps = null

		val transitionReactions = state.reactions.filter(r|r.transition && ! r.unchecked).toList
		val transitionStep = transitionReactions.reverseView.fold(localSteps as Step, [ s, reaction |
			{
				sexec._if(reaction.check.newRef)
				._then(_sequence(
					reaction.effect.newCall,
					transitionImpactVar._assign(state.stateVector.last._integer)
				))
				._else(s)
			}
		])

		if (transitionStep !== null)
			cycle.steps.add(transitionStep)
		else if (localSteps !== null)
			cycle.steps.add(localSteps)

		return cycle
	}

	def localReactions(ExecutionNode it) {
		reactions.filter[r|! r.transition].toList
	}
}
