/**
 * Copyright (c) 2020 itemis AG - All rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * 
 */
package com.yakindu.sct.model.sexec.transformation

import com.google.inject.Inject
import com.yakindu.base.expressions.expressions.ArgumentExpression
import com.yakindu.base.expressions.expressions.ElementReferenceExpression
import com.yakindu.base.expressions.expressions.EventRaisingExpression
import com.yakindu.base.expressions.expressions.FeatureCall
import com.yakindu.base.expressions.util.ExpressionExtensions
import com.yakindu.base.types.Declaration
import com.yakindu.base.types.Direction
import com.yakindu.base.types.Event
import com.yakindu.base.types.Operation
import com.yakindu.base.types.Property
import com.yakindu.base.types.TypedDeclaration
import com.yakindu.base.types.adapter.OriginTracing
import com.yakindu.sct.model.sexec.ExecutionFlow
import com.yakindu.sct.model.sexec.ExecutionRegion
import com.yakindu.sct.model.sexec.ExecutionScope
import com.yakindu.sct.model.sexec.ExecutionState
import com.yakindu.sct.model.sexec.TimeEvent
import com.yakindu.sct.model.sexec.extensions.SexecBuilder
import com.yakindu.sct.model.sexec.extensions.ShadowEventExtensions
import com.yakindu.sct.model.sgraph.FinalState
import com.yakindu.sct.model.sgraph.Region
import com.yakindu.sct.model.sgraph.RegularState
import com.yakindu.sct.model.sgraph.SGraphFactory
import com.yakindu.sct.model.sgraph.Scope
import com.yakindu.sct.model.sgraph.State
import com.yakindu.sct.model.sgraph.Statechart
import com.yakindu.sct.model.sgraph.Vertex
import com.yakindu.sct.model.sgraph.util.StatechartUtil
import com.yakindu.sct.model.stext.stext.EventDefinition
import com.yakindu.sct.model.stext.stext.ImportScope
import com.yakindu.sct.model.stext.stext.OperationDefinition
import com.yakindu.sct.model.stext.stext.VariableDefinition
import java.util.ArrayList
import java.util.Set
import org.eclipse.emf.ecore.EObject
import org.eclipse.emf.ecore.util.EcoreUtil
import org.eclipse.xtext.EcoreUtil2
import org.eclipse.xtext.naming.IQualifiedNameProvider
import com.yakindu.base.expressions.expressions.DeclarationExpression
import com.yakindu.sct.model.sexec.concepts.SubMachine

class StructureMapping {
	 
	@Inject extension SexecElementMapping mapping
	@Inject extension StatechartExtensions sct
	@Inject extension IQualifiedNameProvider
	@Inject extension StatechartUtil
	@Inject extension ExpressionExtensions
	@Inject extension ShadowEventExtensions
	@Inject extension SexecBuilder
	@Inject extension OriginTracing
	@Inject extension SubMachine
	
	
	//==========================================================================
	// DECLARATION SCOPE MAPPING
	//
	
	/**
	 * maps all required scope defined in the statechart to the execution flow.
	 * This includes creating the scopes and adding all relevant declarations. Empty scopes wont be mapped.
	 */
	def ExecutionFlow mapScopes(Statechart sc, ExecutionFlow flow) {
		val usedDeclarations = sc.importedDeclarations
		flow.scopes.addAll(sc.scopes.map(scope | scope.mapScope(usedDeclarations)))
		flow
	}
	
	/**
	 * @return A set of used declaration that are imported from an external resource
	 */
	def protected importedDeclarations(Statechart it) {
		val allDeclarations = it.eAllContents
			.filter(ElementReferenceExpression)
			.map[reference]
			.filter(Declaration)
			.filter[!(eContainer instanceof DeclarationExpression)]
			.toSet
		return if(it.eResource !== null)
			allDeclarations.filter[decl|!decl.eResource.URI.equals(it.eResource.URI)].toSet
		else allDeclarations
	}
	
	
	/**
	 *  Interface and internal scopes have declarations
	 */
	def dispatch Scope mapScope(Scope scope, Set<Declaration> usedDeclarationss) {
		val _scope = scope.createScope
		_scope.declarations.addAll(scope.declarations.map(decl | decl.map).filterNull)
		return _scope
	}
	
	/**
	 * Import scope has imports which needs to be resolved to get all imported variable and operation definitions
	 */
	def dispatch Scope mapScope(ImportScope scope, Set<Declaration> usedDeclarations) {
		val _scope = scope.createScope
		usedDeclarations.forEach[createImportDeclaration(_scope)]
		return _scope
	}
	
	protected dispatch def createImportDeclaration(Property decl, Scope scope) {
		decl.doCreateAndAddImportDecl(scope)
	}

	protected dispatch def createImportDeclaration(Operation decl, Scope scope) {
		decl.doCreateAndAddImportDecl(scope)
	}
	
	protected def doCreateAndAddImportDecl(Declaration decl, Scope scope){
		scope.members += SGraphFactory.eINSTANCE.createImportDeclaration => [
			name = decl.name
			declaration = decl
		]
	}
	
	protected dispatch def createImportDeclaration(EObject decl, Scope scope) {
		// Nothing to do
	}

	def dispatch Declaration map(Declaration decl) {
	}
	
	def dispatch Declaration map(EventDefinition e) {
		e.create
	}
	
	def dispatch Declaration map(VariableDefinition v) {
		v.create
	}
	def dispatch Declaration map(OperationDefinition v) {
		v.create
	}
	 
	
	//==========================================================================
	// REGULAR STATE MAPPING
	//

		
	def ExecutionFlow mapRegularStates(Statechart statechart, ExecutionFlow r){
		val allStates = statechart.allRegularStates
		r.states.addAll(allStates.map( s | s.mapState));
		return r
	}

	def dispatch ExecutionState mapState(FinalState state) {
		val _state = state.create
		_state.leaf = true
		_state.entryAction = null
		_state.exitAction = null
		return _state		
	}
	
	def dispatch ExecutionState mapState(State state) {
		val _state = state.create
		_state.leaf = state.simple && !state.embedsSubMachine
		return _state
	}
	 
	def dispatch ExecutionState mapState(RegularState state) {}

	def ExecutionFlow mapEmbeddedSubMachinesToStates(Statechart statechart, ExecutionFlow flow) {
		val allSubMachineEmbeddingStates = statechart.allRegularStates.filter(State).filter[ embedsSubMachine ]
		val embeddedSubMachineStates = 
			allSubMachineEmbeddingStates
				.map[ s |
					s.embeddedSubMachines
						.map[
							val state = EcoreUtil2.getContainerOfType(it, State)
							val execState = it.create
							execState.leaf = true
							execState.superScope = state.create
							return execState
						]
				]
				.flatten
				.toList
		flow.states.addAll(embeddedSubMachineStates)
		return flow
	}
	
	//==========================================================================
	// REGION MAPPING
	//
	
	def ExecutionFlow mapRegions(Statechart statechart, ExecutionFlow flow){
		val allRegions = statechart.allRegions
		flow.regions.addAll( allRegions.map( r | r.mapRegion));
		return flow	
	}
	
	
	def ExecutionRegion mapRegion(Region region) {
		val _region = region.create
		
		if ( region.composite instanceof Statechart ) _region.superScope = (region.composite as Statechart).create
		else _region.superScope = (region.composite as State).create

		_region.subScopes.addAll( region.vertices.filter( typeof(RegularState) ).map( v | v.create as ExecutionScope ) )
		_region.nodes.addAll( region.vertices.filter( typeof(Vertex) ).map( v | v.mapped  ) )

		return _region
	}
	
	
	//==========================================================================
	// PSEUDO STATE MAPPING
	//

	def ExecutionFlow mapPseudoStates(Statechart statechart, ExecutionFlow r){
		r.nodes.addAll( statechart.allChoices.map( choice | choice.create ) );
		r.nodes.addAll( statechart.allEntries.map( entry | entry.create ) );
		r.nodes.addAll( statechart.allExits.map( exit | exit.create ) );
		r.nodes.addAll( statechart.allSynchronizations.map( sync | sync.create ) );
		return r
	}

	
	
	//==========================================================================
	// TIME EVENT MAPPING
	//
	
	/** Time trigger will be mapped to execution model time events for each real state. */
	def ExecutionFlow mapTimeEvents(Statechart statechart, ExecutionFlow r) {
		var content = EcoreUtil2::eAllContentsAsList(statechart)
		val allStates = content.filter(typeof(State))
		allStates.forEach( s | s.mapTimeEventSpecs)
		statechart.mapTimeEventSpecs
		return r
	}
	
	
	
	def mapTimeEventSpecs(State state) {
		
		val timeEventSpecs = state.timeEventSpecs
		
		val result = new ArrayList<TimeEvent>();
		for (tes : timeEventSpecs ) {
			val timeEvent = tes.createDerivedEvent
			timeEvent.origin = tes
			timeEvent.name = state.fullyQualifiedName + "_time_event_" + timeEventSpecs.indexOf(tes);
			state.statechart.create.timeEventScope.declarations.add(timeEvent);
			result.add(timeEvent);
			
		}	
		
		result
	}
	
	
	def mapTimeEventSpecs(Statechart statechart) {
		
		val timeEventSpecs = statechart.timeEventSpecs
		
		val result = new ArrayList<TimeEvent>();
		for (tes : timeEventSpecs ) {
			val timeEvent = tes.createDerivedEvent
			timeEvent.origin = tes
			timeEvent.name = statechart.name + "_time_event_" + timeEventSpecs.indexOf(tes);
			statechart.create.timeEventScope.declarations.add(timeEvent);
			result.add(timeEvent);
			
		}	
		
		result
	}
	
	//==========================================================================
	// LOCAL OUT EVENT MAPPING
	//
	
	/** creates local events for own out events that are used as triggers/guards */
	def mapLocalOutEvents(Statechart statechart, ExecutionFlow flow) {
		flow.outEventReferences.forEach[ref |
			val outEvent = ref.featureOrReference as Event
			val localOutEventName = outEvent.localOutEventName
			val localOutEvent = createLocalOutEvent(localOutEventName, outEvent, flow)

			// retarget feature call to new shadow event
			EcoreUtil.replace(ref, localOutEvent._ref)
		]
	}
	
	def getOutEventReferences(ExecutionFlow flow)  {
		val allContent = EcoreUtil2::eAllContentsAsList(flow)
		return allContent
			.filter(ArgumentExpression)
			.filter[!(eContainer instanceof EventRaisingExpression)]
			.filter[featureOrReference.isOutEvent]
			.filter[EcoreUtil2::getRootContainer(featureOrReference) === flow]
			.toList
	}
	
	def protected isOutEvent(EObject it) {
		it instanceof Event && (it as Event).direction == Direction.OUT
	}
	
	//==========================================================================
	// REFERENCED MACHINE OUT EVENT MAPPING
	//
	
	/** creates shadow variables for referenced machines' out events */
	def mapReferencedMachineOutEvents(Statechart statechart, ExecutionFlow flow) {
		flow.submachineOutEventCalls.forEach[fc |
			val submachineMember = fc.statechartRefs.head
			val outEvent = fc.feature as Event
			val shadowEventName = fc.shadowEventName
			val shadowEvent = createShadowEvent(shadowEventName, submachineMember, outEvent, flow)

			// retarget feature call to new shadow event
			EcoreUtil.replace(fc, shadowEvent._ref)
		]
	}
	
	def getSubmachineOutEventCalls(ExecutionFlow flow)  {
		val sct = flow.sourceElement as Statechart
		val allContent = EcoreUtil2::eAllContentsAsList(flow)
		return allContent.filter(FeatureCall).filter[isOutEventCall].filter[isCallOnStatechartMember(it, sct)].toList
	}
	
	def getSubmachineOutEventCalls(Statechart sct)  {
		val allContent = EcoreUtil2::eAllContentsAsList(sct)
		return allContent.filter(FeatureCall).filter[isOutEventCall].filter[isCallOnStatechartMember(it, sct)].toList	
	}
	
	protected def boolean isOutEventCall(FeatureCall it) {
		feature instanceof Event && (feature as Event).direction == Direction.OUT
	}
	
	/**
	 * Returns true if the feature call is on a statechart reference which is a direct member of the given statechart, e.g.
	 * <br><br>
	 * <code>submachine.outEvent</code> or 
	 * <br>
	 * <code>submachine.Iface.outEvent</code>.
	 * <br>
	 * Returns false otherwise, especially when the owner is not a direct member of the given statechart, e.g.
	 * <br><br>
	 * <code>submachine1.submachine2.outEvent</code>.
	 * 
	 */
	protected def isCallOnStatechartMember(FeatureCall it, Statechart statechart) {
		if (statechartRefs.size !== 1) return false
		
		var EObject statechartContainer = EcoreUtil2.getContainerOfType(statechartRefs.head, Statechart)
		if (statechartContainer === null) {
			statechartContainer = EcoreUtil2.getContainerOfType(statechartRefs.head, ExecutionFlow)?.sourceElement
		}
		return (statechartContainer !== null && statechart == statechartContainer)
	}
	
	protected def getStatechartRefs(FeatureCall it) {
		toCallStack.map[featureOrReference].filter(VariableDefinition).filter[isStatechartRef]
	}
	
	protected def dispatch isStatechartRef(EObject it) {
		false
	}
	
	protected def dispatch isStatechartRef(TypedDeclaration it) {
		type.isOriginStatechart
	}
	
	
}