/**
 * Copyright (c) 2019-2020 itemis AG - All rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 *
 * Contributors:
 * 	Rene Beckmann - itemis AG
 * 	Mathias Alexander Schulte- itemis AG
 * 	Axel Terfloth - itemis AG
 *
 */
package com.yakindu.yet.sct.generator.c

import com.google.inject.Inject
import com.yakindu.base.types.ComplexType
import com.yakindu.base.types.Direction
import com.yakindu.base.types.EnumerationType
import com.yakindu.base.types.Type
import com.yakindu.base.types.TypeAlias
import com.yakindu.base.types.typesystem.ITypeSystem
import com.yakindu.sct.generator.c.extensions.FileNaming
import com.yakindu.sct.generator.c.extensions.GenmodelEntries
import com.yakindu.sct.generator.c.extensions.Naming
import com.yakindu.sct.generator.c.types.CTypes
import com.yakindu.sct.generator.core.artifacts.IContentTemplate
import com.yakindu.sct.generator.core.artifacts.IGenArtifactConfigurations
import com.yakindu.sct.generator.core.types.ICodegenTypeSystemAccess
import com.yakindu.sct.model.sexec.ExecutionFlow
import com.yakindu.sct.model.sexec.extensions.SExecExtensions
import com.yakindu.sct.model.sgen.GeneratorEntry
import com.yakindu.sct.model.stext.stext.EventDefinition
import com.yakindu.sct.model.stext.stext.InterfaceScope
import com.yakindu.sct.model.stext.stext.InternalScope
import com.yakindu.sct.model.stext.stext.VariableDefinition

class StatechartTracerSource implements IContentTemplate<ExecutionFlow> {
	@Inject extension GenmodelEntries
	@Inject extension Naming
	@Inject protected extension FileNaming
	@Inject extension CTypes
	@Inject extension ITypeSystem
	@Inject extension YETNaming yetNaming
	@Inject extension ICodegenTypeSystemAccess
	@Inject extension CYETExtensions
	@Inject extension SExecExtensions
	
	
	override content(ExecutionFlow flow, GeneratorEntry entry, IGenArtifactConfigurations locations) {
		'''
			«entry.licenseText»
			/* Implements a standard statechart trace handler which adapts to the
			 * YET tracing library.
			 */
			
			
			«includes(flow, entry, locations)»
			
			
			«declarations(flow, entry, locations)»
			
			
			«implementations(flow, entry, locations)»
			
		'''
	}
	
	def protected includes(ExecutionFlow flow, GeneratorEntry entry, extension IGenArtifactConfigurations locations){
		'''
			#include "«(flow.tracerModule.h).relativeTo(flow.tracerModule.c)»"
			#include "«(flow.metaModule.h).relativeTo(flow.tracerModule.c)»"
			
			#include <string.h>
			#include <stdlib.h>
		'''
	}
	
	def protected declarations(ExecutionFlow flow, GeneratorEntry entry, IGenArtifactConfigurations locations){
		'''
			#ifndef SC_UNUSED
			#define SC_UNUSED(P) (void)(P)
			#endif
			
			static «sc_string.name» default_scope_name = "«flow.name»";
			
			static yet_value_serializer feature_value_serializer[«flow.tracedTypedElements.length + 1»];
			
		'''
	}
	
	def CharSequence serializeFunction(Type it, Type startType) {
		if(it === null) {
			return '''yet_serialize_void'''
		} else if (isIntegerType) {
			return '''yet_int_serializer_function(sizeof(«startType.targetLanguageName»), «!(isSuperType(getType(ITypeSystem.UNSIGNED_INTEGER)))»)'''
		} else if (haveCommonType(getType(ITypeSystem.REAL))) {
			return '''yet_real_serializer_function(sizeof(«startType.targetLanguageName»))'''
		} else if (haveCommonType(getType(ITypeSystem.BOOLEAN))) {
			return '''yet_serialize_bool'''
		} else if (haveCommonType(getType(ITypeSystem.STRING))) {
			return '''yet_serialize_string'''
		} else if (isEnumerationType) {
			return '''yet_int_serializer_function(sizeof(«startType.targetLanguageName»), true)'''
		} else if (it instanceof TypeAlias) {
			return type.serializeFunction(startType)
		}
		return '''yet_serialize_void'''
	}
	
	def protected implementations(ExecutionFlow flow, GeneratorEntry entry, IGenArtifactConfigurations locations){
		val tracedTypedElements = flow.tracedTypedElements
		'''
			static void init_feature_value_serializer(void) {
				feature_value_serializer[0] = yet_serialize_void;
				«FOR typedElement : tracedTypedElements»
					feature_value_serializer[«tracedTypedElements.indexOf(typedElement) + 1»] = «typedElement.type.serializeFunction(typedElement.type)»;
				«ENDFOR»
			}
			
			static char* featureName(«sc_integer.name» featureId)
			{
				return «flow.metaFeaturesProperty»[featureId];
			}
			
			static char* featureValue(«sc_integer.name» featureId, const void* valuePtr, char* buf)
			{
				return (feature_value_serializer[featureId])(valuePtr, buf);
			}
			
			static char* stateName(«sc_integer.name» stateId)
			{
				return «flow.metaStatesProperty»[stateId];
			}
			
			static yet_error dispatchMessage(yet_scope *scope, yet_message * msg, char *fqn);
			
			void «functionPrefixStatechart(flow)»_init_sc_tracer(yet_sc_tracer *tracer, «flow.statemachine»* machine, sc_string name)
			{
				if (tracer != sc_null) {
					yet_init_sc_tracer(tracer, machine, &dispatchMessage);
			
					tracer->scope.name = default_scope_name;
					if (name != sc_null) {
						tracer->scope.name = name;
					}
					tracer->traceinfoProvider.featureName = featureName;
					tracer->traceinfoProvider.featureValue = featureValue;
					tracer->traceinfoProvider.stateName = stateName;
					
					machine->trace_handler = &(tracer->traceEventHandler);
				}
				init_feature_value_serializer();
			}
			
			
			/* Implementation of yet_handler callback function. Handles incoming stimuli and call appropriate state machine function. */
			static yet_error dispatchMessage(yet_scope *scope, yet_message * msg, char *fqn)
			{
				«IF flow.allInEvents.isNullOrEmpty && !flow.hasVariables»
				SC_UNUSED(scope);
				SC_UNUSED(msg);
				SC_UNUSED(fqn);
				return 0;
				«ELSE»
				yet_sc_tracer* tracer = scope->instance;
				char* member;
				member = fqn;
				
				«IF !flow.hasVariables»
				SC_UNUSED(msg);
				«ENDIF»
				
				«FOR inEvent : flow.allInEvents»
					if(strcmp(member, "«inEvent.metaName»") == 0) {
						/* Skip one message; this should not be re-raised on host */
						/* TODO: this is not optimal check if we can get rid of it */
						tracer->skip_raised_in_event++; 
						«inEvent.asRaiser»(tracer->machine«IF inEvent.hasValue», «inEvent.type.convertType(inEvent.type)»«ENDIF»);
						return 0;
					}
				«ENDFOR»
				«FOR v : flow.allExternalVariables»
					if(strcmp(member, "«v.metaName»") == 0) {
						«v.type.targetLanguageName» converted = «v.type.convertType(v.type)»;
						«v.asSetter»(tracer->machine, converted);
						return 0;
					}
				«ENDFOR»
				«FOR v : flow.allInternalVariables»
					if(strcmp(member, "«v.metaName»") == 0) {
						«v.type.targetLanguageName» converted = «v.type.convertType(v.type)»;
						((«flow.type»*)tracer->machine)->«(v.eContainer as InternalScope).instance».«v.name.asEscapedIdentifier» = converted;
						return 0;
					}
				«ENDFOR»
				
				
				return YET_ERR_INVALID_KEY;
				«ENDIF»
			}
		'''
	}
	

	def CharSequence convertType(Type it, Type baseType) {
		if (isIntegerType || isEnumerationType) {
			return '''(«baseType.targetLanguageName»)atol(msg->value)'''
		} else if (haveCommonType(getType(ITypeSystem.BOOLEAN))) {
			return '''(msg->value[0] == 't')'''
		} else if (haveCommonType(getType(ITypeSystem.REAL))) {
			'''(«baseType.targetLanguageName»)atof(msg->value)'''
		} else if (haveCommonType(getType(ITypeSystem.STRING))) {
			'''msg->value'''
		} else if (it instanceof TypeAlias) {
			'''«type.convertType(baseType)»'''
		} else {
			'''sc_null'''
		}
	}

	def isIntegerType(Type it) {
		return haveCommonType(getType(ITypeSystem.INTEGER)) && isSuperType(getType(ITypeSystem.INTEGER))
	}
	
	def isEnumerationType(Type it) {
		originType instanceof EnumerationType
	}

	protected def boolean hasVariables(ExecutionFlow flow) {
		!flow.allExternalVariables.isNullOrEmpty || !flow.allInternalVariables.isNullOrEmpty

	}
	
	def protected getAllInEvents(ExecutionFlow it) {
		return scopes.filter(InterfaceScope).map[declarations].flatten.filter(EventDefinition).filter[direction == Direction::IN]
	}
	
	def protected getAllExternalVariables(ExecutionFlow it) {
		return scopes.filter(InterfaceScope).map[declarations].flatten.filter(VariableDefinition)
		.filter[!const]
		.filter[!readonly]
		.filter[!type.isComplexType]
	}
	
	def protected getAllInternalVariables(ExecutionFlow it) {
		return scopes.filter(InternalScope).map[declarations].flatten
		.filter(VariableDefinition).filter[!const]
		.filter[!readonly]
		.filter[!type.isComplexType]
	}
	
	def protected boolean isComplexType(Type it) {
		if(it instanceof EnumerationType) {
			return false
		}else if (it instanceof ComplexType){
			return true
		} else if (it instanceof TypeAlias) {
			return isComplexType(type)
		}
		return false
	}
	
	def protected removePrefix(String s){
		return s.substring(s.indexOf(".")+1)
	}
	
}
