/**
 * Copyright (c) 2025 itemis AG - All rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * 
 */
package com.yakindu.sct.model.stext.concepts

import com.google.inject.Inject
import com.google.inject.Injector
import com.google.inject.Singleton
import com.yakindu.base.expressions.parser.antlr.ExpressionsParser
import com.yakindu.base.types.ComplexType
import com.yakindu.base.types.Operation
import com.yakindu.base.types.Package
import com.yakindu.base.types.TypeBuilder
import com.yakindu.base.types.annotations.TypeAnnotations
import com.yakindu.base.types.libraries.ITypeLibrary
import com.yakindu.base.types.libraries.ITypeLibraryProvider
import java.io.StringReader
import org.eclipse.emf.common.notify.Adapter
import org.eclipse.emf.common.util.URI
import org.eclipse.emf.ecore.EObject
import org.eclipse.emf.ecore.resource.impl.ResourceImpl

/**
 * Type Library for the types responsible to realize the logging concept for statecharts.
 * This built-up type library should be invoked and used for logging implementations.
 * 
 * @author Axel Terfloth - Initial contribution
 */
@Singleton
class LoggerTypeLibrary implements ITypeLibrary{
	
	
	public static val LOGGER_LIBRARY = "create.logger"
	public static val LOGGER_LIBRARY_URI = URI.createURI(LOGGER_LIBRARY)
	
	public static val SC_PACKAGE_NAME = "sc"
	public static val LOGGER_PACKAGE_NAME = "logging"
	
	public static val LOGGER_NAME = "Logger"
	public static val LOGGER_WRITE_NAME = "write"
	
	var Package typesPackage;
	
	@Inject extension TypeBuilder
	@Inject extension TypeAnnotations
	@Inject(optional=true) ExpressionsParser parser;
	
	val resource = new ResourceImpl(LOGGER_LIBRARY_URI) {
		override protected doUnload() {
			// Do no unload in memory resource
		}
	}

	def protected create() {
		typesPackage = createTypesPackage
		resource.URI = LOGGER_LIBRARY_URI
		resource.contents += typesPackage
	}

	def getTypesPackage() {
		if (typesPackage === null) {
			create();
		}
		return typesPackage
	}

	def getResource() {
		getTypesPackage
		resource
	}
	
	protected def createTypesPackage() {
		SC_PACKAGE_NAME._package  => [ createPackage |
			createPackage.member += LOGGER_PACKAGE_NAME._package  => [ package |
				package.member += createLoggerService
			]
		]
		
	}
	
	protected def createLoggerService() {
		_complexType => [
			name = LOGGER_NAME
			annotateAsReferenceType
			features += _op(LOGGER_WRITE_NAME) => [
				_param("msg", _string)
				_param("args", _any)
				parameters.get(1).varArgs = true
				
				doc(
					'''
						Writes a log entry . 
						The log entry consists of an message which optionally can include format specifiers for additionally provided arguments.
					'''
				)	
			]
		]
	}
	
	def getLoggerPackage(){
		getTypesPackage().member.findFirst[name == LOGGER_PACKAGE_NAME] as Package
	}
	
	def getLoggerType() {
		getLoggerPackage().member.findFirst[name == LOGGER_NAME] as ComplexType
	}
	
	def Operation loggerWrite() {
		getLoggerType.features.filter(Operation).filter( m | m.name == LOGGER_WRITE_NAME).head
	}
	
	protected def doc(EObject it, String documentation) {
		if (parser !== null) {
			val doc = parser.parse(new StringReader(documentation));
			eAdapters += doc.rootNode as Adapter
		}
	}
	
	static class Provider implements ITypeLibraryProvider {
		
		@Inject Injector injector
		
		override provide() {
			
			val edfLibrary = injector.getInstance(LoggerTypeLibrary)
			#[ edfLibrary ]
		}
		
	}
	
	override getLibraryPackage() {
		return getTypesPackage
	}
	
	override getLibraryURI() {
		return LOGGER_LIBRARY_URI
	}
	
}