/**
 * Copyright (c) 2020 itemis AG - All rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * 
 */
package com.yakindu.sct.model.stext.validation

import static com.yakindu.sct.model.stext.concepts.StatechartAnnotations.*

import com.google.inject.Inject
import com.yakindu.base.types.Annotation
import com.yakindu.sct.model.stext.concepts.StatechartAnnotations
import com.yakindu.sct.model.stext.extensions.STextExtensions
import com.yakindu.sct.model.stext.stext.StatechartSpecification
import org.eclipse.xtext.validation.Check
import org.eclipse.xtext.validation.CheckType

class SpecificationValidator extends STextBaseValidator {

	@Inject extension StatechartAnnotations
	@Inject extension STextExtensions

	public static final String CONTRADICTORY_ANNOTATIONS_MSG = "Some annotations (%s) have contradictory effects.";
	public static final String CONTRADICTORY_ANNOTATIONS_CODE = "ContradictoryAnnotations";

	@Check(CheckType.FAST)
	def void checkAnnotations(StatechartSpecification statechart) {
		var Annotation eventDriven = statechart.getAnnotationOfType(EVENT_DRIVEN_ANNOTATION)
		var Annotation cycleBased = statechart.getAnnotationOfType(CYCLE_BASED_ANNOTATION)
		if (eventDriven !== null && cycleBased !== null) {
			var String errorMsg = String.format(CONTRADICTORY_ANNOTATIONS_MSG,
				String.join(", ", eventDriven.getType().toString(), cycleBased.getType().toString()))
			error(errorMsg, cycleBased, null, -1, CONTRADICTORY_ANNOTATIONS_CODE)
		}
		var Annotation parentFirst = statechart.getAnnotationOfType(PARENT_FIRST_ANNOTATION)
		var Annotation childFirst = statechart.getAnnotationOfType(CHILD_FIRST_ANNOTATION)
		if (parentFirst !== null && childFirst !== null) {
			var String errorMsg = String.format(CONTRADICTORY_ANNOTATIONS_MSG,
				String.join(", ", parentFirst.getType().toString(), childFirst.getType().toString()))
			error(errorMsg, parentFirst, null, -1, CONTRADICTORY_ANNOTATIONS_CODE)
		}
	}

	public static final String INTERNAL_EVENT_QUEUE_CANT_BE_DEACTIVATED_MSG = "The internal event queue can't be deactivated for @EventDriven statecharts.";
	public static final String INTERNAL_EVENT_QUEUE_CANT_BE_DEACTIVATED_CODE = "InvalidQueueDeactivation";

	@Check(CheckType.FAST)
	def void checkLocalEventQueueForEventDriven(StatechartSpecification specification) {
		val statechart = specification.statechart
		if (statechart !== null && statechart.eventDriven && !statechart.isInternalEventBuffer) {
			error(INTERNAL_EVENT_QUEUE_CANT_BE_DEACTIVATED_MSG,
				specification.getAnnotationOfType(EVENTBUFFERING_ANNOTATION), null, -1,
				INTERNAL_EVENT_QUEUE_CANT_BE_DEACTIVATED_CODE)
		}
	}
}
