/**
 * Copyright (c) 2022 itemis AG - All rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * 
 * Contributors:
 * 	Jonathan Thoene - itemis AG
 * 
 */
package com.yakindu.sctunit.simulation.core.interpreter

import com.google.inject.Inject
import com.yakindu.base.expressions.interpreter.IExpressionInterpreter
import com.yakindu.base.types.Expression
import com.yakindu.base.types.Parameter
import com.yakindu.base.types.inferrer.ITypeSystemInferrer
import com.yakindu.sct.model.sruntime.CompositeSlot
import com.yakindu.sct.model.sruntime.ExecutionContext
import com.yakindu.sct.model.sruntime.ExecutionSlot
import com.yakindu.sct.model.sruntime.SRuntimeFactory
import com.yakindu.sct.model.stext.stext.VariableDefinition
import com.yakindu.sct.simulation.core.sexec.container.SexecExecutionContextInitializer
import com.yakindu.sctunit.sCTUnit.SCTUnitClass
import com.yakindu.sctunit.sCTUnit.SCTUnitOperation
import com.yakindu.sctunit.sCTUnit.VariableDefinitionStatement
import java.util.List
import org.eclipse.xtext.naming.IQualifiedNameProvider
import org.eclipse.xtext.naming.QualifiedName

import static org.eclipse.xtext.EcoreUtil2.*

class DefaultSCTUnitExecutionContextInitializer implements ISCTUnitExecutionContextInitializer {

	@Inject protected extension ITypeSystemInferrer
	@Inject protected extension IQualifiedNameProvider
	@Inject protected extension SexecExecutionContextInitializer
	@Inject IExpressionInterpreter statementInterpreter

	protected val factory = SRuntimeFactory.eINSTANCE

	override initialize(ExecutionContext context, SCTUnitClass unitClass) {
		var slot = factory.createCompositeSlot => [
			name = unitClass.name
			fqName = unitClass.fullyQualifiedName.toString
		]
		initializeGlobalVariables(context, unitClass.variableDefinitions, slot)
		context.slots += slot
	}

	override dispose(ExecutionContext context, SCTUnitOperation unitOperation) {
		val unitClass = getContainerOfType(unitOperation, typeof(SCTUnitClass))
		val classSlot = context.slots.find(unitClass.fullyQualifiedName) as CompositeSlot
		val operationSlot = classSlot.slots.find(unitOperation.fullyQualifiedName)
		val containerSlots = (operationSlot.eContainer as CompositeSlot).slots
		containerSlots.remove(operationSlot)
	}
	
	def protected dispose(CompositeSlot context, SCTUnitOperation unitOperation) {
		val operationSlot = context.slots.find(unitOperation.fullyQualifiedName)
		val containerSlots = (operationSlot.eContainer as CompositeSlot).slots
		containerSlots.remove(operationSlot)
	}

	override initialize(ExecutionContext context, SCTUnitOperation unitOperation, List<Expression> args) {
		val unitClass = getContainerOfType(unitOperation, typeof(SCTUnitClass))
		var operationSlot = initializeOperationSlot(unitOperation, args, context)
		if (context.slots.find(unitClass.fullyQualifiedName) === null)
			initialize(context, unitClass)
			
		(context.slots.find(unitClass.fullyQualifiedName) as CompositeSlot).slots += operationSlot
	}
	
	override initialize(ExecutionContext context, SCTUnitOperation unitOperation, List<Expression> args, SCTUnitOperation owner) {
		var operationSlot = initializeOperationSlot(unitOperation, args, context)
		val ownerSlot = context.slots.find(owner.fullyQualifiedName) as CompositeSlot
		if (ownerSlot.slots.find(unitOperation.fullyQualifiedName) !== null) {
			dispose(ownerSlot, unitOperation)
		}
		(context.slots.find(owner.fullyQualifiedName) as CompositeSlot).slots += operationSlot		
	}
	protected def CompositeSlot initializeOperationSlot(SCTUnitOperation unitOperation, List<Expression> args, ExecutionContext context) {
		var operationSlot = factory.createCompositeSlot => [
			name = unitOperation.name
			type = unitOperation.infer.type
			fqName = unitOperation.fullyQualifiedName.toString
		]
		if (args.size == unitOperation.parameters.size) {
			context.initializeOperationParams(unitOperation.parameters, operationSlot, args)
		}
		return operationSlot
	}

	override initialize(ExecutionContext context, VariableDefinition definition, SCTUnitOperation unitOperation) {
		val unitClass = getContainerOfType(unitOperation, typeof(SCTUnitClass))
		val classSlot = context.slots.find(unitClass.fullyQualifiedName) as CompositeSlot
		val operationSlot = classSlot.slots.find(unitOperation.fullyQualifiedName) as CompositeSlot
		var ExecutionSlot variableSlot = operationSlot.slots.find(definition.fullyQualifiedName)
		
		if (variableSlot === null) {
			variableSlot = definition.createExecutionSlotFor
			operationSlot.slots += variableSlot
		}
		if (definition.initialValue !== null) {
			variableSlot.value = statementInterpreter.evaluate(definition.initialValue, context)
		}
	}

	def protected void initializeOperationParams(ExecutionContext context, List<Parameter> params,
		CompositeSlot operationSlot, List<Expression> args) {
		for (var int i = 0; i < params.size; i++) {
			val index = i;
			operationSlot.slots += factory.createExecutionVariable => [
				name = params.get(index).name
				type = params.get(index).infer.type
				value = statementInterpreter.evaluate(args.get(index), context)
				fqName = params.get(index).getFullyQualifiedName.toString
			]
		}
	}

	def protected void initializeGlobalVariables(ExecutionContext context, List<VariableDefinitionStatement> variables,
		CompositeSlot classSlot) {
		for (var int i = 0; i < variables.size; i++) {
			val index = i;
			classSlot.slots += factory.createExecutionVariable => [
				name = variables.get(index).definition.name
				type = variables.get(index).definition.infer.type
				value = statementInterpreter.evaluate(variables.get(index).definition.initialValue, context)
				fqName = variables.get(index).definition.getFullyQualifiedName.toString
			]
		}
	}
	
	def protected ExecutionSlot find(List<ExecutionSlot> slots, QualifiedName fullyQualifiedName) {
		val found = slots.findFirst[it.fqName == fullyQualifiedName.toString]
		if (found === null) {
			for (slot : slots) {
				if (slot instanceof CompositeSlot) {
					val resultSlot = find(slot.slots, fullyQualifiedName)
					if (resultSlot !== null) {
						return resultSlot
					}
				}
			}
		}
		return found
	}
}
