/*!
 * YAKINDU Player Interface
 * Copyright 2021 itemis AG
 * Licensed under ...
 * 
 * 
 * Client library for interfacing YAKINDU statechart players. 
 */

/** */
class YpiPlayer {
 
    constructor (id, frame) {
        this.id = id;
        this.frame = frame;
    }

    enter() {
        ypi.frame.enter(this.frame);
    }

    exit() {
        ypi.frame.exit(this.frame);
    }

    raise(event) {
        ypi.frame.raise(this.frame, event);
    }

    set(variable, value) {
        ypi.frame.set(this.frame, variable, value);
    }

}



/* All functions will be accessible under the 'ypi' namespace. */
var ypi = {

    frame : {
        /** 
         * To distinguish between different player instances it is possible to assign a player id.
         * This id will be sent with each player event.
         */
         setPlayerId : function (frame, playerId) {
            frame.contentWindow.postMessage({
                cmd: 'setPlayerId',
                playerId: playerId,
            }, '*');
        },

        /**
         * Enters (activates) the player state machine.
         * 
         * @param {Element} frame iframe element of the plyer instance
         */
        enter: function (frame) {
            frame.contentWindow.postMessage({
                cmd: 'enter',
            }, '*');
        },

        /**
         * Exits (deactivates) the player state machine.
         * @param {Element} frame iframe element of the plyer instance
         */
        exit: function (frame) {
            frame.contentWindow.postMessage({
                cmd: 'exit',
            }, '*');
        }, 

        /**
         * Raises an event using its fully qualified name.
         * @param {Element} frame 
         * @param {string} event 
         */
        raise: function(frame, event) {
            frame.contentWindow.postMessage({
                cmd: 'raiseEvent',
                eventId: event
            }, '*');
        },

        /**
         * Sets a statechart variable using its fully qualified name.
         * @param {Element} frame 
         * @param {string} variable 
         * @param {*} value 
         */
        set: function(frame, variable, value) {
            frame.contentWindow.postMessage({
                cmd: 'changeValue',
                varName: variable,
                value: value
            }, '*');
        },


    }, 

    bindings : {

        playerById: new Map(),
        statesById: new Map(),
        valuesById: new Map(),


        scan : function() {
            var playerFrames = document.querySelectorAll('iframe.y-player');
            playerFrames.forEach( p => {
                var pId = p.getAttribute("id");
                ypi.frame.setPlayerId(p, pId);
                var player = new YpiPlayer(pId, p);
                ypi.bindings.playerById.set(pId, player);
            })        
        },

        bind : function() {
            var playerContexts = document.querySelectorAll('div [y-player]');
            playerContexts.forEach( c => {
                var playerName = c.getAttribute('y-player');
                var player = ypi.bindings.playerById.get(playerName);
        
                var enterActions = c.querySelectorAll('.y-do-enter');
                enterActions.forEach( ea => {
                    ea.onclick = function(event) {
                        if (player !== undefined)
                            player.enter();
                    };
                    ea.onmouseenter = (e) => { ea.classList.add('y-do--over');};
                    ea.onmouseleave = (e) => { ea.classList.remove('y-do--over');};
                });
        
                var exitActions = c.querySelectorAll('.y-do-exit');
                exitActions.forEach( ea => {
                    ea.onclick = function(event) {
                        if (player !== undefined)
                            player.exit();
                    };
                    ea.onmouseenter = (e) => { ea.classList.add('y-do--over');};
                    ea.onmouseleave = (e) => { ea.classList.remove('y-do--over');};
                });
        
               var raiseActions = c.querySelectorAll('.y-do-raise');
               raiseActions.forEach( ra => {
                   ra.onclick = function(event) {
                        if (player !== undefined)
                            player.raise(ra.getAttribute('id'));
                   };
                   ra.onmouseenter = (e) => { ra.classList.add('y-do--over');};
                   ra.onmouseleave = (e) => { ra.classList.remove('y-do--over');};
                });
        
                var stateElements = c.querySelectorAll('.y-state');
                stateElements.forEach( se => {
                    var stateId = se.getAttribute("id");
                    var fullStateId = playerName + "." + stateId;
                    var states = ypi.bindings.statesById.get(fullStateId);
                    if (states === undefined) {
                        states = [];
                        ypi.bindings.statesById.set(fullStateId, states);
                    }
                    states.push(se);
                 });
         
                 var valueElements = c.querySelectorAll('.y-variable, .y-value-observer');
                 valueElements.forEach( ve => {
                     var varFqn = ve.getAttribute("id");
                     var varId = varFqn.replace('.', '_').toUpperCase();
                     var fullVarId = playerName + "." + varId;
                     var values = ypi.bindings.valuesById.get(fullVarId);
                     if (values === undefined) {
                        values = [];
                         ypi.bindings.valuesById.set(fullVarId, values);
                     }
                     values.push(ve);
        
                     if(ve.tagName == 'INPUT') {
                        ve.addEventListener("focusout", () => {
                            if (player !== undefined)
                               player.set(varFqn, ve.value);
                        });
                        ve.addEventListener("input", () => {
                            if (player !== undefined)
                                player.set(varFqn, ve.value);
                        });
                        ve.addEventListener("keyup", (e) => {
                            if (e.keyCode === 13) {
                                if (player !== undefined)
                                    player.set(varFqn, ve.value);
                            }
                        });
                     }
                  }); 
         
            });
        }
    } ,

    onMessage : (event) => {
        var data = event.data;
        console.log(data);
        switch (data.cmd) {
            case "stateEntered":
              
                console.log("State entered " + data.playerId + "." + data.name);
                var fullStateName = data.playerId + "." + data.name;
                var stateElements = ypi.bindings.statesById.get(fullStateName);
                if ( stateElements !== undefined ) {
                    stateElements.forEach(e => e.classList.add('y-state--active'));
                }  
                break;
            case "stateExited":
                console.log("State exited " + data.playerId + "." + data.name);
                var fullStateName = data.playerId + "." + data.name
                var stateElements = ypi.bindings.statesById.get(fullStateName);
                if ( stateElements !== undefined ) {
                    stateElements.forEach(e => e.classList.remove('y-state--active'));
                }  
                break;
            case "valueChanged":
                var value = data.value;
                console.log("Value changed  " + data.playerId + "." + data.name + " " + value)
                var fullVarId = data.playerId + "." + data.varId.replace('.', '_').toUpperCase();
                var valueElements = ypi.bindings.valuesById.get(fullVarId);
                if ( valueElements !== undefined ) {
                    valueElements.forEach(e => {
                        if (e.classList.contains("y-value-observer")) {
                            e.onvaluechange(value);
                        }
                        else if (e.tagName == "INPUT") {
                            e.value = value;
                        } else {
                            e.setAttribute("value", "=" + value);
                        }
                    });
                }  
                break;
        }
    }
};


window.addEventListener("message", ypi.onMessage);

window.addEventListener("load", (e) => {
 
    ypi.bindings.scan();
    ypi.bindings.bind();

});

