/**
 * Copyright (c) 2023 itemis AG - All rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * 
 * Contributors:
 * 	Axel Terfloth - itemis AG
 * 
 */
package com.itemis.create.sunit.generator.csharp.nunit.transformation

import com.google.inject.Inject
import com.itemis.create.base.generator.core.GeneratorAssignment
import com.itemis.create.base.generator.core.Transformation
import com.itemis.create.base.generator.core.codepattern.IVariableCode
import com.itemis.create.base.generator.core.concepts.TimedInterface
import com.itemis.create.base.generator.core.concepts.VirtualTimer
import com.itemis.create.base.generator.csharp.codemodel.CsharpClass
import com.itemis.create.base.generator.csharp.codemodel.CsharpCompilationUnit
import com.itemis.create.base.generator.csharp.codemodel.CsharpNamespace
import com.itemis.create.base.generator.csharp.codemodel.CsharpNaming
import com.itemis.create.base.generator.csharp.codemodel.CsharpStatemachineLibrary
import com.itemis.create.base.generator.csharp.codemodel.CsharpTypeBuilder
import com.itemis.create.base.generator.csharp.codepattern.ContainedCode
import com.itemis.create.base.generator.csharp.concepts.EventRaiser
import com.itemis.create.base.generator.csharp.concepts.OutEventSubscriber
import com.itemis.create.base.generator.csharp.transformation.RaiseEventModification
import com.itemis.create.base.generator.csharp.transformation.Slang2CsharpTransformation
import com.itemis.create.base.generator.csharp.transformation.VoidIsDefaultReturnTypeModification
import com.itemis.create.statechart.generator.csharp.codemodel.CsharpStatemachineOperationCallbacks
import com.itemis.create.sunit.SUnitAnnotations
import com.itemis.create.sunit.SUnitLib
import com.itemis.create.sunit.generator.csharp.nunit.codemodel.NUnitAnnotations
import com.itemis.create.sunit.generator.csharp.nunit.codemodel.NUnitClass
import com.itemis.create.sunit.generator.csharp.nunit.codemodel.NUnitLib
import com.itemis.create.sunit.generator.csharp.nunit.concepts.OutEventMembers
import com.yakindu.base.base.NamedElement
import com.yakindu.base.expressions.ExpressionBuilder
import com.yakindu.base.expressions.expressions.ArgumentExpression
import com.yakindu.base.expressions.expressions.AssignmentOperator
import com.yakindu.base.expressions.expressions.ElementReferenceExpression
import com.yakindu.base.expressions.expressions.ExpressionsFactory
import com.yakindu.base.expressions.expressions.FeatureCall
import com.yakindu.base.expressions.expressions.IntLiteral
import com.yakindu.base.expressions.expressions.PrimitiveValueExpression
import com.yakindu.base.expressions.expressions.TimeEventSpec
import com.yakindu.base.expressions.util.ExpressionExtensions
import com.yakindu.base.types.ComplexType
import com.yakindu.base.types.Declaration
import com.yakindu.base.types.Direction
import com.yakindu.base.types.EnumerationType
import com.yakindu.base.types.Expression
import com.yakindu.base.types.Operation
import com.yakindu.base.types.Package
import com.yakindu.base.types.Part
import com.yakindu.base.types.Property
import com.yakindu.base.types.Type
import com.yakindu.base.types.TypeBuilder
import com.yakindu.base.types.TypedDeclaration
import com.yakindu.base.types.TypesFactory
import com.yakindu.sct.generator.core.codemodel.StateEnum
import com.yakindu.sct.model.sexec.TimeEvent
import com.yakindu.sct.model.sgraph.Statechart
import com.yakindu.sct.model.sgraph.util.StatechartUtil
import com.yakindu.sct.model.stext.concepts.StatechartAnnotations
import com.yakindu.sct.model.stext.stext.EventDefinition
import com.yakindu.sct.model.stext.stext.VariableDefinition
import com.yakindu.sct.types.lib.StatechartLibrary
import com.yakindu.sct.types.resource.Statechart2TypeTransformation
import java.util.HashSet
import java.util.List
import java.util.Set
import org.eclipse.emf.ecore.EObject
import org.eclipse.emf.ecore.util.EcoreUtil

/**
 * Transforms SUnit type model to a NUnit C# code model
 * 
 * @author Axel Terfloth
 */
class SUnit2NunitTransformation extends Transformation<Package, Package> {

	@Inject protected extension TypeBuilder
	@Inject protected extension ExpressionBuilder
	@Inject protected extension ExpressionExtensions
	@Inject protected extension NUnitClass
	@Inject protected extension NUnitLib
	@Inject protected extension SUnitAnnotations
	@Inject protected extension SUnitLib
	@Inject protected extension CsharpNaming
	@Inject protected extension CsharpClass
	@Inject protected extension CsharpTypeBuilder
	@Inject protected extension CsharpCompilationUnit
	@Inject protected extension CsharpNamespace
	@Inject protected extension NUnitAnnotations
	// TODO: This should be moved out from sgraph 
	@Inject protected extension StatechartAnnotations
	@Inject protected extension VirtualTimer
	@Inject protected extension TimedInterface
	@Inject protected extension GeneratorAssignment
	@Inject protected extension ContainedCode

	@Inject protected extension RaiseEventModification
	@Inject protected extension Statechart2TypeTransformation
	@Inject protected extension OutEventSubscriber
	@Inject protected extension OutEventMembers
	@Inject protected extension VoidIsDefaultReturnTypeModification
	@Inject protected extension CsharpStatemachineOperationCallbacks
	
	@Inject protected extension StateEnum

	@Inject protected extension IVariableCode

	@Inject protected extension StatechartLibrary

	@Inject protected extension EventRaiser

	@Inject protected extension StatechartUtil

	@Inject protected Slang2CsharpTransformation slang2Csharp

	@Inject protected StatechartLibrary scLibrary
	@Inject protected CsharpStatemachineLibrary smLibrary

	protected extension TypesFactory tFactory = TypesFactory.eINSTANCE
	protected extension ExpressionsFactory expFactory = ExpressionsFactory.eINSTANCE

	override protected toTarget(Package it) {

		smLibrary.statemachineLibrary.transformedRootFrom(scLibrary.typesPackage)
		val interfaceTypes = scLibrary.typesPackage.eAllContents.filter(Type).toSet

		referencedTypes.filter[t|! interfaceTypes.contains(t)].forEach [ sourceType |
			if (sourceType !== null) {
				val sourceDecl = if (sourceType.eContainer !== null && sourceType.eContainer instanceof Package)
						sourceType.eContainer as Package
					else
						sourceType

				val targetDecl = slang2Csharp.transform(sourceDecl)
				targetDecl.transformedRootFrom(sourceDecl)
			}
		]

		it.toNUnit as Package
	}

	override protected void modifyTarget(Package it) {
		interfaceTypeModification
		stmType2stmClass
		applyCsharpNamingConventions
		setOperationCallForReferences
		defineOutEventMembers
		substituteEventRaisingExpressionsByEventRaisers
		// 'substituteOutEventReferencesByOutEventMembers' has to be before defining init op
		substituteOutEventReferencesByOutEventMembers
		addSetUpAndTearDown
		fixClassMemberNameClashes
		useVoidAsDefaultReturnType
		addUserDeclaredAnnotation
	}
	
	def protected stmType2stmClass(Package container){
		
			container.eAllContents.filter(ComplexType).filter[isStatechartType].forEach[st |
				val oldEnum = st.features.filter(EnumerationType).head
				if(oldEnum !== null)
					st.features.remove(oldEnum)
				st.features += (st.originSource as Statechart).defineStateEnum => [eAdapters.clear]
			]		
	}

	def addUserDeclaredAnnotation(Package it) {
		eAllContents.filter(ComplexType).filter[statechartType].head.features.filter(VariableDefinition).forEach [ dec |
			if (dec.initialValue === null)
				dec.userDeclared
		]
	}

	def dispatch Declaration toNUnit(Package sunitPackage) {

		_package => [ p |

			sunitPackage.eAllContents.filter(ComplexType).filter[isTestClass].forEach [ suTestClass |

				p.member += _compilationUnit => [ cu |
					cu.name = suTestClass.csharpName

					cu.copyImports(sunitPackage)
					cu._import("NUnit.Framework")

					var nunitClass = suTestClass.toNUnit
					var namespace = suTestClass.createNamespace
					if (namespace !== null) {
						namespace.member += nunitClass
					}

					cu.member += EcoreUtil.getRootContainer(nunitClass) as Declaration
				]
			]

			p.useNUnitInsteadOfSunitAssert(sunitPackage.sunitAssert)
			p.substituteMockReturn(sunitPackage.sunitReturn)
			p.substituteMocking(sunitPackage.sunitMock)
			p.substituteVerifyCalledTimes(sunitPackage.sunitTimes)
			p.substituteVerifyCalled(sunitPackage.sunitCalled)

			p.transformedRootFrom(sunitPackage)
		]
	}

	def protected substituteMockReturn(Package root, Operation sunitReturn) {
		root.eAllContents.filter(FeatureCall).filter[feature === sunitReturn].forEach [ retFeature |
			retFeature.feature = root.nunitMockReturns
			// If the return value is an Operation call we have to use lambdas
			if (retFeature.arguments.head.value instanceof ElementReferenceExpression) {
				val argRef = retFeature.arguments.head.value as ElementReferenceExpression
				if (argRef.reference instanceof Operation) {
					retFeature.arguments.head.value = argRef._lambda(
						_lambda_param("", _void)
					)
				}
			}
		]
	}

	def protected substituteVerifyCalledTimes(Package root, Operation sunitTimes) {
		root.eAllContents.filter(FeatureCall).filter[feature === sunitTimes].forEach [ retFeature |
			retFeature.feature = root.nunitTimesClass
		]
	}

	def protected substituteMocking(Package root, Operation sunitMock) {
		if (root.eAllContents.filter(ArgumentExpression).exists[featureOrReference === sunitMock])
			root.eAllContents.filter(Package).filter[isCompilationUnit].head._import("Moq")

		root.eAllContents.filter(ArgumentExpression).filter[featureOrReference === sunitMock].forEach [ argExp |

			// Query ocb class for the referenced operation
			val testOcb = root.lookupOcbClassFor(argExp.arguments.head.value)

			// Create a class for the mock<iface>
			val mockClass = root.createMockClassFor(testOcb)

			// Create a part for the mock<iface>
			val mockPart = root.createMockPartFor(mockClass)

			val mockOcb = root.createMockOcbPartFor(testOcb)

			val targetOp = testOcb.targetOperation(argExp)

			val opArg = root.calculateArgumentsForOperation(argExp, targetOp)

			EcoreUtil.replace(
				argExp,
				mockPart._ref._call(root.nunitMockSetup)._with(
					mockOcb._ref._call(targetOp)._with(opArg as Expression[])._lambda(
						_lambda_param(mockOcb.name, mockOcb.type)
					)
				)
			)
		]
	}

	def protected calculateArgumentsForOperation(Package root, ArgumentExpression argExp, Operation targetOp) {
		(argExp.arguments.head.value as ArgumentExpression).arguments.head === null && !targetOp.parameters.nullOrEmpty
			? targetOp.parameters.map[type].map [ t |
			root.nunitItClass._ref._call(root.nunitIsAny.copy => [ op |
				op.typeParameters += createTypeParameter => [
					bound = t
				]
			])
		]
			: (argExp.arguments.head.value as ArgumentExpression).arguments.map[value]
	}

	def protected substituteVerifyCalled(Package root, Operation sunitCalled) {

		if (root.eAllContents.filter(ArgumentExpression).exists[featureOrReference === sunitCalled])
			root.eAllContents.filter(Package).filter[isCompilationUnit].head._import("Moq")

		root.eAllContents.filter(ElementReferenceExpression).filter[featureOrReference === sunitCalled].forEach [ argExp |

			// Query ocb class for the referenced operation
			val testOcb = root.lookupOcbClassFor(argExp.arguments.head.value)

			// Create a class for the mock<iface>
			val mockClass = root.createMockClassFor(testOcb)

			// Create a part for the mock<iface>
			val mockPart = root.createMockPartFor(mockClass)

			val mockOcb = root.createMockOcbPartFor(testOcb)

			val times = argExp.eContainer instanceof FeatureCall ? argExp.eContainer as FeatureCall : null

			val targetOp = testOcb.targetOperation(argExp)

			val opArg = root.calculateArgumentsForOperation(argExp, targetOp)

			if (times !== null) {
				val replacementExpr = mockPart._ref._call(root.nunitMockVerify)._with(
					mockOcb._ref._call(targetOp)._with(opArg as Expression[])._lambda(
						_lambda_param(mockOcb.name, mockOcb.type)
					),
					if (((times.arguments.head.value as PrimitiveValueExpression).value as IntLiteral).value == 0)
						root.nunitTimesClass._ref._call(root.nunitNever)
					else
						root.nunitTimesClass._ref._call(root.nunitExactly)._with(times.arguments.head.value)
				)
				EcoreUtil.replace(times, replacementExpr)
			} else {
				val replacementExpr = mockPart._ref._call(root.nunitMockVerify)._with(
					mockOcb._ref._call(targetOp)._with(opArg as Expression[])._lambda(
						_lambda_param(mockOcb.name, mockOcb.type)
					)
				)
				EcoreUtil.replace(argExp, replacementExpr)
			}

		]
	}

	def protected targetOperation(ComplexType testOcb, ArgumentExpression argExp) {
		val targetOp = testOcb.features.filter(Operation).filter [ op |
			(argExp.arguments.head.value as ArgumentExpression).featureOrReference === op.originSource
		].head

		targetOp === null ? (argExp.arguments.head.value as ArgumentExpression).
			featureOrReference as Operation : targetOp
	}

	def dispatch lookupOcbClassFor(Package root, ArgumentExpression opFeature) {
		root.eAllContents.filter(Operation).filter [
			eContainer.isCsharpInterface && originSource === opFeature.featureOrReference.originSource
		].head.eContainer as ComplexType
	}

	def dispatch lookupOcbClassFor(ComplexType root, Expression opFeature) {
		root.eAllContents.filter(ComplexType).filter[isCsharpInterface].filter [
			features.filter(Operation).exists [
				originSource === opFeature.featureOrReference.originSource
			]
		].head
	}

	def lookupMockPartFor(Package root, ComplexType mockClass) {
		root.eAllContents.filter(ComplexType).filter[isTestFixture].head.features.filter(Part).filter [
			type === mockClass
		].head
	}

	def createMockPartFor(Package root, ComplexType mockClass) {
		if (root.lookupMockPartFor(mockClass) !== null)
			return root.lookupMockPartFor(mockClass)
		else {
			val mockPart = _part(mockClass.mockPartName, mockClass) => [
				_user_declared
				_required
				generateDefinitionWith[variableDeclarationCode]
			]
			root.eAllContents.filter(ComplexType).filter[isTestFixture].head.features += mockPart
			return mockPart
		}
	}

	def String mockPartName(Type mockClass) {
		if (mockClass.superTypes.head.type.eContainer.isNamedIface)
			return (mockClass.superTypes.head.type.eContainer as NamedElement).name +
				mockClass.superTypes.head.type.name + mockClass.name
		else
			return mockClass.superTypes.head.type.name + mockClass.name
	}

	def lookupMockOcbPartFor(Package root, ComplexType mockClass) {
		root.eAllContents.filter(ComplexType).filter[isTestFixture].head.features.filter(Part).filter [
			eContainer.isCsharpInterface && type === mockClass
		].head
	}

	def createMockOcbPartFor(Package root, ComplexType testOcb) {
		if (root.lookupMockOcbPartFor(root.lookupMockClassFor(testOcb)) !== null)
			return root.lookupMockOcbPartFor(root.lookupMockClassFor(testOcb))
		else {
			val mockOcbPart = _part("ocb" + testOcb.name, root.lookupMockClassFor(testOcb)) => [_private]
			testOcb.features += mockOcbPart
			return mockOcbPart
		}
	}

	def lookupMockClassFor(Package root, ComplexType ocbIface) {
		root.eAllContents.filter(ComplexType).filter [
			origin === root.nunitMockClass && superTypes.head.type === ocbIface
		].head
	}

	def createMockClassFor(Package root, ComplexType ocbIface) {
		if (root.lookupMockClassFor(ocbIface) !== null)
			return root.lookupMockClassFor(ocbIface)
		else {
			val mockClass = root.nunitMockClass.copy => [
				_no_namespace
				traceOrigin(root.nunitMockClass)
				superTypes += createTypeSpecifier => [
					type = ocbIface => [_required]
				]
			]
			root.eAllContents.filter(ComplexType).filter[isTestFixture].head.features += mockClass
			return mockClass
		}
	}

	def protected copyImports(Package cu, Package sunitPackage) {
		sunitPackage.eAllContents.filter(Package).filter[isSUnit].forEach [ sunit |
			sunit.imports.forEach [ imp |
				cu._import(imp)
			]
		]
	}

	def protected dispatch Package createNamespace(ComplexType it) {
		if(it.eContainer !== null) return it.eContainer.createNamespace
	}

	def protected dispatch Package createNamespace(EObject it) {
		null
	}

	def protected dispatch Package createNamespace(Void it) {
		null
	}

	def protected dispatch Package createNamespace(Package it) {
		if (! name.isNullOrEmpty) {
			val namespace = _namespace => [ns|ns.name = it.name]
			var parentNamespace = it.eContainer.createNamespace
			if (parentNamespace !== null) {
				parentNamespace.member += namespace
			}
			return namespace
		}
		return null
	}

	def dispatch Declaration toNUnit(ComplexType sunitClass) {

		if (sunitClass.isTestClass) {
			_nunitClass(sunitClass.name) => [ nunitClass |
				sunitClass.features.forEach [ f |
					f.toNUnit.ifPresent [
						nunitClass.features += it
					]
				]

				if (nunitClass.features.filter(Part).filter[type.statechartType].exists [
					!type.getOriginStatechart.eAllContents.filter(TimeEventSpec).nullOrEmpty ||
						type.getOriginStatechart.isCycleBased
				])
					nunitClass.features += virtualTimerVar => [generateDefinitionWith[variableDeclarationCode]]
				nunitClass.transformedFrom(sunitClass)
			]
		} // Create OCB for Statechart Type
		else if (sunitClass.statechartType) {
			sunitClass.operationCallback(sunitClass)
			sunitClass.createCopy as ComplexType => [generateDefinitionWith[containedCode]]
		} else
			sunitClass.createCopy as ComplexType => [generateDefinitionWith[containedCode]]
	}

	def setOperationCallForReferences(Package it) {
		eAllContents.filter(ElementReferenceExpression).filter[reference instanceof Operation].forEach [
			operationCall = true
		]
		eAllContents.filter(FeatureCall).filter[feature instanceof Operation].forEach [
			operationCall = true
		]
	}

	def addSetUpAndTearDown(Package it) {
		val nunitClass = it.eAllContents.filter(ComplexType).filter[isTestFixture].head
		nunitClass.features += nunitClass.createSetUp
		nunitClass.features += nunitClass.createTearDown
		it.eAllContents.filter(Part).filter[type.statechartType].forEach [
			annotations.clear
			_required
			_public
		]
	}

	def interfaceTypeModification(Package it) {
		eAllContents.filter(ComplexType).filter[isNamedIface].forEach [
			// TODO: Should we take care of naming convention here or in the 'CsharpNaming'?
			// (In my opinion in Csharp naming, or in a separate transformation modification as we apply this both here and in NUnit)
			name = "I" + (originSource as NamedElement).name.toFirstUpper
			generateDefinitionWith[containedCode]
		]
	}

	def dispatch protected boolean isAssignableTo(Type it, ComplexType superType) {
		return false
	}

	def dispatch protected boolean isAssignableTo(ComplexType it, ComplexType superType) {
		return ( (it == superType) || it.superTypes.exists[t|t.type.isAssignableTo(superType)] )
	}

	def createSetUp(ComplexType nunitClass) {

		val statemachineVariable = nunitClass.features.filter(Part).filter[type.isStatechartType && !isChildStatechart].
			head

		val timer = nunitClass.features.filter(Part).filter[type === virtualTimerType].head
		val objects = nunitClass.features.filter(Part).map [ p |
			_assignment(
				p._ref,
				createInitializationExpression => [
					arguments += createArgument => [
						parameter = p
					]
				]
			)
		]
		_nunitSetUp("Init_", createTypeSpecifier => [
			type = _void
		]) => [
			implementation = _block(
				objects
			) => [

				val stmType = nunitClass.features.filter(ComplexType).filter[isStatechartType].head

				// Initialize and set mock ocbs
				val allOcbIface = nunitClass.eAllContents.filter(ArgumentExpression).filter [
					featureOrReference !== null && featureOrReference instanceof Operation
				].map [
					stmType.lookupOcbClassFor(it)
				].filterNull.toSet

				allOcbIface.forEach [ testOcb |
					val rootPackage = EcoreUtil.getRootContainer(nunitClass, true) as Package
					val mockClass = rootPackage.createMockClassFor(testOcb)

					val mockPart = rootPackage.createMockPartFor(mockClass)
					if (testOcb.eContainer.namedIface) {
						val ifaceProp = (testOcb.eContainer as ComplexType).ifacePartQuery(stmType)
						expressions += statemachineVariable._ref._dot(ifaceProp)._call(
							(testOcb.eContainer as ComplexType).features.filter(Operation).filter [
								parameters.exists[type === testOcb]
							].head
						)._with(mockPart._ref._dot(rootPackage.nunitMockObject))
					} else {
						expressions += statemachineVariable._ref._call(
							stmType.features.filter(Operation).filter [
								parameters.exists[type === testOcb]
							].head
						)._with(mockPart._ref._dot(rootPackage.nunitMockObject))
					}
				]

				// Timer related assignments
				if (timer !== null) {
					if (statemachineVariable.type.isAssignableTo(smLibrary.cycleBasedStatemachineInterface))
						expressions += statemachineVariable.schedulePeriodicTaskFor(timer, nunitClass)
					if (timedType !== null && (statemachineVariable.type as ComplexType).requiresTimeRelatedConcepts)
						expressions += statemachineVariable._ref._dot(timedType.setTimerServiceOp)._with(timer._ref)
				}

				val allParts = nunitClass.features.filter(Part).filter[isChildStatechart].toList
				
				allParts.add(statemachineVariable)
				
				// Child stm init and set
				allParts.forEach [ stmPart |
					stmPart.getChildStatemachineVariable(allParts).forEach [ stmVar |
						//Filter parts based on their types, whether that matches the required type or not
						val potentialParts = allParts.filter[(origin as EObject).originTraces.filter(VariableDefinition).head === stmVar.origin]
						//If there is multiple fitting types search for the origin parent part
						val targetPart = potentialParts.size == 1 ? potentialParts.head : potentialParts.filter[stmPart.origin === (origin as EObject).originTraces.filter(Part).head].head
						expressions += _assignment(
							if (stmVar.eContainer.isNamedIface)
								stmPart._ref._dot(stmVar.eContainer)._dot(stmVar)
							else
								stmPart._ref._dot(stmVar),
							if(targetPart === null && stmVar.type === stmPart.type)
								stmPart._ref
							else
								targetPart._ref
						)
					]
					// Timer related assignments
					if (timer !== null && stmPart !== statemachineVariable && (stmPart.type as ComplexType).requiresTimeRelatedConcepts) {
						if (stmPart.type.isAssignableTo(smLibrary.cycleBasedStatemachineInterface))
							expressions += stmPart.schedulePeriodicTaskFor(timer, nunitClass)
						if (timedType !== null && (stmPart.type as ComplexType).requiresTimeRelatedConcepts)
							expressions += stmPart._ref._dot(timedType.setTimerServiceOp)._with(timer._ref)
					}
				]

				// Subscription for out events
				nunitClass.eAllContents.filter(ComplexType).filter[statechartType].forEach[ t |
					t.eAllContents.filter(Property).filter [ isHandler ]
					.forEach [ h |
						val partForStm = allParts.filter[type.originSource === t.originSource].head
						if (h.eContainer.isNamedIface)
							expressions +=
								partForStm._ref._dot(
									(h.eContainer.eContainer as ComplexType).features.filter(TypedDeclaration).filter [
										type.originSource === h.eContainer.originSource
									].head)._dot((h.origin as EObject))._assignment(h._ref)=> [
										operator = AssignmentOperator::ADD_ASSIGN
									]
						else
							expressions += partForStm._ref._dot(h.origin as EObject)._assignment(h._ref)=> [
								operator = AssignmentOperator::ADD_ASSIGN
							]
					]
				]
			]
		]
	}

	def protected requiresTimeRelatedConcepts(ComplexType scType) {
		val originSc = scType.getOriginStatechart
		if (!originSc.eAllContents.filter[e|e instanceof TimeEventSpec || e instanceof TimeEvent].nullOrEmpty) {
			createTimedInterface
			scType.features += timedType.copy
		}

	}

	def protected ifacePartQuery(ComplexType ifaceType, ComplexType stmType) {
		stmType.features.filter(Property).filter [ p |
			p.originSource === ifaceType.originSource
		].head
	}

	def protected getChildStatemachineVariable(Part statemachineVariable, List<Part> childStmPart) {
		statemachineVariable.type.eAllContents.filter(VariableDefinition).filter[type !== null && type instanceof ComplexType && !(type instanceof EnumerationType)].filter [ td |
			childStmPart.exists[csp | td.type.originSource === csp.type.originSource] 
		]
	}

	def schedulePeriodicTaskFor(TypedDeclaration statemachineVariable, Part timer, ComplexType nunitClass) {

		timer._ref._dot(timer.type.schedulePeriodicalTaskOp)._with(
			createInitializationExpression => [
				arguments += createArgument => [
					value = createInitializationExpression => [
						arguments += createArgument => [
							value = statemachineVariable._ref
						]
					]
					parameter = timer.type.cycleTimeEventTaskPart.copy
				]
			],
			(nunitClass.origin as EObject).getCyclePeriod._value,
			(nunitClass.origin as EObject).getCyclePeriod._value
		)
	}

	def createTearDown(ComplexType sunitClass) {
		val nullObjects = sunitClass.features.filter(Part).map[_assignment(it._ref, _null)]
		_nunitTearDown("Cleanup_", createTypeSpecifier => [
			type = _void
		]) => [
			implementation = _block(
				nullObjects
			)
		]
	}

	def dispatch Declaration toNUnit(Operation op) {

		(if (op.isSUnitTest) {
			_nunitTest(op.name, op.typeSpecifier) => [if(op.isIgnoredTest) _annotate(NUnitAnnotations.NUNIT_IGNORE)]
		} else {
			_csharpMethod(op.name, op.typeSpecifier)
		}) => [
			op.parameters.forEach[p|it._param(p.name, p.typeSpecifier)]
			it.implementation = EcoreUtil.copy(op.implementation)
			it.transformedFrom(op)
		]
	}

	def dispatch Declaration toNUnit(Property v) {
		v._csharpVariable => [
			transformedFrom(v)
		]
	}

	def dispatch Declaration toNUnit(Part p) {
		p._csharpVariable => [
			transformedFrom(p)
		]
	}

	def dispatch Declaration toNUnit(TypedDeclaration p) {
		p.copy => [traceOrigin(p)]
	}

	def protected useNUnitInsteadOfSunitAssert(Package root, Operation sunitAssert) {

		root.member += defineNunitPackage

		val nunitAssertClass = root.nunitAssertClass
		val nununitAssertTrue = root.nunitAssertIsTrue

		root.eAllContents.filter(ElementReferenceExpression).filter[reference === sunitAssert].forEach [
			EcoreUtil.replace(it, nunitAssertClass._ref._call(nununitAssertTrue) => [ nunitCall |
				it.arguments.forEach[arg|nunitCall.arguments += arg.copy]
			])
		]
	}

	def void substituteOutEventReferencesByOutEventMembers(Package it) {
		it.eAllContents.filter(FeatureCall).filter [ fc |
			fc.feature !== null && fc.feature.origin instanceof EventDefinition &&  (fc.feature.origin as EventDefinition).direction === Direction.OUT
		].forEach [ fc |
			EcoreUtil.replace(fc, createElementReferenceExpression => [ newEr |
				newEr.reference = (fc.feature.origin as EventDefinition).outEventMember
			])
		]
	}

	def void fixClassMemberNameClashes(Package it) {

		it.eAllContents.filter(Package).filter[isCompilationUnit].forEach [ cu |
			val Set<String> usedTypeNames = new HashSet
			usedTypeNames.addAll(it.eAllContents.filter(Type).toSet.map[name])
			usedTypeNames.addAll(cu.referencedTypes.filter[it !== null].map[name])

			cu.eAllContents.filter(Declaration).filter[! (it instanceof Type)].forEach [
				if(usedTypeNames.contains(it.name)) it.name = it.name + "_"
			]
		]
	}

}
