/**
 * Copyright (c) 2016 itemis AG - All rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * 
 * Contributors:
 * 	Thomas Kutz - itemis AG
 * 
 */
package com.yakindu.sct.domain.c.runtime.exec

import com.google.inject.Inject
import com.yakindu.base.expressions.expressions.AssignmentExpression
import com.yakindu.base.expressions.expressions.AssignmentOperator
import com.yakindu.base.expressions.interpreter.SlotResolutionExceptionSupplier
import com.yakindu.base.expressions.interpreter.context.IExecutionSlotResolver
import com.yakindu.sct.model.sruntime.CompositeSlot
import com.yakindu.sct.model.sruntime.ExecutionSlot
import com.yakindu.sct.model.sruntime.ReferenceSlot
import com.yakindu.sct.simulation.core.sexec.interpreter.StextExpressionInterpreter

/**
 * 
 * @author Thomas Kutz - Initial contribution and API
 * 
 */
class CStatementInterpreter extends StextExpressionInterpreter {

	@Inject extension IExecutionSlotResolver
	
	/**
	 * For complex types, assignments are handled in copy-by-value manner.
	 */
	override Object executeAssignment(AssignmentExpression assignment) {
		var scopeVariable = context.resolve(assignment.varRef)
			.orElseThrow(SlotResolutionExceptionSupplier.forContext(assignment.varRef))

		if (scopeVariable instanceof ReferenceSlot) {
			// here we want to assign the slot instead of its value
			var slot = context.resolve(assignment.expression)
				.orElseThrow(SlotResolutionExceptionSupplier.forContext(assignment.expression))
			scopeVariable.reference = slot
			return slot.value
		}

		var result = assignment.expression.execute

		if (assignment.operator == AssignmentOperator::ASSIGN) {
			if (result instanceof CompositeSlot &&
				ts.isSuperType(scopeVariable.type.originType, (result as CompositeSlot).type.originType)) {
				copyValues(scopeVariable, result as CompositeSlot)
			} else {
				scopeVariable.value = if(result !== null) cast(result, scopeVariable.type) else null
			}
		} else {
			var operator = assignFunctionMap.get(assignment.operator.getName())
			scopeVariable.value = if (result !== null)
				cast(evaluate(operator, scopeVariable.getValue, result), scopeVariable.type)
			else
				null
		}
		scopeVariable.value
	}

	protected def dispatch void copyValues(CompositeSlot target, CompositeSlot values) {
		for (ExecutionSlot slot : target.slots) {
			val valueSlot = values.slots.findFirst[name == slot.name]
			copyValues(slot, valueSlot)
		}
	}

	protected def dispatch void copyValues(ExecutionSlot targetVar, ExecutionSlot valueVar) {
		targetVar.value = valueVar.value
	}
}
