/**
 * Copyright (c) 2022 itemis AG - All rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * 
 * Contributors:
 * Andreas Muelder - itemis AG	
 * Thomas Kutz - itemis AG
 * 
 */
package com.yakindu.sct.domain.c.runtime.resource

import com.google.inject.Inject
import com.yakindu.base.types.Declaration
import com.yakindu.base.types.Package
import com.yakindu.base.types.TypeAlias
import com.yakindu.base.types.TypeBuilder
import com.yakindu.base.types.TypesFactory
import com.yakindu.base.types.typesystem.ITypeSystem
import com.yakindu.sct.commons.EmfUriUtil
import com.yakindu.sct.commons.PathHelper
import com.yakindu.sct.domain.c.runtime.CRuntimeActivator
import com.yakindu.sct.domain.c.runtime.resource.transform.ICToModelTrafo
import com.yakindu.sct.domain.c.runtime.resource.transform.PackageHierarchy
import com.yakindu.sct.domain.c.runtime.resource.transform.c.CTypeNaming
import com.yakindu.sct.domain.c.runtime.resource.transform.c.UnresolvableTypeProblem
import com.yakindu.sct.generator.c.typesystem.CTypeSystem
import java.io.IOException
import java.io.InputStream
import java.nio.file.Path
import java.util.Map
import org.eclipse.cdt.core.CCorePlugin
import org.eclipse.cdt.core.dom.IName
import org.eclipse.cdt.core.dom.ast.IASTNameOwner
import org.eclipse.cdt.core.dom.ast.IASTTranslationUnit
import org.eclipse.cdt.core.dom.ast.IBinding
import org.eclipse.cdt.core.dom.ast.IProblemBinding
import org.eclipse.cdt.core.model.CoreModel
import org.eclipse.cdt.core.model.ICElement
import org.eclipse.cdt.core.model.ITranslationUnit
import org.eclipse.cdt.internal.core.model.CModelManager
import org.eclipse.core.resources.IFile
import org.eclipse.core.resources.ResourcesPlugin
import org.eclipse.core.runtime.CoreException
import org.eclipse.emf.common.util.URI
import org.eclipse.emf.ecore.resource.impl.ResourceImpl
import org.eclipse.xtend.lib.annotations.Accessors
import org.eclipse.xtext.naming.QualifiedName

/**
 * Resource implementation for C header files with implicit transformation into Types metamodel when loading this resource.
 * 
 * @author Thomas Kutz - itemis AG
 */
class CHeaderResource extends ResourceImpl {
	
	protected Package rootPackage;
	@Accessors protected IASTTranslationUnit ast

	/** the C typesystem where built-in types are defined */
	@Inject protected ITypeSystem ts

	protected static extension NameToIdConverter = new NameToIdConverter

	@Inject protected extension ICToModelTrafo trafo
	
	@Inject protected extension CachingPathResolver

	@Inject extension CTypeNaming
	
	@Inject protected extension PathHelper
	
	@Inject protected extension ITypeSystem
	@Inject protected extension TypeBuilder
	
	protected Path location

	protected extension TypesFactory factory = TypesFactory.eINSTANCE

	override void doLoad(InputStream inputStream, Map<?, ?> options) throws IOException {
		try {
			ast = createAST()
			rootPackage = createRootPackage
			try {
				if (ast !== null) {
					ast.index?.acquireReadLock
					trafo.transformToModel(this, options)
				}
			} finally {
				ast?.index?.releaseReadLock
			}
			getContents().add(rootPackage)
		} catch (Exception e) {
			// Print errors because the base resource drops them silently
			e.printStackTrace
			throw e
		}
	}

	def getIFile() {
		EmfUriUtil.toFile(getURI())
	}
	
	def Path getResourceLocation() {
		if (location === null) {
			var file = getIFile()
			if (file !== null)
				location = file.toPath
			else
				location = URI.toPath
		}
		return location
	}

	def Path pathToDeclaringHeader(IASTNameOwner specifier) {
		val name = specifier.name
		if (name === null || name.toString.isEmpty) {
			// unnamed types can not be included from somewhere else
			return resourceLocation
		}

		try {
			// ast might be null, see com.yakindu.sct.domain.c.runtime.resource.CHeaderResource.createFileAST()
			if (ast.index !== null)
				ast.index.acquireReadLock
			return pathToDeclaringHeader(specifier.ASTName.resolveBinding)
		} finally {
			if (ast.index !== null)
				ast.index.releaseReadLock
		}
	}

	def Path pathToDeclaringHeader(IBinding binding) {
		if (binding !== null && !(binding instanceof IProblemBinding)) {
			var IName[] declNames = ast.getDefinitions(binding)
			val path = declNames?.findFirst[fileLocation !== null]
			if (path !== null)
				return path.fileLocation.fileName.resolve
		}
		return resourceLocation
	}
	
	def addToPackage(Declaration member) {
		addToPackage(rootPackage, member)
	}
	
	
	def addToPackage(Declaration member, PackageHierarchy hierarchy) {
		var pkg = rootPackage
		for (seg : hierarchy?.getSegments) {
			var next = pkg.member.filter(Package).filter[name == seg].head
			if (next === null) {
				next = createPackage => [name = seg]
				pkg.member += next
			}
			pkg = next
		}
		addToPackage(pkg, member)
	}
	
	def protected IASTTranslationUnit createAST() {
		var IASTTranslationUnit ast
		if (getURI().isPlatform) {
			val res = EmfUriUtil.toFile(getURI())
			if(res !== null) {
				val unit = CoreModel.getDefault().create(res.fullPath)
				ast = createASTfromUnit(unit)
			}
		} else if (getURI().isFile) {
			if (URI.query.nullOrEmpty) {
				throw new IllegalStateException("Context workspace project required for file URIs!")
			}
			var project = ResourcesPlugin.workspace.root.getProject(URI.query)
			var cProject = CCorePlugin.getDefault().coreModel.CModel.getCProject(project.name)
			val unit = CModelManager.^default.createTranslationUnitFrom(cProject, normalizeURI())
			ast = createASTfromUnit(unit)
		}
		ast
	}
	
	protected def java.net.URI normalizeURI() {
		val uri = new java.net.URI(getURI().trimQuery.toString())
		uri.normalize
	}

	protected def IASTTranslationUnit createASTfromUnit(ICElement unit) {
		if (unit === null || !(unit instanceof ITranslationUnit))
			return null
		
		val tu = unit as ITranslationUnit
		val index = CCorePlugin.indexManager.getIndex(tu.CProject)
		try {
			index.acquireReadLock
			return tu.getAST(index, tu.getASTStyle);
		} catch (CoreException e) {
			e.printStackTrace()
			return null
		} finally {
			index.releaseReadLock
		}
	}
	
	def protected int getASTStyle(ITranslationUnit unit) {
		if (!isIndexerEnabled(unit) || isIndexerRunning) {
			// we do not have an index at this point, either because it is disable or still running, so we need to parse all needed headers
			return ITranslationUnit.AST_SKIP_FUNCTION_BODIES
					.bitwiseOr(ITranslationUnit.AST_CONFIGURE_USING_SOURCE_CONTEXT)
		} else {
			// there is an index, so let's skip all headers and resolve references with the help of the index
			return ITranslationUnit.AST_SKIP_FUNCTION_BODIES
					.bitwiseOr(ITranslationUnit.AST_SKIP_ALL_HEADERS)
					.bitwiseOr(ITranslationUnit.AST_CONFIGURE_USING_SOURCE_CONTEXT)
		}
	}
	
	protected def boolean isIndexerEnabled(ITranslationUnit unit) {
		CCorePlugin.indexManager.isProjectIndexed(unit.CProject)
	}
	
	protected def boolean isIndexerRunning() {
		!CCorePlugin.indexManager.isIndexerIdle
	}

	def protected Package createRootPackage() {
		val res = getIFile()
		val String packageName = {
			if (res !== null)
				computePackageName(res)
			else
				computePackageName(getURI())
		}
		createPackage => [
			it.domainID = CRuntimeActivator.C_DOMAIN_ID
			it.name = packageName
			_no_namespace
		]
	}

	def computePackageName(URI uri) {
		val pathToHeader = uri.trimFileExtension.lastSegment
		QualifiedName.create(pathToHeader.convertToValidIdentifier).toString
	}

	def static computePackageName(IFile headerFile) {
		val resolvedFile = new LinkedFolderResolver().resolve(headerFile)
		val pathToHeader = resolvedFile.projectRelativePath.removeFileExtension
		QualifiedName.create(pathToHeader.segments.map[convertToValidIdentifier]).toString
	}

	def protected addToPackage(Package pkg, Declaration member) {
		if (member !== null && member.name != CTypeNaming.ANONYMOUS_NAME && member != ts.getType(CTypeSystem.UNSUPPORTED_TYPE)) {
			if(member instanceof TypeAlias) {
				pkg.removeIncompleteForwardDeclaredTypeAlias(member)
			}
			pkg.member += member
		}
	}
	
	def void removeIncompleteForwardDeclaredTypeAlias(Package pkg, TypeAlias alias) {
		val existingTypeAlias = pkg.eAllContents.filter(TypeAlias).filter[isAny(type)].filter[it.name == alias.name].
			toList
		if (!existingTypeAlias.nullOrEmpty) {
			alias.removeUnresolvableTypeProblem
		}
		pkg.member.removeAll(existingTypeAlias)
	}

	def void removeUnresolvableTypeProblem(TypeAlias alias) {
		val resolvedErrors = errors.toList.filterNull.filter [
			message == UnresolvableTypeProblem.getMessage(alias.type.name)
		]
		errors.removeAll(resolvedErrors)
	}

	
	def getPackageMembers() {
		rootPackage.member
	}
}
