/**
 * Copyright (c) 2025 itemis AG - All rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * 
 */
package com.itemis.create.statechart.generator.csharp.transformation

import com.google.inject.Inject
import com.itemis.create.base.generator.core.Transformation
import com.itemis.create.base.generator.csharp.codemodel.CsharpStatemachineLibrary
import com.yakindu.base.base.NamedElement
import com.yakindu.base.expressions.ExpressionBuilder
import com.yakindu.base.expressions.expressions.ArgumentExpression
import com.yakindu.base.expressions.expressions.ElementReferenceExpression
import com.yakindu.base.expressions.expressions.FeatureCall
import com.yakindu.base.expressions.util.ExpressionExtensions
import com.yakindu.base.types.Argument
import com.yakindu.base.types.ComplexType
import com.yakindu.base.types.Declaration
import com.yakindu.base.types.EnumerationType
import com.yakindu.base.types.Enumerator
import com.yakindu.base.types.Operation
import com.yakindu.base.types.Type
import com.yakindu.base.types.TypedDeclaration
import com.yakindu.base.types.TypesFactory
import com.yakindu.sct.generator.core.codemodel.StatemachineClass
import com.yakindu.sct.model.sgraph.Statechart
import com.yakindu.sct.model.sgraph.Vertex
import com.yakindu.sct.types.lib.StatechartLibrary
import com.yakindu.sct.types.resource.Statechart2TypeTransformation
import java.util.Iterator
import org.eclipse.emf.ecore.EObject
import org.eclipse.emf.ecore.util.EcoreUtil

/**
 * There is currently a gap in our Type Description, specifically how they created and stored 
 * One statechart type description gets created and stored when the statechart is loaded into memory,
 * and a second one when the generation is started. As the exec flow gets created from the first one
 * this class is required to link the gap between those, as it would take more effort, to have only one
 * type descriptor for the whole runtime (which would be the clean solution).
 * 
 * @author laszlo kovacs - Initial contribution.
 */
class SctTypes2CsharpTypesTransformation extends Transformation<ComplexType, ComplexType> {
	
	@Inject protected CsharpStatemachineLibrary smLibrary
	@Inject protected StatechartLibrary lib
	@Inject protected extension StatemachineClass
	@Inject protected extension ExpressionExtensions
	@Inject protected extension ExpressionBuilder
	@Inject protected extension Statechart2TypeTransformation
	
	extension TypesFactory tFactory = TypesFactory.eINSTANCE
	
	override protected toTarget(ComplexType source) {
		source.substitueStmTypes
		source.substitueStateEnumMethodMember(source.stateMachineClass)
		source.transformedRootFrom(source)
		source
	}
	
	def protected substitueStmTypes(ComplexType flow) {
		flow.eAllContents.filter(TypedDeclaration).filter[typeSpecifier !== null].forEach[
			if(typeSpecifier.type.name == lib.cycleBased.name)
				typeSpecifier = createTypeSpecifier => [ ts |
					ts.type = smLibrary.cycleBasedStatemachineInterface
				]
			else if(typeSpecifier.type.name == lib.eventDriven.name)
				typeSpecifier = createTypeSpecifier => [ ts |
					ts.type = smLibrary.eventDrivenStatemachineInterface
				]
			else if(typeSpecifier.type.name == lib.statemachine.name)
				typeSpecifier = createTypeSpecifier => [ ts |
					ts.type = smLibrary.statemachineInterface
				]
		]
	}
	
	def substitueStateEnumMethodMember(Declaration method, ComplexType smClass) {
		
		method.eAllContents.filter(TypedDeclaration).filter[!(it instanceof Operation) && type !== null].toList.forEach [ source |
			val target = smClass.stmModelFor(source).eAllContents.filter(Declaration).getTarget(source.type) as Type
			if(target !== null && (target instanceof Enumerator && target.originSource instanceof Vertex)
				|| (target instanceof EnumerationType && source.type instanceof EnumerationType)
			){
				source.typeSpecifier = createTypeSpecifier => [ ts |
					ts.type = target
				]
			}
		]
		
		
		method.eAllContents.filter(Argument).map[value].filter(ArgumentExpression).toList.forEach [ source |
			val target = smClass.stmModelFor(source).eAllContents.filter(Declaration).getTarget(source)
			if(target !== null && target instanceof Enumerator && target.originSource instanceof Vertex){
				EcoreUtil.replace(source, target._ref)
			}
		]

		// Have to do this separately and in order to be able to replace both ref and owner
		method.eAllContents.filter(ElementReferenceExpression).toList.forEach [ source |
			val target = smClass.stmModelFor(source).eAllContents.filter(Declaration).getTarget(source)
			if(target !== null && target instanceof Enumerator && target.originSource instanceof Vertex){
				EcoreUtil.replace(source, target._ref)
			} else if(target instanceof EnumerationType && target.originSource instanceof Statechart){
				EcoreUtil.replace(source, target._ref)
			}
		]

		method.eAllContents.filter(FeatureCall).toList.forEach [ source |
			val target = smClass.stmModelFor(source).eAllContents.filter(Declaration).getTarget(source)
			if(target !== null && target instanceof Enumerator && (target.originSource instanceof Vertex || target.originSource instanceof Enumerator)){
				EcoreUtil.replace(source, target._ref)
			} else if(target instanceof EnumerationType && target.originSource instanceof Statechart){
				EcoreUtil.replace(source, target._ref)
			}
		]
	}
	
	def dispatch protected stmModelFor(ComplexType smClass, ArgumentExpression source){
		if(!smClass.features.filter[isStatechartType].nullOrEmpty){
			val referedStm = smClass.features.filter(ComplexType).filter[isStatechartType || isStatemachineClass].filter[
				
				features.filter(EnumerationType).head.enumerator.exists[e |
					e.sameEnumeratorAs(source.featureOrReference as NamedElement)
				]
			// Filter to get the proper stmType
			].filter[source.featureOrReference.eContainer.originSource === it.originSource].head
			if(referedStm !== null)
				return referedStm
		}
		return source.featureOrReference.stateMachineClass === null ? smClass : source.featureOrReference.stateMachineClass
	}
	
	def dispatch protected stmModelFor(ComplexType smClass, TypedDeclaration source){
		source.type.stateMachineClass === null ? smClass : source.type.stateMachineClass
	}
	
	def dispatch protected Declaration getTarget(Iterator<Declaration> content, ArgumentExpression source) {
		content.filter [ c |
			if(c instanceof ComplexType) c.eAllContents.filter(Declaration).getTarget(source)
			c.origin !== null && source.featureOrReference !== null && (c.originSource ===  source.featureOrReference.originSource || c.sameEnumeratorAs(source.featureOrReference))
		].head
	}
	
	def dispatch protected Declaration getTarget(Iterator<Declaration> content, Type source) {
		content.filter [ c |
			if(c instanceof ComplexType) c.eAllContents.filter(Declaration).getTarget(source)
			c.origin !== null && source !== null && c.originSource ===  source.originSource
		].head
	}
	
	def protected boolean sameEnumeratorAs(EObject target, EObject source){
		if(target instanceof NamedElement && target instanceof Enumerator && source instanceof NamedElement && source instanceof Enumerator){
			if((target as NamedElement).name === (source as NamedElement).name){
				return true
			} else if (target.origin instanceof Enumerator){
				return (target.origin as Enumerator).sameEnumeratorAs(source)
			} else
				return false
		} else
			return false
			
	}
	
}