/**
 * Copyright (c) 2025 itemis AG - All rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * 
 */
package com.yakindu.sct.json.transformation.json2xmi

import com.yakindu.base.types.adapter.ImplicitDefinitions
import com.yakindu.sct.json.transformation.model.JsonCell
import com.yakindu.sct.json.transformation.model.JsonChoice
import com.yakindu.sct.json.transformation.model.JsonEntry
import com.yakindu.sct.json.transformation.model.JsonExit
import com.yakindu.sct.json.transformation.model.JsonFinal
import com.yakindu.sct.json.transformation.model.JsonGraph
import com.yakindu.sct.json.transformation.model.JsonNote
import com.yakindu.sct.json.transformation.model.JsonRegion
import com.yakindu.sct.json.transformation.model.JsonRoot
import com.yakindu.sct.json.transformation.model.JsonState
import com.yakindu.sct.json.transformation.model.JsonStatechart
import com.yakindu.sct.json.transformation.model.JsonSynchronization
import com.yakindu.sct.json.transformation.model.JsonTransition
import com.yakindu.sct.json.transformation.model.util.NodeModelFactory
import com.yakindu.sct.json.transformation.model.util.NotationUtil
import com.yakindu.sct.model.sgraph.Choice
import com.yakindu.sct.model.sgraph.ChoiceKind
import com.yakindu.sct.model.sgraph.Entry
import com.yakindu.sct.model.sgraph.EntryKind
import com.yakindu.sct.model.sgraph.Exit
import com.yakindu.sct.model.sgraph.FinalState
import com.yakindu.sct.model.sgraph.Region
import com.yakindu.sct.model.sgraph.SGraphFactory
import com.yakindu.sct.model.sgraph.State
import com.yakindu.sct.model.sgraph.Statechart
import com.yakindu.sct.model.sgraph.Synchronization
import com.yakindu.sct.model.sgraph.Transition
import com.yakindu.sct.model.sgraph.Vertex
import java.util.HashMap
import java.util.HashSet
import java.util.Map
import java.util.Set
import org.eclipse.emf.common.util.URI
import org.eclipse.emf.ecore.EObject
import org.eclipse.emf.ecore.resource.Resource
import org.eclipse.emf.ecore.util.EcoreUtil
import org.eclipse.gmf.runtime.emf.core.resources.GMFResource
import org.eclipse.gmf.runtime.notation.Diagram
import org.eclipse.gmf.runtime.notation.MeasurementUnit
import org.eclipse.gmf.runtime.notation.Node
import org.eclipse.gmf.runtime.notation.NotationFactory
import org.eclipse.gmf.runtime.notation.View
import org.eclipse.gmf.runtime.notation.Routing
import com.yakindu.sct.json.transformation.model.JsonTransitionAnchor
import org.eclipse.gmf.runtime.notation.Edge

/**
 * 
 * @author andreas muelder - Initial contribution and API
 * @author axel terfloth - extensions
 * 
 */
class JSONToXMITransformation {

	extension SGraphFactory = SGraphFactory.eINSTANCE
	extension NotationFactory = NotationFactory.eINSTANCE
	extension NotationUtil = new NotationUtil
	extension NodeModelFactory = new NodeModelFactory
	extension ImplicitDefinitions = new ImplicitDefinitions

	static class Context {
		JsonGraph graph
		Map<EObject, String> idMap = new HashMap
		Map<EObject, JsonCell> jsonMap = new HashMap
		Map<String, Vertex> vertexMap = new HashMap
		Map<String, Node> nodeMap = new HashMap
		Set<JsonCell> implicitCells = new HashSet

		def protected useId(EObject it, JsonCell cell) {
			jsonMap.put(it, cell)
			if (! cell?.id.nullOrEmpty) {
				idMap.put(it, cell.id)
				if(it instanceof Vertex) vertexMap.put(cell.id, it)
			}
		}

		def <T> T json(EObject it, Class<T> clazz) {
			if(it === null) return null
			val json = jsonMap.get(it)
			if (json !== null && clazz.isInstance(json)) {
				return json as T
			}
		}
	}

	def Resource transform(JsonRoot root, boolean includeDiagram) {
		val extension context = new Context
		context.graph = root.graph

		val resource = root.transformRoot(context)
		if (includeDiagram) {
			resource.transformDiagram(context)
			root.graph.cells.filter(JsonNote).forEach[it.transformNote(resource.contents.filter(Diagram).head, context)]
		}
		resource
	}

	// --------------------------------------------
	// transformation of semantic model
	def protected Resource transformRoot(JsonRoot root, extension Context context) {
		val resource = new GMFResource(URI.createFileURI("dummy.sct"))

		val jsonStatechart = root.graph.findStatechart
		val statechart = transformElement(jsonStatechart, context)

		resource.contents += statechart
		resource.allContents.forEach[if(idMap.containsKey(it)) resource.setID(it, idMap.get(it))]

		resource
	}

	def dispatch protected EObject transformElement(JsonStatechart jsonStatechart, extension Context context) {

		// create region for vertices without region
		val verticesWithoutRegion = graph.findTopLevelVertices.filter[!(it instanceof JsonNote)]
		if (verticesWithoutRegion.size > 0) {
			val jsonRegion = new JsonRegion => [ r |
				r.id = EcoreUtil.generateUUID
				verticesWithoutRegion.forEach[parent = r.id]
			]
			implicitCells += jsonRegion
			graph.cells.add(graph.cells.indexOf(jsonStatechart) + 1, jsonRegion)
		}

		val statechart = createStatechart => [
			name = jsonStatechart.name
			specification = jsonStatechart.specification
			documentation = jsonStatechart.documentation
			regions += graph.findTopLevelRegions.map[transformElement(it, context)].filter(Region)
			useId(jsonStatechart)
		]

		// Order by priority before creating the transitions
		val transitions = graph.cells.filter(JsonTransition).groupBy[sourceId]
		transitions.forEach [ sourceId, jsonTransitions |
			jsonTransitions.sortBy[it.priority.priorityAsInt].forEach [
				it.transformElement(context)
			]
		]
		statechart
	}

	def int priorityAsInt(String it) {
		if (nullOrEmpty)
			return 0
		try {
			Integer.parseInt(it)
		} catch (Exception e) {
			return 0
		}
	}

	def dispatch protected EObject transformElement(JsonNote jsonState, extension Context context) {
	}

	def dispatch protected EObject transformElement(JsonState jsonState, extension Context context) {

		val state = createState => [
			name = jsonState.name
			specification = jsonState.specification
			documentation = jsonState.documentation
			useId(jsonState)
		]

		val children = graph.findChildren(jsonState).filter(JsonRegion)
		val regions = children.map [ child |
			transformElement(child, context)
		].filter(Region)

		state.regions += regions
		state
	}

	def dispatch protected EObject transformElement(JsonRegion jsonRegion, extension Context context) {

		val region = createRegion => [
			name = jsonRegion.name
			useId(jsonRegion)
			implicit = implicitCells.contains(jsonRegion)
		]

		val children = graph.findChildren(jsonRegion).filter[!(it instanceof JsonTransition)]
		val vertices = children.map[child|transformElement(child, context)].filter(Vertex)

		region.vertices += vertices
		region
	}

	def dispatch protected EObject transformElement(JsonEntry jsonEntry, extension Context context) {
		createEntry => [
			name = graph.findNodeLabel(jsonEntry)?.name ?: null
			kind = EntryKind.getByName(jsonEntry.entryKind.toFirstLower)
			useId(jsonEntry)
		]
	}

	def dispatch protected EObject transformElement(JsonExit jsonExit, extension Context context) {
		createExit => [
			name = graph.findNodeLabel(jsonExit).name
			useId(jsonExit)
		]
	}

	def dispatch protected EObject transformElement(JsonFinal jsonFinal, extension Context context) {
		createFinalState => [
			name = jsonFinal.name
			useId(jsonFinal)
		]
	}

	def dispatch protected EObject transformElement(JsonChoice jsonChoice, extension Context context) {
		createChoice => [
			name = jsonChoice.name
			kind = ChoiceKind.DYNAMIC
			useId(jsonChoice)
		]
	}

	def dispatch protected EObject transformElement(JsonSynchronization jsonSync, extension Context context) {
		createSynchronization => [
			useId(jsonSync)
		]
	}

	def dispatch protected EObject transformElement(JsonTransition jsonTransition, extension Context context) {
		createTransition => [
			specification = jsonTransition.specification
			documentation = jsonTransition.documentation
			source = vertexMap.get(jsonTransition.sourceId)
			target = vertexMap.get(jsonTransition.targetId)
			useId(jsonTransition)
		]
	}

	// --------------------------------------------
	// transformation of notation model
	def protected void transformDiagram(Resource resource, extension Context context) {

		val statechart = resource.contents.filter(Statechart).head

		if (statechart !== null) {
			val diagram = createDiagram => [
				measurementUnit = MeasurementUnit.PIXEL_LITERAL
				type = "itemis.create.ui.editor.editor.StatechartDiagramEditor"
				styles += createBooleanValueStyle => [
					name = "inlineDefinitionSection"
				]
				element = statechart
			]
			resource.contents += diagram

			statechart.transformView(diagram, context)
		}
	}

	def dispatch protected void transformView(Statechart statechart, View container, extension Context context) {
		statechart.regions.forEach[transformView(container, context)]
		statechart.eAllContents.filter(Transition).forEach[transformView(container, context)]
	}

	def dispatch protected void transformView(Region region, View container, extension Context context) {
		val jsonRegion = region.json(JsonRegion)
		val shape = region.createRegionNode
		shape.layoutConstraint = graph.toRelativeBounds(jsonRegion) => [
			width = width
			height = height + 20 // offset for name label
		]
		container.insertChild(shape)
		val compartmentNode = shape.children.filter(View).findFirst[type == "RegionCompartment"]
		shape.styles += createShapeStyle => [
			fillColor = jsonRegion.fillColor?.toShapeColor ?: 16448250
			lineColor = jsonRegion.strokeColor?.toShapeColor ?: 12632256
		]
		region.vertices.forEach[v|transformView(v, compartmentNode, context)]
	}

	def dispatch protected void transformView(State state, View container, extension Context context) {
		val jsonState = state.json(JsonState)
		val shape = state.createStateNode
		shape.layoutConstraint = graph.toRelativeBounds(jsonState)
		nodeMap.put(jsonState.id, shape)
		container.insertChild(shape)
		// Region Layout
		shape.styles += createBooleanValueStyle => [
			name = "isHorizontal"
			booleanValue = jsonState.regionLayout === null || jsonState.regionLayout == "horizontal" ? true : false
		]
		// Show Documentation
		shape.styles += createStringValueStyle => [
			name = "featureToShow"
			stringValue = jsonState.showSpecification ? "specification" : "documentation"
		]
		shape.styles += createShapeStyle => [
			fillColor = jsonState.fillColor?.toShapeColor ?: 15720400
			lineColor = jsonState.strokeColor?.toShapeColor ?: 12632256
		]
		val compartmentNode = shape.children.filter(View).findFirst[type == "StateFigureCompartment"]

		state.regions.forEach[transformView(compartmentNode, context)]
	}

	def dispatch protected void transformView(Entry entry, View container, extension Context context) {
		val jsonEntry = entry.json(JsonEntry)
		val entryShape = entry.createEntryNode
		entryShape.layoutConstraint = graph.toRelativeBounds(jsonEntry)
		nodeMap.put(jsonEntry.id, entryShape)
		container.insertChild(entryShape)

	}

	def dispatch protected void transformView(Exit exit, View container, extension Context context) {
		val jsonExit = exit.json(JsonExit)
		val exitShape = exit.createExitNode
		exitShape.layoutConstraint = graph.toRelativeBounds(jsonExit)

		nodeMap.put(jsonExit.id, exitShape)

		container.insertChild(exitShape)
	}

	def dispatch protected void transformView(FinalState finalState, View container, extension Context context) {
		val jsonFinal = finalState.json(JsonFinal)
		val finalShape = finalState.createFinalNode
		finalShape.layoutConstraint = graph.toRelativeBounds(jsonFinal)
		nodeMap.put(jsonFinal.id, finalShape)
		container.insertChild(finalShape)
	}

	def dispatch protected void transformView(Choice choice, View container, extension Context context) {
		val jsonChoice = choice.json(JsonChoice)
		val choiceShape = choice.createChoiceNode
		choiceShape.layoutConstraint = graph.toRelativeBounds(jsonChoice)

		nodeMap.put(jsonChoice.id, choiceShape)

		container.insertChild(choiceShape)
	}

	def dispatch protected void transformView(Synchronization sync, View container, extension Context context) {
		val jsonSync = sync.json(JsonSynchronization)
		vertexMap.put(jsonSync.id, sync)
		val syncShape = sync.createSynchNode
		syncShape.layoutConstraint = graph.toRelativeBounds(jsonSync)
		nodeMap.put(jsonSync.id, syncShape)
		container.insertChild(syncShape)
	}

	def dispatch protected void transformView(Transition transition, View container, extension Context context) {

		val jsonTransition = transition.json(JsonTransition)
		val edge = transition.createTransitionEdge

		// Show Documentation
		edge.styles += createStringValueStyle => [
			name = "featureToShow"
			stringValue = jsonTransition.showSpecification ? "specification" : "documentation"
		]

		edge.styles += createConnectorStyle => [
			routing == if(jsonTransition.router?.name == "orthogonal") Routing.RECTILINEAR_LITERAL else null
		]

		edge.source = nodeMap.get(jsonTransition.sourceId)
		edge.target = nodeMap.get(jsonTransition.targetId)
		annotateLayoutInfo(jsonTransition, edge)

		edge.insertChild(createDecorationNode => [
			type = "TransitionExpression"
			layoutConstraint = createLabelPosition(jsonTransition)
		])

		(container as Diagram).insertEdge(edge)
	}

	/**
	 * Since the router in GMF and jointjs work differently, we can not transform bendpoints and anchors points 1:1.
	 * We store the jointjs information 
	 */
	def annotateLayoutInfo(JsonTransition transition, Edge edge) {
		// Bendpoints
		if (!transition.vertices.isNullOrEmpty) {
			val vertices = transition.vertices.map[toString].join(",")
			edge.styles += createStringValueStyle => [
				name = "joint:vertices"
				stringValue = vertices
			]

		}
		// Source anchor position
		val source = transition.source.anchor
		if (source?.args?.get("dx") !== null && source?.args?.get("dy") !== null) {
			edge.styles += createStringValueStyle => [
				name = "joint:source:anchor"
				stringValue = '''«source.args.get("dx")»;«source.args.get("dy")»'''.toString
			]
		}
		// Target anchor position
		val target = transition.target.anchor
		if (target?.args?.get("dx") !== null && target?.args?.get("dy") !== null) {
			edge.styles += createStringValueStyle => [
				name = "joint:target:anchor"
				stringValue = '''«target.args.get("dx")»;«target.args.get("dy")»'''.toString
			]
		}
		// Label Position:
		if (transition.labelDistance !== null && transition.labelOffset !== null) {
			edge.styles += createStringValueStyle => [
				name = "joint:label:distance"
				stringValue = "" + transition.labelDistance
			]
			edge.styles += createStringValueStyle => [
				name = "joint:label:offset"
				stringValue = "" + transition.labelOffset
			]
		}
	}

	def createAnchor(JsonTransitionAnchor anchor, Edge edge) {
		edge.styles += createStringValueStyle => [
			name = "jointJsAnchors"
			stringValue = "" + anchor.args.get("dx") + anchor.args.get("dy")
		]
	}

	def protected void transformNote(JsonNote note, Diagram diagram, extension Context context) {
		diagram.insertChild(createNoteShape => [
			description = note.name
			layoutConstraint = graph.toRelativeBounds(note)

		])
	}

}
