/**
 * Copyright (c) 2025 itemis AG - All rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * 
 */
package com.yakindu.sct.json.transformation.model

import java.util.ArrayList
import java.util.HashMap
import java.util.List
import java.util.Map
import org.eclipse.xtend.lib.annotations.Accessors
import org.eclipse.xtend.lib.annotations.ToString

/**
 * 
 * @author andreas muelder - Initial contribution and API
 * 
 */
@Accessors
@ToString
class JsonRoot {
	JsonGraph graph
}

@Accessors
@ToString
class JsonGraph {

	List<JsonCell> cells = newArrayList;

	def List<JsonRegion> findTopLevelRegions() {
		cells.filter(JsonRegion).filter[parent.nullOrEmpty].toList
	}

	def List<JsonCell> findTopLevelVertices() {
		cells.filter(JsonCell).filter[parent.nullOrEmpty].filter[!(it instanceof JsonStatechart)].filter [
			!(it instanceof JsonRegion)
		].filter[!(it instanceof JsonTransition)].toList
	}

	def List<JsonCell> findChildren(JsonCell parentCell) {
		cells.filter[parent == parentCell.id].toList
	}

	def JsonCell findParent(JsonCell cell) {
		cells.findFirst [
			it.id == cell.parent
		]
	}

	def JsonNodeLabel findNodeLabel(JsonEntry cell) {
		cells.findFirst[it.id == cell.embeds.head] as JsonNodeLabel
	}

	def JsonNodeLabel findNodeLabel(JsonExit cell) {
		cells.findFirst[it.id == cell.embeds.head] as JsonNodeLabel
	}

	def findStatechart() {
		cells.filter(JsonStatechart).head
	}

	def List<JsonCell> findContainerHierachy(JsonCell cell, List<JsonCell> container) {
		val parent = findParent(cell)
		if (parent !== null) {
			container += parent
			return findContainerHierachy(parent, container)
		}
		return container
	}

	def findOutgoingTransitions(JsonCell cell) {
		cells.filter(JsonTransition).filter[sourceId == cell.id]
	}
}

@Accessors
class JsonPosition {
	public static val EMPTY = new JsonPosition => [
		x = 0
		y = 0
	]
	double x
	double y

	def shift(JsonPosition shift) {
		this.x += shift.x
		this.y += shift.y
	}

	override toString() {
		'''(«x»;«y»)'''.toString
	}

	def static JsonPosition fromString(String str) {
		new JsonPosition => [
			val cleaned = str.trim.replace("(", "").replace(")", "")
			val parts = cleaned.split(";")
			x = Double.parseDouble(parts.get(0))
			y = Double.parseDouble(parts.get(1))
		]
	}

}

@Accessors
@ToString
class JsonSize {
	public static val EMPTY = new JsonSize => [
		width = 0
		height = 0
	]
	double width
	double height
}

@Accessors
@ToString
class JsonCell {
	JsonPosition position
	Map<String, Object> attrs = newHashMap
	JsonSize size
	String id
	String parent
	int z

	// Getters & Setters for Values that are part of attrs array for convenience
	def String getName() {
		(attrs?.get("name") as Map<String, Object>)?.get("text") as String
	}

	def void setName(String value) {
		var nameMap = (ensureAttrs().get("name") as Map<String, Object>) ?: newHashMap
		nameMap.put("text", value)
		ensureAttrs().put("name", nameMap)
	}

	def String getFillColor() {
		(attrs?.get("body") as Map<String, Object>)?.get("fill") as String
	}

	def void setFillColor(String hexColor) {
		var bodyMap = (ensureAttrs().get("body") as Map<String, Object>) ?: newHashMap
		bodyMap.put("fill", hexColor)
		ensureAttrs().put("body", bodyMap)
	}

	def String getStrokeColor() {
		(attrs?.get("body") as Map<String, Object>)?.get("stroke") as String
	}

	def void setStrokeColor(String hexColor) {
		var bodyMap = (ensureAttrs().get("body") as Map<String, Object>) ?: newHashMap
		bodyMap.put("stroke", hexColor)
		ensureAttrs().put("body", bodyMap)
	}

	def String getSpecification() {
		(attrs?.get("specification") as Map<String, Object>)?.get("text") as String
	}

	def void setSpecification(String value) {
		var specMap = (ensureAttrs().get("specification") as Map<String, Object>) ?: newHashMap
		specMap.put("text", value)
		ensureAttrs().put("specification", specMap)
	}

	def String getDocumentation() {
		(attrs?.get("documentation") as Map<String, Object>)?.get("text") as String
	}

	def void setDocumentation(String value) {
		var docMap = (ensureAttrs().get("documentation") as Map<String, Object>) ?: newHashMap
		docMap.put("text", value)
		ensureAttrs().put("documentation", docMap)
	}

	def protected Map<String, Object> ensureAttrs() {
		if(attrs === null) attrs = newHashMap
		attrs
	}
}

@Accessors
@ToString
class JsonNote extends JsonNodeLabel {
}

@Accessors
@ToString
class JsonNodeLabel extends JsonCell {
	// Getters & Setters for Values that are part of attrs array for convenience
	override String getName() {
		(attrs?.get("label") as Map<String, Object>)?.get("text") as String
	}

	override void setName(String value) {
		if(attrs === null) attrs = newHashMap
		var nameMap = attrs.get("label") as Map<String, Object>
		if (nameMap === null) {
			nameMap = newHashMap
			attrs.put("label", nameMap)
		}
		nameMap.put("text", value)
	}
}

@Accessors
@ToString
class JsonStatechart extends JsonCell {
}

@Accessors
@ToString
class JsonSpecificationElement extends JsonCell {
	def boolean getShowSpecification() {
		(attrs?.get("specification") as Map<String, Object>)?.get("show") as Boolean ?: true
	}

	def void setShowSpecification(boolean value) {
		if(attrs === null) attrs = newHashMap
		var nameMap = attrs.get("specification") as Map<String, Object>
		if (nameMap === null) {
			nameMap = newHashMap
			attrs.put("specification", nameMap)
		}
		nameMap.put("show", value)
	}
}

@Accessors
@ToString
class JsonState extends JsonSpecificationElement {
	List<String> embeds = newArrayList;

	def String getRegionLayout() {
		(attrs?.get("layout") as Map<String, Object>)?.get("region") as String
	}

	def void setRegionLayout(String value) {
		if(attrs === null) attrs = newHashMap
		var nameMap = attrs.get("layout") as Map<String, Object>
		if (nameMap === null) {
			nameMap = newHashMap
			attrs.put("layout", nameMap)
		}
		nameMap.put("region", value)
	}
}

@Accessors
@ToString
class JsonRegion extends JsonCell {
	List<String> embeds = newArrayList;
}

@Accessors
@ToString
class JsonEntry extends JsonCell {
	String entryKind
	List<String> embeds = newArrayList;
}

@Accessors
@ToString
class JsonExit extends JsonCell {
	List<String> embeds = newArrayList;
}

@Accessors
@ToString
class JsonFinal extends JsonCell {
}

@Accessors
@ToString
class JsonChoice extends JsonCell {
}

@Accessors
@ToString
class JsonSynchronization extends JsonCell {
}

@Accessors
@ToString
class JsonTransition extends JsonSpecificationElement {
	JsonTransitionSource source
	JsonTransitionTarget target
	JsonTransitionRouter router
	List<Map<String, Object>> labels = new ArrayList
	List<JsonPosition> vertices = newArrayList

	// Getters & Setters for Values that are part of attrs array for convenience
	override String getSpecification() {
		val labels = labels.head
		val map = labels?.get("attrs") as Map<String, Object>
		val innerText = map?.get("text") as Map<String, Object>
		innerText?.get("text") as String
	}

	override void setSpecification(String value) {
		if (labels.size == 0)
			labels.add(new HashMap)
		val label = labels.head
		var map = label.get("attrs") as Map<String, Object>
		if (map === null) {
			map = newHashMap
			label.put("attrs", map)
		}
		var innerText = map.get("text") as Map<String, Object>
		if (innerText === null) {
			innerText = newHashMap
			map.put("text", innerText)
		}
		innerText?.put("text", value)
	}

	def Double getLabelDistance() {
		val labels = labels.head
		val map = labels?.get("position") as Map<String, Object>
		map?.get("distance") as Double
	}

	def void setLabelDistance(Double value) {
		if (labels.size == 0)
			labels.add(new HashMap)
		val label = labels.head
		var map = label.get("position") as Map<String, Object>
		if (map === null) {
			map = newHashMap
			label.put("position", map)
		}
		map.put("distance", value)
	}

	def void setLabelOffset(String value) {
		if (labels.size == 0)
			labels.add(new HashMap)
		val label = labels.head
		var map = label.get("position") as Map<String, Object>
		if (map === null) {
			map = newHashMap
			label.put("position", map)
		}
		val offset = value.split(";")
		if (offset.length == 2) {
			val offsetAsPoint = newHashMap
			offsetAsPoint.put("x", Double.parseDouble(offset.get(0)))
			offsetAsPoint.put("y", Double.parseDouble(offset.get(1)))
			map.put("offset", offsetAsPoint)
		} else
			map.put("offset", Double.parseDouble(value))

	}

	def String getLabelOffset() {
		val labels = labels.head
		val map = labels?.get("position") as Map<String, Object>
		val offset = map?.get("offset")
		if (offset instanceof Map) {
			return offset.get("x") + ";" + offset.get("y")
		}
		return "" + offset
	}

	def String getPriority() {
		if (labels.size < 2)
			return null
		val labels = labels.get(1)
		val map = labels?.get("attrs") as Map<String, Object>
		val innerText = map?.get("label") as Map<String, Object>
		innerText?.get("text") as String
	}

	def void setPriority(String value) {
		while (labels.size < 2) {
			labels.add(new HashMap)
		}
		val label = labels.get(1)
		var map = label.get("attrs") as Map<String, Object>
		if (map === null) {
			map = newHashMap
			label.put("attrs", map)
		}
		var innerText = map.get("label") as Map<String, Object>
		if (innerText === null) {
			innerText = newHashMap
			map.put("label", innerText)
		}
		innerText?.put("text", value)
	}

	override String getDocumentation() {
		if (labels.size < 3)
			return null
		val labels = labels.get(3)
		val map = labels?.get("attrs") as Map<String, Object>
		val innerText = map?.get("text") as Map<String, Object>
		innerText?.get("text") as String
	}

	override void setDocumentation(String value) {
		while (labels.size < 4) {
			labels.add(new HashMap)
		}
		val label = labels.get(3)
		var map = label.get("attrs") as Map<String, Object>
		if (map === null) {
			map = newHashMap
			label.put("attrs", map)
		}
		var innerText = map.get("text") as Map<String, Object>
		if (innerText === null) {
			innerText = newHashMap
			map.put("text", innerText)
		}
		innerText?.put("text", value)
	}

	def String getSourceId() {
		source?.id
	}

	def String getTargetId() {
		target.id
	}
}

@Accessors
@ToString
class JsonTransitionRouter {
	String name

	new() {
	}

	new(String style) {
		this.name = style
	}
}

@Accessors
@ToString
class JsonTransitionSource {
	String id
	JsonTransitionAnchor anchor
}

@Accessors
@ToString
class JsonTransitionTarget {
	String id
	JsonTransitionAnchor anchor
}

@Accessors
@ToString
class JsonTransitionAnchor {
	String name
	Map<String, Object> args = new HashMap
}
