/**
 * Copyright (c) 2025 itemis AG - All rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * 
 */
package com.yakindu.sct.json.transformation.model.util

import com.yakindu.sct.json.transformation.model.JsonCell
import com.yakindu.sct.json.transformation.model.JsonGraph
import com.yakindu.sct.json.transformation.model.JsonNote
import com.yakindu.sct.json.transformation.model.JsonPosition
import com.yakindu.sct.json.transformation.model.JsonSize
import com.yakindu.sct.json.transformation.model.JsonState
import com.yakindu.sct.json.transformation.model.JsonSynchronization
import com.yakindu.sct.json.transformation.model.JsonTransition
import com.yakindu.sct.json.transformation.model.JsonTransitionAnchor
import java.util.List
import org.eclipse.emf.ecore.EObject
import org.eclipse.emf.ecore.util.EcoreUtil
import org.eclipse.gmf.runtime.notation.Bounds
import org.eclipse.gmf.runtime.notation.Diagram
import org.eclipse.gmf.runtime.notation.Edge
import org.eclipse.gmf.runtime.notation.Location
import org.eclipse.gmf.runtime.notation.Node
import org.eclipse.gmf.runtime.notation.NotationFactory
import org.eclipse.gmf.runtime.notation.RelativeBendpoints
import org.eclipse.gmf.runtime.notation.datatype.RelativeBendpoint
import com.yakindu.sct.json.transformation.model.JsonRegion

/**
 * 
 * @author andreas muelder - Initial contribution and API
 * @author axel terfloth - Fixes
 * 
 */
class NotationUtil {

	extension NotationFactory = NotationFactory.eINSTANCE

	def toRelativeBounds(JsonGraph graph, JsonCell cell) {

		val offset = graph.findParent(cell)?.position ?: new JsonPosition()
		createBounds => [
			if (cell.position !== null) {
				x = cell.position.x as int - offset.x as int
				y = cell.position.y as int - offset.y as int - 20
			}
			// cell.size equals null -> use the default size
			if (cell.size !== null) {
				width = cell.size.width as int
				height = cell.size.height as int
			}
		]
	}

	def setBounds(JsonCell cell, Node view, JsonGraph graph) {
		if (view?.layoutConstraint === null || !(view.layoutConstraint instanceof Bounds))
			return;
		val notationBounds = view.layoutConstraint as Bounds
		cell.position = new JsonPosition => [
			x = notationBounds.x
			y = notationBounds.y
		]
		val parent = graph.findParent(cell)
		val parentPosition = parent?.position ?: JsonPosition.EMPTY

		cell.position.shift(parentPosition)
		// Add Gap for Region name label
		if (parent instanceof JsonRegion) {
			cell.position.y = cell.position.y + 20
		}
		cell.size = new JsonSize => [
			width = notationBounds.width < 0 ? cell.defaultSize.width : notationBounds.width
			height = notationBounds.height < 0 ? cell.defaultSize.height : notationBounds.height
		]
	}

	def getDefaultSize(JsonCell cell) {

		return switch (cell) {
			// Web Editor can handle default sizes for states and regions
			JsonState: new JsonSize => [width = -1 height = -1]
			JsonRegion: new JsonSize => [width = -1 height = -1]
			JsonSynchronization: new JsonSize => [width = 8 height = 32]
			JsonNote: new JsonSize => [width = 100 height = 50]
			default: new JsonSize => [width = 15 height = 15]
		}

	}

	def Node findNotationView(EObject semanticElement, Diagram diagram, String type) {
		if(diagram === null) return null
		diagram?.eAllContents().filter(Node).findFirst[it.type == type && EcoreUtil.equals(element, semanticElement)];
	}

	def Edge findNotationEdge(EObject semanticElement, Diagram diagram, String type) {
		if(diagram === null) return null
		diagram?.eAllContents().filter(Edge).findFirst[it.type == type && EcoreUtil.equals(element, semanticElement)];
	}

	def List<JsonPosition> transformRelativeBendpoints(Edge edge, JsonCell source) {
		if (edge === null)
			return newArrayList

		if (!(edge.bendpoints instanceof RelativeBendpoints))
			return newArrayList

		val bendpoints = (edge.bendpoints as RelativeBendpoints).points.filter(RelativeBendpoint).toList
		if (bendpoints.size < 2)
			return newArrayList
		bendpoints.subList(1, bendpoints.size - 1).map [ rp |
			new JsonPosition => [
				x = source.position.x + rp.sourceX
				y = source.position.y + rp.sourceY
			]
		].toList
	}

	def Location createLabelPosition(JsonTransition transition) {
		createLocation => [
			x = 10
			y = 10
		]
	}

	def toHexColor(int shapeColor) {
		val r = shapeColor % 256
		val g = (shapeColor / 256) % 256
		val b = (shapeColor / 65536) % 256
		String::format("#%02X%02X%02X", r, g, b)
	}

	def Integer toShapeColor(String hex) {
		val clean = hex.replace("#", "")
		val r = Integer::parseInt(clean.substring(0, 2), 16)
		val g = Integer::parseInt(clean.substring(2, 4), 16)
		val b = Integer::parseInt(clean.substring(4, 6), 16)

		return b * 65536 + g * 256 + r
	}

}
