/**
 * Copyright (c) 2022-2025 itemis AG - All rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * 
 * Contributors:
 * 	Finlay Weegen - itemis AG
 * 
 */

package com.yakindu.sctunit.coverage.report

import com.yakindu.base.types.Property
import com.yakindu.sct.model.sgraph.Choice
import com.yakindu.sct.model.sgraph.FinalState
import com.yakindu.sct.model.sgraph.Region
import com.yakindu.sct.model.sgraph.State
import com.yakindu.sct.model.sgraph.Statechart
import com.yakindu.sct.model.sgraph.Synchronization
import com.yakindu.sct.model.sgraph.Transition
import com.yakindu.sct.simulation.core.coverage.Measurement
import com.yakindu.sct.simulation.core.coverage.MeasurementExtension
import com.yakindu.sctunit.coverage.report.CoverageReportAdapterFactoryLabelProvider
import com.yakindu.sctunit.coverage.report.ReportElement
import com.yakindu.sctunit.sCTUnit.SCTUnitClass
import java.util.ArrayList
import java.util.Date
import org.eclipse.emf.edit.provider.ComposedAdapterFactory
import java.math.BigDecimal
import java.math.MathContext
import java.math.RoundingMode
import com.yakindu.sctunit.sCTUnit.SCTUnitSuite
import org.eclipse.emf.ecore.EObject

class HTMLCoverageReport implements ICoverageReport {

	extension MeasurementExtension mExtension
	extension ReportExtensions rExtension
	
	val composedAdapterFactory = new ComposedAdapterFactory(ComposedAdapterFactory.Descriptor.Registry.INSTANCE)
	val labelProvider = new CoverageReportAdapterFactoryLabelProvider(composedAdapterFactory)
	val Measurement measurement
	var statechartList = new ArrayList<Statechart>
	var testCases = new ArrayList<Measurement>
	var avgTotalCoverage = new BigDecimal(0.0)
	var statechartCounter = 0
	var idCounter = 0
	var iconCounter = 0
	var int treeTableCounter = 0
	var int numberOfImages
	var ArrayList<ReportElement> reportElements
	var int numberOfTestCases

	new(Measurement measurement) {
		this.measurement = measurement
		this.mExtension = new MeasurementExtension()
		this.rExtension = new ReportExtensions
	}

	override String generate(ArrayList<ReportElement> reportElements) {
		this.numberOfImages = numberOfImages
		this.reportElements = reportElements
		this.numberOfTestCases = measurement.testCaseCount.testCases;
		populateStatechartSet(measurement)
		avgTotalCoverage = measurement.coveragePercent
		toHtml(testCases).toString
	}

	def toHtml(Object[] testCases) '''
		<!DOCTYPE html>
		<head>
		    <meta charset="UTF-8" />
		    <meta http-equiv="X-UA-Compatible" content="IE=edge"/>
		    <meta name="viewport" content="width=device-width, initial-scale=1.0"/>
		    <script type="text/javascript" src="./js/jquery-1.10.2.min.js"></script>
		    <script type="text/javascript" src="./js/bootstrap.bundle.min.js"></script>
		    <link rel="stylesheet" href="./css/jquery.treegrid.css">
		    <script type="text/javascript" src="./js/jquery.treegrid.js"></script>
		    <link rel="stylesheet" href="./css/bootstrap.min.css">
		    <link rel="stylesheet" href="./css/custom-styles.css">
		    <script src="./js/stylesheetToggle.js"></script>
			<link id="stylesheet" rel="stylesheet" href="./css/custom-styles-color.css">
			<title>Coverage Report</title>
			<script>
				
				let currentStyle = "color";
				var stylesheet = document.getElementById("stylesheet")
		
				function toggleStylesheet() {
					console.log(stylesheet)
					let progressBars = document.getElementsByClassName("progress-bar");
					if (currentStyle == "color") {
						stylesheet.setAttribute("href", "./css/custom-styles-grey.css")
						Array.from(progressBars).forEach(bar => bar.classList.replace("bg-success", "bg-secondary"));
						currentStyle = "grey";
					} else if (currentStyle == "grey") {
						stylesheet.href = "./css/custom-styles-color.css";
						Array.from(progressBars).forEach(bar => bar.classList.replace("bg-secondary", "bg-success"));
						currentStyle = "color";
					}
				}
			</script>
		
		<body class="p-4 summary container mw-50">
		    <div class="d-flex flex-row justify-content-between align-middle">
				<div class="d-flex align-items-center">
					<p class="display-6 mb-0">Statechart Test Coverage Report<br/>	
					for «measurement.subject.subjectType» '«measurement.subjectID»'</p>	
				</div>
				<div id="logo"></div>
					    </div>
				  <hr>
				  <span style="float: right; color: blue; text-decoration: underline;" onclick="toggleStylesheet()">Toggle Greyscale</span>
				  <div>
				      <h2 class="pb-2">Summary</h2>

				      <p>Report created: «new Date()»</p>
				      <p>Coverage measured: «new Date(measurement.timestamp)»</p>
				      <p>Total test cases: «numberOfTestCases»</p>
				      <div>
				      	<p>Overall report coverage:</p>
				      	<div class="progress w-25">
				      	        <div class="progress-bar progress-bar-striped bg-success" role="progressbar" style="width: «avgTotalCoverage»%" aria-valuenow="50"
				      	          aria-valuemin="0" aria-valuemax="100">«avgTotalCoverage.toPlainString»%</div>
				      	      </div>
				      </div>
				      
				  </div>
				  <br>
				  <p>A total of «statechartList.toSet.length < 2 ? "1 statechart was" : statechartList.toSet.length + " statecharts were"» tested:</p>
				  <br>
				  <br>
				  <div>
				  	«FOR reportElement : reportElements»
				  		<h4>«reportElement.measurement.parent.name»: «reportElement.measurement.name»</h4>
				  		«statechartStatisticsToHtml((reportElement.measurement.subject as Statechart), reportElement.measurement, testCases, statechartCounter)»
				  		<table class="tree«treeTableCounter» table">
				  			«toHtmlRows(reportElement.measurement, 0, true, idCounter)»
				  			«incrementTreeTableCounter»
				  		</table>
				  		«outputImages(reportElement)»
				  		 <br>
				  		 <hr>
				  		 <br>
				  	«ENDFOR»
				  </div>
		«generateTreegridScripts»
		</body>
		</html>
	'''

	def statechartStatisticsToHtml(Statechart it, Measurement measurement, Object testCases, int i) {
		var statistics = '''
			<br>
			<p>Number of States: «it.allStates.length»</p>
			<p>Number of Transitions: «it.allTransitions.length»</p>
			<p>Overall statechart coverage: «measurement.valueText»</p>
		'''
		statechartCounter++
		statistics
	}

	def String toHtmlRows(Measurement it, int i, boolean firstRow, int next) {
		idCounter++
		iconCounter++
		var tempCounter = idCounter
		var percent = coveragePercent.intValue
		return '''
			<tr style="margin-right:40px;" class="treegrid-«idCounter» «(firstRow === true ? "" : ("treegrid-parent-" + next))»">
				<td>«it.itemDescription»</td>
				<td style="width:25%">
					<div class="progress" style="width: 100%; float: right;">
					    <div class="progress-bar progress-bar-striped bg-success" role="progressbar" style="width: «percent»%" aria-valuenow="50"
					      aria-valuemin="0" aria-valuemax="100">«it.valueText»</div>
					               <div class="progress-bar" role="progressbar" style="width:«(100 - percent)»%; «(percent < 1 ? "color:#000;")»" aria-valuenow="50"
					                 aria-valuemin="0" aria-valuemax="100">«(percent < 1 ? it.valueText : "")»</div>
					  </div>	
				</td>
			</tr>
				«IF it.hasChildren»
					«FOR child : filterChildren(it)»
						«toHtmlRows(child,i+1,false,tempCounter)»
					«ENDFOR»
				«ENDIF»
		'''
	}

	def outputImages(ReportElement reportElement) '''
		«FOR i : 0..reportElement.images.size»
		<div>
			<img src="./img/«reportElement.imageName(i)»">
		</div>
		«ENDFOR»
	'''

	def String generateTreegridScripts() '''
		«FOR i : 0..statechartList.length»
			<script type="text/javascript">
				$(document).ready(function () {
					$('.tree«i»').treegrid();
				});
			</script>
		«ENDFOR»
	'''

	def void incrementTreeTableCounter() {
		treeTableCounter++
	}

	def filterChildren(Measurement it) {
		return it.children.filter [
			it.subject instanceof SCTUnitClass || subject instanceof Property || subject instanceof State ||
				subject instanceof Transition || subject instanceof Statechart || subject instanceof Region ||
				subject instanceof Choice || subject instanceof FinalState || subject instanceof Synchronization
		].toList
	}

	def dispatch element(Object it) { null }

	def dispatch element(Measurement it) { subject }


	def coveragePercent(Measurement it) {
		new BigDecimal(it.coverage.coverage * 100.0).round(new MathContext(5, RoundingMode.HALF_UP))
	}
	
	def valueText(Measurement it) {
		var cov = coverage(it)
		coveragePercent.toPlainString + "% (" + cov.weight + ")"
	}

	
	def itemDescription(Object it) {
		labelProvider.getText(it)
	}

	def getAllTransitions(Statechart statechart) {
		statechart.eAllContents.filter(Transition).toList
	}

	def getAllStates(Statechart statechart) {
		statechart.eAllContents.filter(State).toList
	}

	def void populateStatechartSet(Measurement measurement) {
		if (measurement === null) {
			return;
		}
		if (measurement.subject instanceof Statechart) {
			statechartList.add(measurement.subject as Statechart)
		}
		if (measurement.getChildren() !== null) {
			for (Measurement child : measurement.getChildren()) {
				populateStatechartSet(child);
			}
		}
	}
	
	def dispatch subjectType(SCTUnitClass it) '''Test Class'''
	def dispatch subjectType(SCTUnitSuite it) '''Test Suite'''
	def dispatch subjectType(EObject it) '''«eClass.name»'''
	def dispatch subjectType(Object it) '''«class.simpleName»'''
	def dispatch subjectType(Void it) ''''''
	
}

