/**
 * Copyright (c) 2024 itemis AG - All rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * 
 */
package com.yakindu.base.types.scoping

import com.yakindu.base.types.Type
import com.yakindu.base.types.libraries.ITypeLibraryProvider
import java.util.Set
import org.eclipse.emf.common.util.URI
import org.eclipse.emf.ecore.EClass
import org.eclipse.emf.ecore.EObject
import org.eclipse.emf.ecore.EcorePackage
import org.eclipse.xtext.scoping.IScope
import org.eclipse.xtext.scoping.Scopes
import org.eclipse.xtext.scoping.impl.AbstractScope

import static extension com.yakindu.base.types.TypesUtil.*
import com.yakindu.base.types.Declaration
import com.yakindu.base.types.AnnotationType

/**
 * This scope provides all description for declarations from type libraries which are identified by the configured of library URLs.
 * 
 * @author axel terfloth - Initial contribution
 */

class TypeLibraryScope extends AbstractScope {
	
	protected ITypeLibraryProvider libraryProvider
	protected Set<URI> importedURIs
		
	new(IScope outer, ITypeLibraryProvider context, Set<URI> importedLibs) {
		super(outer, false)
		this.libraryProvider = context
		this.importedURIs = importedLibs
		
		if (importedURIs === null) importedURIs = #{}
	}
	
	def protected getImportedDeclarations() {
		val importedLibraries = libraryProvider
			.provide
			.filter[ importedURIs.contains(libraryURI) ]
			.map   [ libraryPackage ]
			.toList
		
		importedLibraries
			.map[ eAllContents.toList.filter(Declaration).toList ]
			.flatten
	}
	
	override protected getAllLocalElements() {
		Scopes.scopedElementsFor(importedDeclarations)
	}
	
	/** A specialized TypeLibraryScope which provides access to all types which are subtypes of the specified emf meta type which is EObject by default. */
	static class TypeScope extends TypeLibraryScope {

		protected EClass type
		
		new(IScope outer, ITypeLibraryProvider context, Set<URI> importedLibs) {
			this(outer, context, importedLibs, EcorePackage.Literals.EOBJECT)
		}

		new(IScope outer, ITypeLibraryProvider context, Set<URI> importedLibs, EClass type) {
			super(outer, context, importedLibs)
			this.type = type
			if (this.type === null ) this.type = EcorePackage.Literals.EOBJECT
		}
		
		override protected getAllLocalElements() {
				val importedTypes = importedDeclarations.filter(Type)
				Scopes.scopedElementsFor(importedTypes) + Scopes.scopedElementsFor(importedTypes, [EObject e | e.toQualifiedName()]).filter[type.isSuperTypeOf(it.EClass)]
			
		}
	}
	
	/** A specialized TypeLibraryScope which provides access to all AnnotationType instances provided by the specified type libraries.. */
	static class AnnotationTypeScope extends TypeLibraryScope {

		new(IScope outer, ITypeLibraryProvider context, Set<URI> importedLibs) {
			super(outer, context, importedLibs)
		}

		override protected getAllLocalElements() {
			Scopes.scopedElementsFor(importedDeclarations.filter(AnnotationType))
		}
	}
	
}