/**
 * Copyright (c) 2022 itemis AG - All rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * 
 * Contributors:
 * 	Andreas Muelder - itemis AG
 * 
 */
package com.yakindu.sct.generator.scxml

import com.google.inject.Inject
import com.yakindu.base.base.DocumentedElement
import com.yakindu.base.expressions.expressions.ReactionTrigger
import com.yakindu.base.expressions.expressions.RegularEventSpec
import com.yakindu.base.expressions.expressions.TimeEventSpec
import com.yakindu.sct.generator.core.ISGraphGenerator
import com.yakindu.sct.model.sgen.GeneratorEntry
import com.yakindu.sct.model.sgraph.Choice
import com.yakindu.sct.model.sgraph.Entry
import com.yakindu.sct.model.sgraph.EntryKind
import com.yakindu.sct.model.sgraph.Exit
import com.yakindu.sct.model.sgraph.FinalState
import com.yakindu.sct.model.sgraph.Reaction
import com.yakindu.sct.model.sgraph.Region
import com.yakindu.sct.model.sgraph.State
import com.yakindu.sct.model.sgraph.Statechart
import com.yakindu.sct.model.sgraph.Synchronization
import com.yakindu.sct.model.sgraph.Transition
import com.yakindu.sct.model.sgraph.Vertex
import com.yakindu.sct.model.stext.stext.BuiltinEventSpec
import com.yakindu.sct.model.stext.stext.DefaultTrigger
import java.util.List
import org.apache.commons.lang.StringEscapeUtils
import org.eclipse.emf.ecore.EObject
import org.eclipse.xtext.generator.IFileSystemAccess
import org.eclipse.xtext.naming.IQualifiedNameProvider

/**
 * 
 * @author andreas muelder - Initial contribution and API
 * 
 */
class SCXMLGenerator implements ISGraphGenerator {

	@Inject
	protected SCXMLModelModifications modifications
	@Inject
	protected SCXMLDataModelGenerator dataModel
	@Inject
	protected extension IQualifiedNameProvider
	@Inject
	extension protected SCXMLExpressions
	@Inject
	extension protected SCXMLReactiveElementGenerator reactiveElementGenerator

	override generate(Statechart sc, GeneratorEntry entry, IFileSystemAccess fsa) {
		val scCopy = modifications.modify(sc)
		fsa.generateFile(scCopy.name + '.scxml', scCopy.generate as String)
	}

	def dispatch String generate(Statechart it) '''
		<?xml version="1.0" encoding="UTF-8"?>
		«generateDocumentation»
		<scxml xmlns="http://www.w3.org/2005/07/scxml" version="1.0" datamodel="ecmascript" name="«name»">
			«dataModel.generate(it)»
			«IF regions.size > 1»
				<parallel>
			«ENDIF»
			«FOR region : regions»
				<state id="«region.fullyQualifiedName.toString»">
					«region.generate»
					«IF region == regions.head»
						«it.generateLocalReactions»
					«ENDIF»
				</state>
			«ENDFOR»
			«IF regions.size > 1»
				</parallel>
			«ENDIF»
		</scxml>
	'''

	def private String initialState(Region region) {
		val entry = region.getDefaultEntry
		if (entry !== null) {
			return if (entry.kind == EntryKind.INITIAL)
				entry.outgoingTransitions.head.target.transitionTarget.toString
			else
				entry.fullyQualifiedName.toString
		}
		""
	}

	def private getDefaultEntry(Region region) {
		region.vertices.filter(Entry).filter[isDefault].head
	}

	def dispatch CharSequence generate(Region it) '''
		«generateInitialTransition»
		«FOR vertex : vertices»
			«vertex.generate»
		 «ENDFOR»
	'''

	def dispatch CharSequence generate(Choice it) '''
		<state id="«it.fullyQualifiedName.toString»">
			«it.outgoingTransitions.filter[!isDefault].toList.generateTransitions»
			«it.outgoingTransitions.filter[isDefault].toList.generateTransitions»
		</state>
	'''

	def protected isDefault(Transition transition) {
		return if (transition.trigger instanceof ReactionTrigger) {
			val reactionTrigger = transition.trigger as ReactionTrigger
			reactionTrigger.triggers.empty && reactionTrigger.guard === null
		} else if (transition.trigger instanceof DefaultTrigger) {
			true
		} else if (transition.trigger === null) {
			true
		} else
			false
	}

	def dispatch CharSequence generate(State it) {
		if (orthogonal)
			generateParallelState
		else if (composite)
			generateCompositeState
		else
			generateLeafState
	}

	def CharSequence generateLeafState(State it) '''
		«generateDocumentation»
		<state id="«fullyQualifiedName.toString»">
			«dataModel.generate(it)»
			«generateLocalReactions»
			«generateOutgoingTransition»
		</state>
	'''

	def CharSequence generateCompositeState(State it) '''
		«generateDocumentation»
		<state id="«fullyQualifiedName.toString»">
			«dataModel.generate(it)»
			«generateLocalReactions»
			«FOR region : regions»
				«region.generate»
			«ENDFOR»
			«generateOutgoingTransition»
		</state>
	'''

	def CharSequence generateParallelState(State it) '''
		«generateDocumentation»
		<parallel id="«fullyQualifiedName.toString»">
			«dataModel.generate(it)»
			«generateLocalReactions»
			«FOR region : regions»
				<state id="«region.fullyQualifiedName.toString»">
					«region.generate»
				</state>
			«ENDFOR»
			«generateOutgoingTransition»
		</parallel>
	'''

	def dispatch CharSequence generate(Exit it) '''
	'''

	def dispatch CharSequence generate(Reaction it) '''
		«trigger.event» «trigger?.condition»
	'''

	def dispatch CharSequence generate(Transition it) '''
		«generateDocumentation»
		<transition event="«trigger?.event»" cond="«trigger?.condition»" target="«transitionTarget(target)»">
			«IF effect !== null»«reactiveElementGenerator.generate(it.effect)»«ENDIF»
		</transition>
	'''

	protected def dispatch transitionTarget(Vertex it) {
		fullyQualifiedName.toString
	}

	protected def dispatch generate(Synchronization it) {
	}

	protected def dispatch transitionTarget(Synchronization it) {
		// Join semantics here (Fork semantic is removed by SynchronizationModification), we use multiple targets
		'''«FOR target : outgoingTransitions.map[target] SEPARATOR ' '»«target.fullyQualifiedName.toString»«ENDFOR»'''
	}

	def dispatch CharSequence generate(FinalState it) '''
		<final id="«fullyQualifiedName.toString»"/>	
	'''

	def dispatch CharSequence generate(Entry it) {
		switch kind {
			case it.kind == EntryKind.DEEP_HISTORY: '''
				«var effect = outgoingTransitions?.head?.effect»
				<history type="deep" id="«fullyQualifiedName.toString»">
					«generateOutgoingTransition»
					«IF outgoingTransitions.size == 0»
						<transition target="«parentRegion.initialState»">
						«IF effect !== null»«reactiveElementGenerator.generate(effect)»«ENDIF»
						</transition>
					«ENDIF»
				</history>
			'''
			case EntryKind.SHALLOW_HISTORY: '''
				«var effect = outgoingTransitions?.head?.effect»
				<history type = "shallow" id="«fullyQualifiedName.toString»">
					«generateOutgoingTransition»
					«IF outgoingTransitions.size == 0»
						<transition target="«parentRegion.initialState»">
						«IF effect !== null»«reactiveElementGenerator.generate(effect)»«ENDIF»
						</transition>
					«ENDIF»
				</history>
			'''
			case EntryKind.INITIAL: ''''''
		}
	}

	def dispatch CharSequence event(
		ReactionTrigger it) '''«FOR trigger : triggers SEPARATOR ', '»«trigger.event»«ENDFOR»'''

	def dispatch event(BuiltinEventSpec it) ''''''

	def dispatch event(TimeEventSpec it) '''«fullyQualifiedName»'''

	def dispatch event(RegularEventSpec it) '''«getEvent?.code»'''

	def dispatch event(DefaultTrigger it) ''''''

	def dispatch condition(ReactionTrigger it) '''«guard?.expression?.code»'''

	def dispatch condition(DefaultTrigger it) ''''''

	def dispatch CharSequence generate(EObject it) '''
		Missig dispatch for «it»
	'''

	def protected generateInitialTransition(Region it) '''
		«var effect = defaultEntry?.outgoingTransitions?.head?.effect»
		<initial>
			<transition target="«initialState»" type="internal">
			«IF effect !== null»«reactiveElementGenerator.generate(effect)»«ENDIF»
			</transition>
		</initial>
	'''

	def protected generateOutgoingTransition(Vertex vertex) '''
		«vertex.outgoingTransitions.generateTransitions»
	'''

	def protected generateTransitions(List<Transition> transitions) '''
		«FOR transition : transitions»
			«transition.generate.toString.removeEmptyAttributes»
		«ENDFOR»
	'''

	def protected removeEmptyAttributes(String it) {
		replaceAll('cond=""', "").replaceAll('event=""', "")
	}
	
	def protected generateDocumentation(DocumentedElement it){
		'''«IF !documentation.nullOrEmpty»<!-- «StringEscapeUtils.escapeXml(documentation)» -->«ENDIF»'''
	}
}
