/*
* Copyright (c) 2022-2023 itemis AG - All rights Reserved
* Unauthorized copying of this file, via any medium is strictly prohibited
*
*/
package com.yakindu.base.expressions

import com.yakindu.base.expressions.expressions.AdditiveOperator
import com.yakindu.base.expressions.expressions.ArgumentExpression
import com.yakindu.base.expressions.expressions.AssignmentExpression
import com.yakindu.base.expressions.expressions.BitwiseAndExpression
import com.yakindu.base.expressions.expressions.BitwiseOrExpression
import com.yakindu.base.expressions.expressions.BitwiseXorExpression
import com.yakindu.base.expressions.expressions.BlockExpression
import com.yakindu.base.expressions.expressions.ElementReferenceExpression
import com.yakindu.base.expressions.expressions.ExpressionsFactory
import com.yakindu.base.expressions.expressions.FeatureCall
import com.yakindu.base.expressions.expressions.IfExpression
import com.yakindu.base.expressions.expressions.InitializationExpression
import com.yakindu.base.expressions.expressions.LambdaExpression
import com.yakindu.base.expressions.expressions.LogicalAndExpression
import com.yakindu.base.expressions.expressions.LogicalNotExpression
import com.yakindu.base.expressions.expressions.LogicalOrExpression
import com.yakindu.base.expressions.expressions.LogicalRelationExpression
import com.yakindu.base.expressions.expressions.MetaCall
import com.yakindu.base.expressions.expressions.MultiplicativeOperator
import com.yakindu.base.expressions.expressions.NumericalAddSubtractExpression
import com.yakindu.base.expressions.expressions.NumericalMultiplyDivideExpression
import com.yakindu.base.expressions.expressions.ParenthesizedExpression
import com.yakindu.base.expressions.expressions.PrimitiveValueExpression
import com.yakindu.base.expressions.expressions.RelationalOperator
import com.yakindu.base.expressions.expressions.ReturnExpression
import com.yakindu.base.expressions.expressions.TryExpression
import com.yakindu.base.expressions.expressions.WhileExpression
import com.yakindu.base.types.Declaration
import com.yakindu.base.types.Enumerator
import com.yakindu.base.types.Expression
import com.yakindu.base.types.Operation
import com.yakindu.base.types.Parameter
import com.yakindu.base.types.TypeSpecifier
import com.yakindu.base.types.TypesFactory
import java.util.List
import org.eclipse.emf.ecore.EObject
import java.math.BigInteger
import com.yakindu.base.expressions.expressions.AssignmentOperator

class ExpressionBuilder {
	
	extension ExpressionsFactory = ExpressionsFactory.eINSTANCE
	extension TypesFactory = TypesFactory.eINSTANCE

	def PrimitiveValueExpression _null() {
		createPrimitiveValueExpression => [
			value = createNullLiteral
		]
	}


	def PrimitiveValueExpression _true() {
		createPrimitiveValueExpression => [
			value = createBoolLiteral => [value = true]
		]
	}

	def PrimitiveValueExpression _false() {
		createPrimitiveValueExpression => [
			value = createBoolLiteral => [value = false]
		]
	}

	def PrimitiveValueExpression _integer(BigInteger l) {
		createPrimitiveValueExpression => [
			value = createIntLiteral => [value = l.longValue]
		]
	}
	
	def PrimitiveValueExpression _integer(Long l) {
		createPrimitiveValueExpression => [
			value = createIntLiteral => [value = l.longValue]
		]
	}
	
	def PrimitiveValueExpression _integer(Integer i) {
		createPrimitiveValueExpression => [
			value = createIntLiteral => [value = i]
		]
	}

	def PrimitiveValueExpression _double(Double d) {
		createPrimitiveValueExpression => [
			value = createDoubleLiteral => [value = d]
		]
	}
	
	def PrimitiveValueExpression _float(Float f) {
		createPrimitiveValueExpression => [
			value = createFloatLiteral => [value = f]
		]
	}
	
	def PrimitiveValueExpression _string(String s) {
		createPrimitiveValueExpression => [
			value = createStringLiteral => [value = s]
		]
	}
	
	def InitializationExpression _initialization(List<? extends Expression> args) {
		createInitializationExpression => [ e |
			args.forEach[ m |
				e.arguments += createArgument => [
					value = m
				]
			]
		]		
	}
	
	def Expression _value(Object it) {
		primitive
	}
	
	def protected dispatch Expression primitive(List<Object> struct){
		createInitializationExpression => [ae |
			struct.forEach[m |
				ae.arguments += createArgument => [
					value = m.primitive
				]
			]
		]
	}
	
	def protected dispatch Expression primitive(Boolean b) {
		createPrimitiveValueExpression => [
			value = createBoolLiteral => [value = b]
		]
	}
	
	def protected dispatch Expression primitive(Integer it) {
		_integer
	}

	def protected dispatch Expression primitive(BigInteger it) {
		_integer
	}

	def protected dispatch Expression primitive(Long it) {
		_integer
	}

	def protected dispatch Expression primitive(Double it) {
		_double
	}

	def protected dispatch Expression primitive(Float it) {
		_float
	}

	def protected dispatch Expression primitive(String it) {
		_string
	}

	def protected dispatch Expression primitive(Enumerator it) {
		_ref
	}
	
	def protected dispatch Expression primitive(Void it) {
		_null
	}


	def ElementReferenceExpression _ref(EObject p, Expression... args) {
		if(p instanceof ElementReferenceExpression) return p
		createElementReferenceExpression => [
			reference = p
			operationCall = p instanceof Operation
			arguments.addAll(args.map[arg|createArgument => [value = arg]])
		]
	}
	
	def ElementReferenceExpression _ref_array(EObject p, Expression... args) {
		if(p instanceof ElementReferenceExpression) return p
		createElementReferenceExpression => [
			reference = p
			arrayAccess = true
			operationCall = p instanceof Operation
			arraySelector.addAll(args)
		]
	}

	def FeatureCall _dot(Expression o, EObject p, Expression... args) {
		createFeatureCall => [
			owner = o
			feature = p
			operationCall = p instanceof Operation
			arguments.addAll(args.map[arg|createArgument => [value = arg]])
		]
	}

	def MetaCall _meta(Expression o, Declaration p, Expression... arguments) {
		createMetaCall => [
			owner = o
			feature = p
			operationCall = p instanceof Operation
			arguments.addAll(arguments.map[arg|createArgument => [value = arg]])
		]
	}
	
	def LambdaExpression _lambda( Expression e2, Parameter... e1) {
		createLambdaExpression => [
			parameters += e1
			expression = e2
		]
	}
	
	def NumericalAddSubtractExpression _plus(Expression e1, Expression e2) {
		createNumericalAddSubtractExpression => [
			leftOperand = e1
			rightOperand = e2
			operator = AdditiveOperator.PLUS
		]
	}
	
	def NumericalAddSubtractExpression _minus(Expression e1, Expression e2) {
		createNumericalAddSubtractExpression => [
			leftOperand = e1
			rightOperand = e2
			operator = AdditiveOperator.MINUS
		]
	}
	
	def NumericalMultiplyDivideExpression _multiply(Expression e1, Expression e2) {
		createNumericalMultiplyDivideExpression => [
			leftOperand = e1
			rightOperand = e2
			operator = MultiplicativeOperator.MUL
		]
	}
	
	def NumericalMultiplyDivideExpression _divide(Expression e1, Expression e2) {
		createNumericalMultiplyDivideExpression => [
			leftOperand = e1
			rightOperand = e2
			operator = MultiplicativeOperator.DIV
		]
	}
	
	def NumericalMultiplyDivideExpression _modulo(Expression e1, Expression e2) {
		createNumericalMultiplyDivideExpression => [
			leftOperand = e1
			rightOperand = e2
			operator = MultiplicativeOperator.MOD
		]
	}
	
	def BitwiseAndExpression _bitwiseAnd(Expression e1, Expression e2) {
		createBitwiseAndExpression => [
			leftOperand = e1
			rightOperand = e2
		]
	}
	
	def BitwiseOrExpression _bitwiseOr(Expression e1, Expression e2) {
		createBitwiseOrExpression => [
			leftOperand = e1
			rightOperand = e2
		]
	}
	
	def BitwiseXorExpression _bitwiseXor(Expression e1, Expression e2) {
		createBitwiseXorExpression => [
			leftOperand = e1
			rightOperand = e2
		]
	}
	

	
	def LogicalAndExpression _and(Expression e1, Expression e2) {
		createLogicalAndExpression => [
			leftOperand = e1
			rightOperand = e2
		]
	}
	def LogicalNotExpression _not(Expression e) {
		createLogicalNotExpression => [
			operand = e
		]
	}
	
	def LogicalOrExpression _or(Expression e1, Expression e2) {
		createLogicalOrExpression => [
			leftOperand = e1
			rightOperand = e2
		]
	}
	
	def LogicalRelationExpression _equals(Expression e1, Expression e2) {
		createLogicalRelationExpression => [
			leftOperand = e1
			rightOperand = e2
			operator = RelationalOperator.EQUALS
		]
	}
	
	def LogicalRelationExpression _notEquals(Expression e1, Expression e2) {
		createLogicalRelationExpression => [
			leftOperand = e1
			rightOperand = e2
			operator = RelationalOperator.NOT_EQUALS
		]
	}
	
	
	def LogicalRelationExpression _equalOrGreater(Expression e1, Expression e2) {
		createLogicalRelationExpression => [
			leftOperand = e1
			rightOperand = e2
			operator = RelationalOperator.GREATER_EQUAL
		]
	}
	
	def LogicalRelationExpression _smaller(Expression e1, Expression e2) {
		createLogicalRelationExpression => [
			leftOperand = e1
			rightOperand = e2
			operator = RelationalOperator.SMALLER
		]
	}
	
	def ParenthesizedExpression _parenthesis(Expression e) {
		createParenthesizedExpression => [
			expression = e
		]
	}
	
	def AssignmentExpression _assignment(Expression prop, Expression value) {
		createAssignmentExpression => [ ae |
			ae.varRef = prop
			ae.expression = value
		]
	}
	
	def BlockExpression _block(Expression... expSequence) {
		createBlockExpression => [
			expressions.addAll(expSequence)
		]
	}
	
	
	def ElementReferenceExpression _call(Operation op) {
		createElementReferenceExpression => [
			reference = op
			operationCall = true
		]
	}
	
	def FeatureCall _call(Expression ref, Operation op) {
		createFeatureCall => [
			owner = ref
			feature = op
			operationCall = true
		]
	}
	
	def <T extends ArgumentExpression> T _with(T it, Expression... params) {
		for (param : params) {
			it.arguments.add(createArgument => [arg|arg.value = param])
		}
		return it
	}
	
	def ReturnExpression _return(Expression value) {
		createReturnExpression => [
			it.expression = value
		]
	}
	
	def IfExpression _if(Expression condition, Expression then, Expression ^else) {
		createIfExpression => [
			it.condition = condition 
			it.then = then
			it.^else = ^else
		]
	}

	def IfExpression _if(Expression condition) {
		createIfExpression => [
			it.condition = condition 
		]
	}

	def IfExpression _then(IfExpression it, Expression then) {
		it.then = then
		return it
	}


	def IfExpression _else(IfExpression it, Expression ^else) {
		it.^else = ^else
		return it
	}


	def WhileExpression _while(Expression condition, Expression body) {
		createWhileExpression => [
			it.condition = condition 
			it.body = body
		]
	}


	def _declare(Declaration decl) {
		createDeclarationExpression => [
			declaration = decl
		]
	}
	
	def _cast(Expression exp, TypeSpecifier ts) {
		createTypeCastExpression => [
			operand = exp
			typeSpecifier = ts
		]
	}
	
	def _assign(Expression assignee, Expression value) {
		createAssignmentExpression => [ ae |
			ae.varRef = assignee
			ae.expression = value
		]
	}
	
	def _assign(Expression assignee, AssignmentOperator op, Expression value) {
		createAssignmentExpression => [ ae |
			ae.varRef = assignee
			ae.operator = op
			ae.expression = value
		]
	}
	

	def _raise(Expression ev, Expression value) {
		createEventRaisingExpression => [ er |
			er.event = ev
			er.value = value
		]
	}
	
	def _raise(Expression ev) {
		_raise(ev, null)
	}
	
	def _valueOf(Expression event) {
		createEventValueReferenceExpression => [ v |
			v.value = event
		]
	}
	
	def TryExpression _try(Expression tryBlck) {
		createTryExpression => [ tr |
			tr.^try = tryBlck
		]
	}
	
	def TryExpression _catch(TryExpression tryBlck, Expression catchBlock) {
		tryBlck.^catch = catchBlock
		tryBlck
	}
	
	def TryExpression _finally(TryExpression tryBlck, Expression finallyBlock) {
		tryBlck.^finally = finallyBlock
		tryBlck	
	}
	
	def _this() {
		createThisExpression
	}
	
	def _verbatim(String code) {
		createVerbatimExpression => [ it.code = code ]
	}
}
