/**
 * Copyright (c) 2020-2025 itemis AG - All rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * 
 */
package com.yakindu.base.expressions.inferrer

import com.google.common.collect.Maps
import com.google.inject.Inject
import com.yakindu.base.base.NamedElement
import com.yakindu.base.expressions.expressions.ArgumentExpression
import com.yakindu.base.expressions.expressions.AssignmentExpression
import com.yakindu.base.expressions.expressions.BitwiseAndExpression
import com.yakindu.base.expressions.expressions.BitwiseOrExpression
import com.yakindu.base.expressions.expressions.BitwiseXorExpression
import com.yakindu.base.expressions.expressions.BlockExpression
import com.yakindu.base.expressions.expressions.BoolLiteral
import com.yakindu.base.expressions.expressions.ConditionalExpression
import com.yakindu.base.expressions.expressions.DeclarationExpression
import com.yakindu.base.expressions.expressions.DoubleLiteral
import com.yakindu.base.expressions.expressions.ElementReferenceExpression
import com.yakindu.base.expressions.expressions.EventRaisingExpression
import com.yakindu.base.expressions.expressions.EventValueReferenceExpression
import com.yakindu.base.expressions.expressions.FeatureCall
import com.yakindu.base.expressions.expressions.FloatLiteral
import com.yakindu.base.expressions.expressions.HexLiteral
import com.yakindu.base.expressions.expressions.IfExpression
import com.yakindu.base.expressions.expressions.InitializationExpression
import com.yakindu.base.expressions.expressions.IntLiteral
import com.yakindu.base.expressions.expressions.LogicalAndExpression
import com.yakindu.base.expressions.expressions.LogicalNotExpression
import com.yakindu.base.expressions.expressions.LogicalOrExpression
import com.yakindu.base.expressions.expressions.LogicalRelationExpression
import com.yakindu.base.expressions.expressions.NullLiteral
import com.yakindu.base.expressions.expressions.NumericalAddSubtractExpression
import com.yakindu.base.expressions.expressions.NumericalMultiplyDivideExpression
import com.yakindu.base.expressions.expressions.NumericalUnaryExpression
import com.yakindu.base.expressions.expressions.ParenthesizedExpression
import com.yakindu.base.expressions.expressions.PostFixUnaryExpression
import com.yakindu.base.expressions.expressions.PrimitiveValueExpression
import com.yakindu.base.expressions.expressions.ReturnExpression
import com.yakindu.base.expressions.expressions.ShiftExpression
import com.yakindu.base.expressions.expressions.StringLiteral
import com.yakindu.base.expressions.expressions.SwitchExpression
import com.yakindu.base.expressions.expressions.TypeCastExpression
import com.yakindu.base.expressions.expressions.UnaryOperator
import com.yakindu.base.expressions.util.ExpressionExtensions
import com.yakindu.base.types.Annotation
import com.yakindu.base.types.AnnotationType
import com.yakindu.base.types.Argument
import com.yakindu.base.types.ArrayTypeSpecifier
import com.yakindu.base.types.ComplexType
import com.yakindu.base.types.EnumerationType
import com.yakindu.base.types.Enumerator
import com.yakindu.base.types.Event
import com.yakindu.base.types.Expression
import com.yakindu.base.types.GenericElement
import com.yakindu.base.types.Operation
import com.yakindu.base.types.Package
import com.yakindu.base.types.Parameter
import com.yakindu.base.types.Property
import com.yakindu.base.types.Type
import com.yakindu.base.types.TypeAlias
import com.yakindu.base.types.TypeParameter
import com.yakindu.base.types.TypeSpecifier
import com.yakindu.base.types.TypedDeclaration
import com.yakindu.base.types.TypesUtil
import com.yakindu.base.types.inferrer.AbstractTypeSystemInferrer
import com.yakindu.base.types.typesystem.EventValueMetaFeature
import com.yakindu.base.types.typesystem.ITypeSystem
import com.yakindu.base.types.util.ArgumentSorter
import com.yakindu.base.types.validation.IValidationIssueAcceptor
import com.yakindu.base.types.validation.IValidationIssueAcceptor.ListBasedValidationIssueAcceptor
import java.util.ArrayList
import java.util.List
import java.util.Map
import org.eclipse.emf.common.util.EList
import org.eclipse.emf.ecore.EObject
import org.eclipse.xtext.EcoreUtil2

import static com.yakindu.base.types.typesystem.ITypeSystem.*

import static extension org.eclipse.xtext.EcoreUtil2.*

/** 
 * 
 * @author andreas muelder - Initial contribution and API
 * @author axel terfloth - Extension for InitializationExpression
 * @author laszlo kovacs - Extension for InitializationExpression
 * 
 */
class ExpressionsTypeInferrer extends AbstractTypeSystemInferrer implements ExpressionsTypeInferrerMessages {

	public static final String ISSUE_CODE_MISSING_INITIALIZATION_ARGUMENT = "InitializationExpression.MissingArgument";
	public static final String ISSUE_MESSAGE_MISSING_INITIALIZATION_ARGUMENT = "Missing initialization argument. Expected %d arguments instead of %d for type %s.";

	public static final String ISSUE_CODE_FEATURE_CALL_IS_NOT_AN_OPERATION_CALL = "FeatureCall.NotOperationCall";
	public static final String ISSUE_MESSAGE_FEATURE_CALL_IS_NOT_AN_OPERATION_CALL = "Feature call is not an operation call. \"%s\" is a Property, not an Operation";

	public static final String ISSUE_CODE_SURPLUS_INITIALIZATION_ARGUMENT = "InitializationExpression.SurplusArgument";
	public static final String ISSUE_MESSAGE_SURPLUS_INITIALIZATION_ARGUMENT = "Surplus initialization argument. Expected %d arguments instead of %d for type %s.";

	public static final String ISSUE_CODE_INITIALIZATION_UNSUPPORTED_TYPE = "InitializationExpression.UnsupportedType";
	public static final String ISSUE_MESSAGE_INITIALIZATION_UNSUPPORTED_TYPE = "Initialization expression is not applicable for type %s.";

	public static final String ISSUE_MESSAGE_UNDEFINED_TYPE = "Undefined type for initialization expression.";

	public static final String ISSUE_MISSING_KEY_MSG = "Missing key in map initialization.";
	public static final String ISSUE_MISSING_KEY_CODE = "MissingKey";

	public static final String MISSING_VALUE = "Need to assign a value to an event of type %s.";
	public static final String EVENT_DEFINITION = "Cannot assign a value of type %s to an event of type %s.";

	@Inject protected TypeParameterInferrer typeParameterInferrer
	@Inject protected extension ExpressionExtensions utils
	@Inject protected extension EventValueMetaFeature
	@Inject protected ITypeSystem typeSystem
	@Inject protected extension TypesUtil

	def InferenceResult doInfer(AssignmentExpression e) {
		var InferenceResult result1 = inferTypeDispatch(e.getVarRef())
		var InferenceResult result2 = inferTypeDispatch(e.getExpression())
		assertAssignable(result1, result2, String.format(ASSIGNMENT_OPERATOR, e.getOperator(), result1, result2))
		return inferTypeDispatch(e.getVarRef())
	}

	def protected parameterForArgument(Argument it) {
		if (parameter !== null)
			return parameter

		val exp = eContainer as ArgumentExpression
		val feature = exp.featureOrReference
		switch ( feature ) {
			Operation case exp.arguments.indexOf(it) < feature.parameters.size:
				feature.parameters.get(exp.arguments.indexOf(it))
			default:
				null
		}
	}

	def InferenceResult doInfer(ConditionalExpression e) {
		var InferenceResult result1 = inferTypeDispatch(e.getTrueCase())
		var InferenceResult result2 = inferTypeDispatch(e.getFalseCase())
		assertCompatible(result1, result2, String.format(COMMON_TYPE, result1, result2))
		assertIsSubType(inferTypeDispatch(e.getCondition()), getResultFor(BOOLEAN), CONDITIONAL_BOOLEAN)
		return getCommonType(result1, result2)
	}

	def InferenceResult doInfer(LogicalOrExpression e) {
		var InferenceResult result1 = inferTypeDispatch(e.getLeftOperand())
		var InferenceResult result2 = inferTypeDispatch(e.getRightOperand())
		assertIsSubType(result1, getResultFor(BOOLEAN), String.format(LOGICAL_OPERATORS, "||", result1, result2))
		assertIsSubType(result2, getResultFor(BOOLEAN), String.format(LOGICAL_OPERATORS, "||", result1, result2))
		return getResultFor(BOOLEAN)
	}

	def InferenceResult doInfer(LogicalAndExpression e) {
		var InferenceResult result1 = inferTypeDispatch(e.getLeftOperand())
		var InferenceResult result2 = inferTypeDispatch(e.getRightOperand())
		assertIsSubType(result1, getResultFor(BOOLEAN), String.format(LOGICAL_OPERATORS, "&&", result1, result2))
		assertIsSubType(result2, getResultFor(BOOLEAN), String.format(LOGICAL_OPERATORS, "&&", result1, result2))
		return getResultFor(BOOLEAN)
	}

	def InferenceResult doInfer(LogicalNotExpression e) {
		var InferenceResult type = inferTypeDispatch(e.getOperand())
		assertIsSubType(type, getResultFor(BOOLEAN), String.format(LOGICAL_OPERATOR, "!", type))
		return getResultFor(BOOLEAN)
	}

	def InferenceResult doInfer(BitwiseXorExpression e) {
		var InferenceResult result1 = inferTypeDispatch(e.getLeftOperand())
		var InferenceResult result2 = inferTypeDispatch(e.getRightOperand())
		assertIsSubType(result1, getResultFor(INTEGER), String.format(BITWISE_OPERATORS, "^", result1, result2))
		assertIsSubType(result2, getResultFor(INTEGER), String.format(BITWISE_OPERATORS, "^", result1, result2))
		return getResultFor(INTEGER)
	}

	def InferenceResult doInfer(BitwiseOrExpression e) {
		var InferenceResult result1 = inferTypeDispatch(e.getLeftOperand())
		var InferenceResult result2 = inferTypeDispatch(e.getRightOperand())
		assertIsSubType(result1, getResultFor(INTEGER), String.format(BITWISE_OPERATORS, "|", result1, result2))
		assertIsSubType(result2, getResultFor(INTEGER), String.format(BITWISE_OPERATORS, "|", result1, result2))
		return getResultFor(INTEGER)
	}

	def InferenceResult doInfer(BitwiseAndExpression e) {
		var InferenceResult result1 = inferTypeDispatch(e.getLeftOperand())
		var InferenceResult result2 = inferTypeDispatch(e.getRightOperand())
		assertIsSubType(result1, getResultFor(INTEGER), String.format(BITWISE_OPERATORS, "&", result1, result2))
		assertIsSubType(result2, getResultFor(INTEGER), String.format(BITWISE_OPERATORS, "&", result1, result2))
		return getResultFor(INTEGER)
	}

	def InferenceResult doInfer(ShiftExpression e) {
		var InferenceResult result1 = inferTypeDispatch(e.getLeftOperand())
		var InferenceResult result2 = inferTypeDispatch(e.getRightOperand())
		assertIsSubType(result1, getResultFor(INTEGER),
			String.format(BITWISE_OPERATORS, e.getOperator(), result1, result2))
		assertIsSubType(result2, getResultFor(INTEGER),
			String.format(BITWISE_OPERATORS, e.getOperator(), result1, result2))
		return getResultFor(INTEGER)
	}

	def InferenceResult doInfer(LogicalRelationExpression e) {
		var InferenceResult result1 = inferTypeDispatch(e.getLeftOperand())
		var InferenceResult result2 = inferTypeDispatch(e.getRightOperand())
		assertCompatible(result1, result2, String.format(COMPARSION_OPERATOR, e.getOperator(), result1, result2))
		return getResultFor(BOOLEAN)
	}

	def InferenceResult doInfer(NumericalAddSubtractExpression e) {
		var InferenceResult result1 = inferTypeDispatch(e.getLeftOperand())
		var InferenceResult result2 = inferTypeDispatch(e.getRightOperand())
		assertCompatible(result1, result2, String.format(ARITHMETIC_OPERATORS, e.getOperator(), result1, result2))
		assertIsSubType(result1, getResultFor(REAL),
			String.format(ARITHMETIC_OPERATORS, e.getOperator(), result1, result2))
		return getCommonType(result1, result2)
	}

	def InferenceResult doInfer(NumericalMultiplyDivideExpression e) {
		var InferenceResult result1 = inferTypeDispatch(e.getLeftOperand())
		var InferenceResult result2 = inferTypeDispatch(e.getRightOperand())
		assertCompatible(result1, result2, String.format(ARITHMETIC_OPERATORS, e.getOperator(), result1, result2))
		assertIsSubType(result1, getResultFor(REAL),
			String.format(ARITHMETIC_OPERATORS, e.getOperator(), result1, result2))
		return getCommonType(result1, result2)
	}

	def InferenceResult doInfer(NumericalUnaryExpression e) {
		var InferenceResult result1 = inferTypeDispatch(e.getOperand())
		if (e.getOperator() === UnaryOperator.COMPLEMENT)
			assertIsSubType(result1, getResultFor(INTEGER),
				String.format(BITWISE_OPERATOR, Character.valueOf('~').charValue, result1))
		else {
			assertIsSubType(result1, getResultFor(REAL), String.format(ARITHMETIC_OPERATOR, e.getOperator(), result1))
		}
		return result1
	}

	def InferenceResult doInfer(PostFixUnaryExpression expression) {
		var InferenceResult result = inferTypeDispatch(expression.getOperand())
		assertIsSubType(result, getResultFor(REAL), null)
		return result
	}

	def InferenceResult doInfer(TypeCastExpression e) {
		var InferenceResult result1 = inferTypeDispatch(e.getOperand())
		var InferenceResult result2 = inferTypeDispatch(e.getTypeSpecifier())
		assertCompatible(result1, result2, String.format(CAST_OPERATORS, result1, result2))
		return result2
	}

	def InferenceResult doInfer(IfExpression it) {
		var InferenceResult condition = inferTypeDispatch(it.getCondition())
		assertIsSubType(condition, getResultFor(BOOLEAN), CONDITIONAL_BOOLEAN)
		if (it.getElse() !== null) {
			var InferenceResult thenResult = inferTypeDispatch(it.getThen())
			var InferenceResult elseResult = inferTypeDispatch(it.getElse())
			if (registry.isVoid(thenResult.type) || registry.isVoid(elseResult.type)) {
				return getResultFor(VOID)
			}
			assertCompatible(thenResult, elseResult, null)
			return getCommonType(thenResult, elseResult)
		}
		return getResultFor(VOID)
	}

	def InferenceResult doInfer(SwitchExpression it) {
		val InferenceResult condition = inferTypeDispatch(it.^switch)
		cases.forEach [ c |
			val caseType = inferTypeDispatch(c.^case)
			assertIsSubType(caseType, condition, null)
		]
		val result = cases.map[inferTypeDispatch(then)].reduce[p1, p2|getCommonType(p1, p2)]
		if (^default !== null) {
			val defType = inferTypeDispatch(^default)
			return getCommonType(defType, result)
		}
		result
	}

	def InferenceResult doInfer(BlockExpression it) {
		var InferenceResult result = getResultFor(VOID)
		var List<Expression> expressions = it.getExpressions()
		for (Expression expression : expressions) {
			result = inferTypeDispatch(expression)
		}
		return result
	}

	def InferenceResult doInfer(ReturnExpression exp) {
		if(exp.getExpression() === null) return getResultFor(VOID)
		return inferTypeDispatch(exp.getExpression())
	}

	def InferenceResult doInfer(EventRaisingExpression e) {
		val event = if(e.event.featureOrReference instanceof Event) e.event.featureOrReference as Event else null
		var eventType = if(event !== null) inferTypeDispatch(event.getTypeSpecifier()) else null
		eventType = eventType ?: getResultFor(VOID);

		if (e.getValue() === null) {
			assertSame(eventType, getResultFor(VOID), String.format(MISSING_VALUE, eventType));
			return getResultFor(VOID);
		}
		val valueType = inferTypeDispatch(e.getValue());
		assertAssignable(eventType, valueType, String.format(EVENT_DEFINITION, valueType, eventType));

		return valueType;
	}

	def InferenceResult doInfer(EventValueReferenceExpression e) {
		val definition = e.value.featureOrReference
		val result = definition.inferEventValue
		return if(result !== null) result else inferTypeDispatch(e.getValue());
	}

	def InferenceResult inferEventValue(EObject declaration) {
		if (declaration !== null && declaration instanceof Event) {
			val event = declaration as Event
			return if (event.getTypeSpecifier() === null)
				getResultFor(VOID)
			else
				inferTypeDispatch(event.getTypeSpecifier());
		}
		return null
	}

	def InferenceResult doInfer(DeclarationExpression it) {
		return getResultFor(VOID)
	}

	def InferenceResult doInfer(EnumerationType enumType) {
		return InferenceResult.from(enumType)
	}

	def InferenceResult doInfer(Enumerator enumerator) {
		return InferenceResult.from(EcoreUtil2.getContainerOfType(enumerator, Type))
	}

	def InferenceResult doInfer(Type type) {
		return InferenceResult.from(type.getOriginType())
	}

	def InferenceResult doInfer(Package pkg) {
		return null
	}

	/** 
	 * The type of a type alias is its (recursively inferred) base type, i.e. type
	 * aliases are assignable if their inferred base types are assignable.
	 */
	def InferenceResult doInfer(TypeAlias typeAlias) {
		return inferTypeDispatch(typeAlias.getTypeSpecifier())
	}

	def InferenceResult doInfer(FeatureCall e) {

		var feature = e.getFeature();
		var owner = e.getOwner();

        if (feature === null) {
            return getAnyType
        }

		// check meta features which require specific handling 
		if (e.feature.isEventValueProperty && owner !== null) {
			val result = e.owner.featureOrReference.inferEventValue
			if(result !== null) return result
		}

		// map to hold inference results for type parameters
		var Map<TypeParameter, InferenceResult> inferredTypeParameterTypes = Maps.newHashMap()
		if(owner !== null) typeParameterInferrer.inferTypeParametersFromOwner(inferTypeDispatch(e.getOwner()), inferredTypeParameterTypes)

		if (e.isOperationCall()) {
			if (! feature.eIsProxy) {
				if (feature instanceof Operation) {
					return inferOperation(e, feature, inferredTypeParameterTypes)
				}
				error(
					String.format(ISSUE_MESSAGE_FEATURE_CALL_IS_NOT_AN_OPERATION_CALL, 
						feature.featureName), 
						ISSUE_CODE_FEATURE_CALL_IS_NOT_AN_OPERATION_CALL)
			}
		}
		var InferenceResult result = inferWithoutErrors(e.getFeature())

		if (result !== null) {
			result = typeParameterInferrer.buildInferenceResult(result, inferredTypeParameterTypes, acceptor)
		}
		if (result === null) {
			return getAnyType
		}

		if (e.isArrayAccess && result?.type.name == ITypeSystem.ARRAY) {
			return getResult(result, e.arraySelector.size)
		}
		return result
	}

	protected def dispatch String featureName(EObject it) {
		""
	}

	protected def dispatch String featureName(NamedElement it) {
		name
	}



	def InferenceResult doInfer(ElementReferenceExpression e) {
		if (e.isOperationCall()) {
			if (e.getReference() !== null && !e.getReference().eIsProxy()) {
				return inferOperation(e, (e.getReference() as Operation),
					Maps.<TypeParameter, InferenceResult>newHashMap())
			} else {
				return getAnyType()
			}
		}
		var result = inferWithoutErrors(e.getReference())
		if (e.isArrayAccess && typeSystem.isArray(result?.type)) {
			// check key
			val indexType = e.arraySelector.lastOrNull.inferTypeDispatch
			assertAssignable(InferenceResult.from(typeSystem.getType(ITypeSystem.INTEGER)), indexType,
				String.format(INCOMPATIBLE_TYPES, indexType, ITypeSystem.INTEGER))
			return getResult(result, e.arraySelector.size)
		} else if (e.isArrayAccess && typeSystem.isMap(result?.type)) {
			// check key
			val indexType = e.arraySelector.lastOrNull.inferTypeDispatch
			assertAssignable(result.bindings.head, indexType,
				String.format(INCOMPATIBLE_TYPES, indexType, result.bindings.head.type.name))
			return result.bindings.lastOrNull
		}
		result
	}

	def protected InferenceResult inferOperation(ArgumentExpression e, Operation op,
		Map<TypeParameter, InferenceResult> typeParameterMapping) {
		// resolve type parameter from operation call
		var List<InferenceResult> argumentTypes = getArgumentTypes(e.expressions)
		var List<Parameter> parameters = op.getParameters()
		var List<InferenceResult> argumentsToInfer = new ArrayList()
		var List<TypedDeclaration> parametersToInfer = new ArrayList()
		for (var int i = 0; i < parameters.size(); i++) {
			if (!typeParameterMapping.containsKey(parameters.get(i).getType())) {
				parametersToInfer.add(parameters.get(i))
				if (i < argumentTypes.size()) {
					argumentsToInfer.add(argumentTypes.get(i))
				}
			}
		}
		typeParameterInferrer.inferTypeParametersFromOperationArguments(parametersToInfer, argumentsToInfer,
			typeParameterMapping, acceptor)
		validateParameters(typeParameterMapping, op, e.arguments, acceptor)
		return inferReturnType(e, op, typeParameterMapping)
	}

	def protected InferenceResult getTargetType(ArgumentExpression exp) {

		var result = getTargetTypeForAssignment(exp)
		if(result !== null) return result

		result = getTargetTypeForVariableInit(exp)
		if(result !== null) return result

		result = getTargetTypeForOperationArgument(exp)
		if(result !== null) return result

		return null
	}

	def protected getTargetTypeForAssignment(ArgumentExpression exp) {
		var EObject container = exp.eContainer()
		if (container instanceof AssignmentExpression) {
			var AssignmentExpression assignment = (container as AssignmentExpression)
			if (assignment.getExpression() === exp) {
				var Expression varRef = ((container as AssignmentExpression)).getVarRef()
				return inferTypeDispatch(varRef)
			}
		}
		return null
	}

	def protected getTargetTypeForVariableInit(ArgumentExpression exp) {
		var EObject container = exp.eContainer()
		if (container instanceof Property) {
			var Property property = (container as Property)
			if (property.getInitialValue() === exp) {
				return inferTypeDispatch(property.getTypeSpecifier())
			}
		}
		return null
	}

	def protected getTargetTypeForOperationArgument(ArgumentExpression exp) {
		var EObject container = exp.eContainer()
		if (container instanceof Argument) {
			var Argument argument = (container as Argument)
			if (argument.getValue() === exp) {
				var ArgumentExpression argumentExpression = (argument.eContainer() as ArgumentExpression)
				var int index = argumentExpression.expressions.indexOf(argument.getValue())
				var Operation op = (utils.featureOrReference(argumentExpression) as Operation)
				var Parameter param = op.getParameters().get(index)
				return inferTypeDispatch(param)
			}
		}
		return null
	}

	/** 
	 * Can be extended to e.g. add operation caller to argument list for extension
	 * methods
	 */
	def protected List<InferenceResult> getArgumentTypes(List<Expression> args) {
		var List<InferenceResult> argumentTypes = new ArrayList()
		for (Expression arg : args) {
			argumentTypes.add(inferTypeDispatch(arg))
		}
		return argumentTypes
	}

	def protected InferenceResult inferReturnType(ArgumentExpression e, Operation operation,
		Map<TypeParameter, InferenceResult> inferredTypeParameterTypes) {
		var InferenceResult returnType = inferTypeDispatch(operation)
		// if return type is not generic nor type parameter, we can return it immediately
		if (returnType !== null) {
			var Type type = returnType.getType()
			if (!(type instanceof TypeParameter) &&
				(!(type instanceof GenericElement) || ((type as GenericElement)).getTypeParameters().isEmpty())) {
				return returnType
			}
		}
		inferByTargetType(e, operation, inferredTypeParameterTypes)
		returnType = typeParameterInferrer.buildInferenceResult(returnType, inferredTypeParameterTypes, acceptor)
		if (returnType === null) {
			return getAnyType()
		}
		return returnType
	}

	def private void inferByTargetType(ArgumentExpression e, Operation operation,
		Map<TypeParameter, InferenceResult> inferredTypeParameterTypes) {
		// use target type inference
		var InferenceResult targetType = getTargetType(e)
		if (targetType !== null && targetType.getType !== null && !(targetType.getType instanceof TypeParameter)) {
			typeParameterInferrer.inferTypeParametersFromTargetType(targetType, operation, inferredTypeParameterTypes,
				acceptor)
		}
	}

	def protected InferenceResult getAnyType() {
		return InferenceResult.from(registry.getType(ANY))
	}

	/** 
	 * Takes the operation parameter type and performs a lookup for all contained
	 * type parameters by using the given type parameter inference map.<br>
	 * The parameter types are validated against the operation call's argument
	 * types.
	 * @throws TypeParameterInferrenceException
	 */
	def Map<TypeParameter, InferenceResult> validateParameters(Map<TypeParameter, InferenceResult> typeParameterMapping,
		Operation operation, List<Argument> args, IValidationIssueAcceptor acceptor) {

		var parameters = operation.getParameters()
		var argumentOrders = ArgumentSorter.getArgumentOrders(args, parameters)
			
		for (var int i = 0; i < parameters.size(); i++) {
			val index = i
			var expression = argumentOrders.findFirst[ order === index]?.argument?.value
			
			var Parameter parameter = parameters.get(i)
			var InferenceResult parameterType = inferTypeDispatch(parameter)
			var InferenceResult argumentType = inferTypeDispatch(expression)
			parameterType = typeParameterInferrer.buildInferenceResult(parameterType, typeParameterMapping,
				acceptor)
			assertAssignable(parameterType, argumentType,
				String.format(INCOMPATIBLE_TYPES_FOR_PARAM, argumentType, parameterType, parameter.name))
		}
		if (operation.isVariadic() && args.size() - 1 >= operation.getVarArgIndex()) {
			var expressions = ArgumentSorter.getOrderedExpressions(args, parameters)
			var Parameter parameter = operation.getParameters().get(operation.getVarArgIndex())
			var List<Expression> varArgs = expressions.subList(operation.getVarArgIndex(), args.size() - 1)
			var InferenceResult parameterType = inferTypeDispatch(parameter)
			for (Expression expression : varArgs) {
				parameterType = typeParameterInferrer.buildInferenceResult(parameterType, typeParameterMapping,
					acceptor)
				var InferenceResult argumentType = inferTypeDispatch(expression)
				assertAssignable(parameterType, argumentType,
					String.format(INCOMPATIBLE_TYPES, argumentType, parameterType))
			}
		}
		return typeParameterMapping
	}

	def InferenceResult doInfer(ParenthesizedExpression e) {
		return inferTypeDispatch(e.getExpression())
	}

	def InferenceResult doInfer(PrimitiveValueExpression e) {
		return inferTypeDispatch(e.getValue())
	}

	def InferenceResult doInfer(BoolLiteral literal) {
		return getResultFor(BOOLEAN)
	}

	def InferenceResult doInfer(IntLiteral literal) {
		return getResultFor(INTEGER)
	}

	def InferenceResult doInfer(HexLiteral literal) {
		return getResultFor(INTEGER)
	}

	def InferenceResult doInfer(DoubleLiteral literal) {
		return getResultFor(REAL)
	}

	def InferenceResult doInfer(FloatLiteral literal) {
		return getResultFor(REAL)
	}

	def InferenceResult doInfer(StringLiteral literal) {
		return getResultFor(STRING_LITERAL)
	}

	def InferenceResult doInfer(NullLiteral literal) {
		return getResultFor(NULL)
	}

	def InferenceResult doInfer(Property e) {
		var InferenceResult result = inferTypeDispatch(e.getTypeSpecifier())
		assertNotType(result, VARIABLE_VOID_TYPE, getResultFor(VOID))
		if(e.getInitialValue() === null) return result
		var InferenceResult result2 = inferTypeDispatch(e.getInitialValue())
		assertAssignable(result, result2, String.format(PROPERTY_INITIAL_VALUE, result2, result))
		return result
	}

	def InferenceResult doInfer(Operation e) {
		return if(e.getTypeSpecifier() === null) getResultFor(VOID) else inferTypeDispatch(e.getTypeSpecifier())
	}

	def InferenceResult doInfer(Parameter e) {
		return inferTypeDispatch(e.getTypeSpecifier())
	}

	def InferenceResult doInfer(TypeSpecifier specifier) {
		if (specifier.getType() instanceof GenericElement &&
			((specifier.getType() as GenericElement)).getTypeParameters().size() > 0) {
			var List<InferenceResult> bindings = new ArrayList()
			var EList<TypeSpecifier> arguments = specifier.getTypeArguments()
			for (TypeSpecifier typeSpecifier : arguments) {
				var InferenceResult binding = inferTypeDispatch(typeSpecifier)
				if (binding !== null) {
					bindings.add(binding)
				}
			}
			var Type type = inferTypeDispatch(specifier.getType()).getType()
			if (typeSystem.isArray(type)) {
				var arraySize = 0
				if (specifier instanceof ArrayTypeSpecifier) {
					arraySize = specifier.size
				}
				return InferenceResult.from(type, bindings, arraySize)
			} else if (typeSystem.isMap(type)) {
				return InferenceResult.from(type, bindings)
			} else {
				return InferenceResult.from(type, bindings)
			}
		}
		return inferTypeDispatch(specifier.getType())
	}

	def InferenceResult doInfer(Annotation ad) {
		validateParameters(newHashMap, ad.type, ad.arguments, acceptor)
		return getResultFor(VOID)
	}

	def protected void inferAnnotationProperty(AnnotationType type, EList<Argument> arguments) {
		var EList<Parameter> properties = type.parameters
		if (properties.size() === arguments.size()) {
			for (var int i = 0; i < properties.size(); i++) {
				var InferenceResult type1 = inferTypeDispatch(properties.get(i))
				var InferenceResult type2 = inferTypeDispatch(arguments.get(i))
				assertCompatible(type1, type2, String.format(INCOMPATIBLE_TYPES, type1, type2))
			}
		}
	}

	// =================================================================	
	// infer InitializationExpression
	def InferenceResult doInfer(InitializationExpression e) {
		val targetResult = e.inferByContext(e.eContainer)
		if (targetResult === null)
			return null
		val targetType = targetResult.type
		if (typeSystem.isUndefined(targetType)) {
			error(ISSUE_MESSAGE_UNDEFINED_TYPE, UNDEFINED_TYPE_CODE)
		// Target can be an array
		} else if (typeSystem.isArray(targetType)) {
			val expressions = e.arguments.map[value]
			expressions.forEach [ exp |
				var argumentType = inferTypeDispatch(exp)
				assertAssignable(targetResult.bindings.head, argumentType,
					String.format(INCOMPATIBLE_TYPES, argumentType, targetResult.bindings.head))
			]
		// Or a map
		} else if (typeSystem.isMap(targetType)) {
			val keys = e.arguments.map[key]
			keys.forEach [ exp |
				if (exp === null) {
					error(ISSUE_MISSING_KEY_MSG, ISSUE_MISSING_KEY_CODE)
					return;
				}
				var argumentType = inferTypeDispatch(exp)
				assertAssignable(targetResult.bindings.head, argumentType,
					String.format(INCOMPATIBLE_TYPES, argumentType, targetResult.bindings.head))
			]

			val values = e.arguments.map[value]
			values.forEach [ exp |
				var argumentType = inferTypeDispatch(exp)
				assertAssignable(targetResult.bindings.lastOrNull, argumentType,
					String.format(INCOMPATIBLE_TYPES, argumentType, targetResult.bindings.lastOrNull))
			]

		// Or a complexType
		} else if (targetType instanceof ComplexType) {
			val initFeatures = targetType.initializableFeatures
			val expressions = ArgumentSorter.getOrderedExpressions(e.arguments, initFeatures)
			for (var int i = 0; i < initFeatures.size(); i++) {
				if (expressions.size() > i) {
					var parameterType = inferTypeDispatch(targetType.allFeatures.get(i))
					var argumentType = inferTypeDispatch(expressions.get(i))
					assertAssignable(parameterType, argumentType,
						String.format(INCOMPATIBLE_TYPES, argumentType, parameterType))
				}
			}

			if (initFeatures.size > expressions.size) {
				error(
					String.format(ISSUE_MESSAGE_MISSING_INITIALIZATION_ARGUMENT, initFeatures.size, expressions.size,
						targetType.name), ISSUE_CODE_MISSING_INITIALIZATION_ARGUMENT)
			}
			if (initFeatures.size < expressions.size) {
				error(
					String.format(ISSUE_MESSAGE_SURPLUS_INITIALIZATION_ARGUMENT, initFeatures.size, expressions.size,
						targetType.name), ISSUE_CODE_SURPLUS_INITIALIZATION_ARGUMENT)
			}
		} else {
			error(String.format(ISSUE_MESSAGE_INITIALIZATION_UNSUPPORTED_TYPE, targetType.name),
				ISSUE_CODE_INITIALIZATION_UNSUPPORTED_TYPE)

		}

		return targetResult
	}

	def protected dispatch inferByContext(EObject e, EObject context) {
		getResultFor(UNDEFINED)
	}

	def protected dispatch inferByContext(InitializationExpression e, AssignmentExpression context) {
		inferTypeDispatch(context.varRef)
	}

	def protected dispatch inferByContext(InitializationExpression e, EventRaisingExpression context) {
		var result = context.event.featureOrReference.inferEventValue
		return if(result !== null) result else getResultFor(UNDEFINED)
	}

	def dispatch InferenceResult inferByContext(InitializationExpression e, Argument argument) {
		val result = if (argument.parameter !== null) {
				inferTypeDispatch(argument.parameter.type)
			} else {
				argument.inferByContext(argument.eContainer)
			}

		return if(result !== null) result else getResultFor(UNDEFINED)
	}

	def dispatch InferenceResult inferByContext(InitializationExpression e, Property context) {
		inferTypeDispatch(context)
	}

	def dispatch InferenceResult inferByContext(Argument e, ArgumentExpression context) {
		val param = parameterForArgument(e)

		if (param !== null)
			inferTypeDispatch(parameterForArgument(e).typeSpecifier)
		else
			getResultFor(UNDEFINED)
	}

	def dispatch InferenceResult inferByContext(Argument e, InitializationExpression context) {
		val argumentResult = context.inferByContext(context.eContainer)
		val contextType = argumentResult.type
		switch ( contextType ) {
			case typeSystem.isArray(contextType):
				argumentResult.bindings.head
			case typeSystem.isMap(contextType):
				argumentResult.bindings.lastOrNull
			ComplexType case context.arguments.indexOf(e) < contextType.initializableFeatures.size:
				inferTypeDispatch(contextType.initializableFeatures.get(context.arguments.indexOf(e)))
			default:
				getResultFor(UNDEFINED)
		}
	}

	def dispatch InferenceResult inferByContext(InitializationExpression e, ReturnExpression context) {
		return context.allContainers.filter(Operation).head.typeSpecifier.inferTypeDispatch
	}

	def dispatch InferenceResult inferByContext(InitializationExpression e, Expression context) {
		return inferTypeDispatch(context);
	}

	def protected initializableFeatures(ComplexType it) {
		return allFeatures.filter(Property).toList
	}

	// =================================================================	
	// helpers
	def protected InferenceResult getResult(InferenceResult result, int i) {
		if (i == 1) {
			return result.bindings.head
		}
		val j = i - 1;
		return getResult(result.bindings.head, j)
	}

	def protected inferWithoutErrors(EObject obj) {
		var tmp = acceptor
		try {
			acceptor = new ListBasedValidationIssueAcceptor()
			var InferenceResult result = inferTypeDispatch(obj)
			result
		} finally {
			acceptor = tmp
		}
	}
}
