/**
 * Copyright (c) 2025 itemis AG - All rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * 
 */
package com.yakindu.base.expressions.terminals

import com.google.inject.Inject
import java.util.HashSet
import java.util.Set
import org.apache.log4j.Logger
import org.eclipse.xtext.AbstractRule
import org.eclipse.xtext.GrammarUtil
import org.eclipse.xtext.TerminalRule
import org.eclipse.xtext.conversion.IValueConverter
import org.eclipse.xtext.conversion.IValueConverterService
import org.eclipse.xtext.conversion.ValueConverterException
import org.eclipse.xtext.nodemodel.INode

/**
 * This string value converter implementation considers a set of keywords which are not converted 
 * and delegates all over values to the value converter for the included terminal rule which is considered 
 * to be the ID rule by default.
 * 
 * The implementation especially supports using keywords as IDs with or without using escaping. 
 * This makes enables backwards compatibility from grammar version which require escaping to those which do not. 
 * 
 * This implementation is RuleSpecific and it retrieves the keywords from the rule definition. 
 * So it can be generically applied to different rules which follow the same pattern like
 * 'IDWithKeywords' or 'RefIDWithKeywords'. It expects that the rule consists of of a set of RuleCall and 
 * Keyword elements. Only a single rule call to a TerminalRule is expected. This terminal rule is used for 
 * all non keyword conversions.
 * 
 * @author axel terfloth
 * 
 */
class IDWithKeywordsValueConverter implements IValueConverter<String>, IValueConverter.RuleSpecific {

	static final Logger log = Logger.getLogger(IDWithKeywordsValueConverter)
	
	/** The converter service is required to delegate to the ID value converter if values is no keyword. */
	@Inject protected IValueConverterService converterService

	protected AbstractRule rule
	protected TerminalRule valueTerminalRule
	protected HashSet<String> matchingKeywords = new HashSet
	
	override toString(String value) throws ValueConverterException {
		if (matchingKeywords.contains(value) || valueTerminalRule === null)
			return value
		
		// delegate to default ID conversion if 'value' is not a matching keyword
		return converterService.toString(value, valueTerminalRule?.name)
	}
	
	override toValue(String string, INode node) throws ValueConverterException {
		val idValue = 
			if (matchingKeywords.contains(string) || valueTerminalRule === null)
				string
			else
				converterService.toValue(string, valueTerminalRule?.name , node)
		
		return (idValue instanceof String) ? idValue : string
	}
	
	override setRule(AbstractRule rule) throws IllegalArgumentException {
		this.rule = rule
		valueTerminalRule = null
		matchingKeywords.clear
	
		if (rule !== null) {
			matchingKeywords.addAll(rule.collectAllKeywords)
		}
	}
	
	def protected Set<String> collectAllKeywords(AbstractRule rule){ 
		val keywords = new HashSet<String>
		val visited = new HashSet<AbstractRule>
		val terminals = new HashSet<TerminalRule>
		collectKeywordsRecursive(rule, keywords, visited, terminals)
		if (terminals.size == 0) {
			log.warn('''No terminal rule found in rule "«rule.name»". Expect rule calls a terminal rule (transitively).''')
		} else if (terminals.size > 1) {
			log.warn('''Found multiple terminal rules referenced by "«rule.name»": «FOR r : terminals SEPARATOR ", "»"«r.name»"«ENDFOR». Expected a single reference. Ignore all terminal rules.''')
		} else {
			valueTerminalRule = terminals.head
		}
		
		return keywords
	}
	
	def protected void collectKeywordsRecursive(AbstractRule rule, Set<String> keywords, Set<AbstractRule> visited, Set<TerminalRule> terminals) {
		if (rule === null || visited.contains(rule))
			return;
		
		visited.add(rule);
		
		keywords.addAll(GrammarUtil.containedKeywords(rule).map[value])		
		GrammarUtil
			.containedRuleCalls(rule)
			.map[it.rule]
			.forEach[
				if (it instanceof TerminalRule)
					terminals += it
				else
					collectKeywordsRecursive(keywords, visited, terminals)
			]
	}
}
