/** 
 * Copyright (c) 2014-2025 itemis AG - All rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 */
package com.yakindu.base.expressions.validation

import com.google.common.collect.Lists
import com.google.inject.Inject
import com.yakindu.base.base.NamedElement
import com.yakindu.base.expressions.expressions.ArgumentExpression
import com.yakindu.base.expressions.expressions.AssignmentExpression
import com.yakindu.base.expressions.expressions.ElementReferenceExpression
import com.yakindu.base.expressions.expressions.EventValueReferenceExpression
import com.yakindu.base.expressions.expressions.ExpressionsPackage
import com.yakindu.base.expressions.expressions.FeatureCall
import com.yakindu.base.expressions.expressions.PostFixUnaryExpression
import com.yakindu.base.expressions.expressions.PrimitiveValueExpression
import com.yakindu.base.expressions.expressions.StringLiteral
import com.yakindu.base.expressions.util.ExpressionExtensions
import com.yakindu.base.types.AnnotatableElement
import com.yakindu.base.types.AnnotatedElement
import com.yakindu.base.types.Annotation
import com.yakindu.base.types.Argument
import com.yakindu.base.types.Enumerator
import com.yakindu.base.types.Event
import com.yakindu.base.types.Expression
import com.yakindu.base.types.Operation
import com.yakindu.base.types.Package
import com.yakindu.base.types.Parameter
import com.yakindu.base.types.Property
import com.yakindu.base.types.Type
import com.yakindu.base.types.TypesPackage
import com.yakindu.base.types.annotations.DeprecatedAnnotations
import com.yakindu.base.types.annotations.TypeAnnotations
import com.yakindu.base.types.inferrer.ITypeSystemInferrer
import com.yakindu.base.types.util.ArgumentSorter
import com.yakindu.base.types.validation.IValidationIssueAcceptor
import com.yakindu.base.types.validation.TypeValidator
import com.yakindu.base.types.validation.TypesJavaValidator
import java.util.ArrayList
import java.util.List
import org.eclipse.emf.common.util.EList
import org.eclipse.emf.common.util.TreeIterator
import org.eclipse.emf.ecore.EClass
import org.eclipse.emf.ecore.ENamedElement
import org.eclipse.emf.ecore.EObject
import org.eclipse.emf.ecore.EPackage
import org.eclipse.xtext.EcoreUtil2
import org.eclipse.xtext.nodemodel.ICompositeNode
import org.eclipse.xtext.nodemodel.util.NodeModelUtils
import org.eclipse.xtext.validation.Check
import org.eclipse.xtext.validation.CheckType
import org.eclipse.xtext.validation.ComposedChecks

import static com.yakindu.base.types.annotations.DeprecatedAnnotations.*

import static extension com.yakindu.base.types.TypesUtil.*

/** 
 * @author andreas muelder - Initial contribution and API
 */
@ComposedChecks(validators=#[TypesJavaValidator, ReturnValidator])
class ExpressionsValidator extends AbstractExpressionsValidator implements IValidationIssueAcceptor {

	@Inject ITypeSystemInferrer typeInferrer
	@Inject TypeValidator typeValidator
	@Inject extension ExpressionExtensions
	@Inject protected extension DeprecatedAnnotations
	@Inject protected extension TypeAnnotations
	@Inject protected extension ArgumentSorter

	@Check
	def void checkExpression(Expression expression) {
		// Only infer root expressions since inferType infers the expression
		// containment hierarchy
		if(!(expression.eContainer() instanceof Expression)) typeInferrer.infer(expression, this)
	}

	@Check(CheckType.FAST)
	def void checkExpression(Property expression) {
		if(expression.getType() === null || expression.getType().eIsProxy()) return;
		typeInferrer.infer(expression, this)
	}

	override void accept(ValidationIssue issue) {
		switch (issue.getSeverity()) {
			case ERROR: {
				error(issue.getMessage(), null, issue.getIssueCode())
			}
			case WARNING: {
				warning(issue.getMessage(), null, issue.getIssueCode())
			}
			case INFO: {
			}
		}
	}

	public static final String POSTFIX_ONLY_ON_VARIABLES_CODE = "PostfixOnlyOnVariables"
	public static final String POSTFIX_ONLY_ON_VARIABLES_MSG = "Invalid argument to operator '++/--'"

	@Check
	def void checkPostFixOperatorOnlyOnVariables(PostFixUnaryExpression expression) {
		if (!(expression.getOperand() instanceof ElementReferenceExpression) &&
			!(expression.getOperand() instanceof FeatureCall)) {
			error(POSTFIX_ONLY_ON_VARIABLES_MSG, expression, null, POSTFIX_ONLY_ON_VARIABLES_CODE)
		}
	}

	public static final String ERROR_SURPLUS_PARAMETER_ASSIGNMENT_CODE = "ErrorSurplusParameterAssignment"
	public static final String ERROR_SURPLUS_PARAMETER_ASSIGNMENT_MSG = "Surplus assignment to parameter '%s'."

	public static final String ERROR_ARG_FOR_UNDEFINED_PARAMETER_CODE = "ErrorArgForUndefinedParameter"
	public static final String ERROR_ARG_FOR_UNDEFINED_PARAMETER_MSG = "Argument refers to undefined parameter '%s'."

	public static final String WARNING_REORDERD_ARGUMENT_LISTS_SHOULD_ONLY_USE_NAMED_ARGUMENTS_CODE = "Warning.ReorderedArgumentListShouldOnlyUseNamedArguments"
	public static final String WARNING_REORDERD_ARGUMENT_LISTS_SHOULD_ONLY_USE_NAMED_ARGUMENTS_MSG = "When using named arguments in an unordered fashion all arguments should be named."

	@Check
	def void checkParameters(ArgumentExpression exp) {
		val declaration = exp.featureOrReference
		var errors = 0
				
		if (declaration instanceof Operation) {
			val argumentOrders = ArgumentSorter.getArgumentOrders(exp.arguments, declaration.parameters)
			
			if (! argumentOrders.empty) {
				val maxArgumentsSize = Math.max(argumentOrders.size, declaration.parameters.size)
				val Argument[] sortedArguments = newArrayOfSize(maxArgumentsSize)
				
				for ( ao : argumentOrders) {
					val argument = ao.argument
					val paramNameOrIndex = (ao.name.isNullOrEmpty) ? argumentOrders.indexOf(ao) : ao.name
		
					if ( ! ao.valid ) {
						error(
							String.format(ERROR_ARG_FOR_UNDEFINED_PARAMETER_MSG, paramNameOrIndex) ,
							argument, null, ERROR_ARG_FOR_UNDEFINED_PARAMETER_CODE)
						errors++
					}
					if (sortedArguments.get(ao.order) === null) {
						sortedArguments.set(ao.order, argument)
					} else {
						error(String.format(ERROR_SURPLUS_PARAMETER_ASSIGNMENT_MSG, paramNameOrIndex),
							argument, null, ERROR_SURPLUS_PARAMETER_ASSIGNMENT_CODE)
						errors++
					}
				}
				
				// TODO : validate number of arguments here instead of using 'assertOperationArguments()'.
				//        more information is collected here
				
				// if arguments were reordered partially and not all arguments were named we will provide a warrning
				
				var reorderExists = (0..argumentOrders.size-1).exists[ i | argumentOrders.get(i).order !== i]
				var unnamedExists = (0..argumentOrders.size-1).exists[ i | argumentOrders.get(i).name.nullOrEmpty]
				if (errors === 0 && reorderExists && unnamedExists)
					warning(
						WARNING_REORDERD_ARGUMENT_LISTS_SHOULD_ONLY_USE_NAMED_ARGUMENTS_MSG,
						exp, null, WARNING_REORDERD_ARGUMENT_LISTS_SHOULD_ONLY_USE_NAMED_ARGUMENTS_CODE)
			}
		}
	}
	
	public static final String WARNING_AMBIGOUS_MSG = "%s is ambiguous. Please consider to use fully qualified name.";
	public static final String WARNING_AMBIGOUS_CODE = "ambigousType";

	@Check(CheckType.FAST)
	def void checkAmbigousEnumInGuard(FeatureCall it) {
		if(featureOrReference instanceof Enumerator){
			val type = featureOrReference.eContainer as Type
			if(EcoreUtil2.getRootContainer(type) instanceof Package){
				val node = NodeModelUtils.getNode(it)
				val packageContainedTypes = (EcoreUtil2.getRootContainer(type) as Package).collectPackageContainedTypes(newHashSet)
				if(packageContainedTypes.exists[it !== type && name !== null && name.equals(type.name)] && node.text.contains(type.name) && !node.text.contains(type.toQualifiedName.toString))
					warning(String.format(WARNING_AMBIGOUS_MSG, type.name), it, null, WARNING_AMBIGOUS_CODE)
			}
		}
	}

	public static final String ERROR_ASSIGNMENT_TO_CONST_CODE = "AssignmentToConst"
	public static final String ERROR_ASSIGNMENT_TO_CONST_MSG = "Assignment to constant not allowed."

	@Check(CheckType.FAST)
	def void checkAssignmentToFinalVariable(AssignmentExpression exp) {
		var EObject referencedObject = exp.getVarRef().featureOrReference
		if (referencedObject instanceof Property) {
			if (referencedObject.isConst()) {
				error(ERROR_ASSIGNMENT_TO_CONST_MSG, ExpressionsPackage.Literals.ASSIGNMENT_EXPRESSION__VAR_REF,
					ERROR_ASSIGNMENT_TO_CONST_CODE)
			}
		}
	}

	public static final String ERROR_POST_FIX_TO_CONST_CODE = "PostFixToConst"
	public static final String ERROR_POST_FIX_TO_CONST_MSG = "Increment or decrement to constant not allowed."

	@Check(CheckType.FAST)
	def void checkPostFixUnaryExpressionToFinalVariable(PostFixUnaryExpression exp) {
		var EObject referencedObject = exp.getOperand().featureOrReference
		if (referencedObject instanceof Property) {
			if (referencedObject.isConst()) {
				error(ERROR_POST_FIX_TO_CONST_MSG, exp, null, ERROR_POST_FIX_TO_CONST_CODE)
			}
		}
	}

	public static final String ERROR_LEFT_HAND_ASSIGNMENT_CODE = "LeftHandAssignment"
	public static final String ERROR_LEFT_HAND_ASSIGNMENT_MSG = "The left-hand side of an assignment must be a variable."

	@Check(CheckType.FAST)
	def void checkLeftHandAssignment(AssignmentExpression expression) {
		var Expression varRef = expression.getVarRef()
		if (varRef instanceof FeatureCall) {
			var EObject referencedObject = varRef.getFeature()
			if (!(referencedObject.eIsProxy) && !(referencedObject instanceof Property)) {
				error(ERROR_LEFT_HAND_ASSIGNMENT_MSG, ExpressionsPackage.Literals.ASSIGNMENT_EXPRESSION__VAR_REF,
					ERROR_LEFT_HAND_ASSIGNMENT_CODE)
			}
		} else if (varRef instanceof ElementReferenceExpression) {
			var EObject referencedObject = varRef.getReference()
			if (!(referencedObject.eIsProxy) && !(referencedObject instanceof Property) && !(referencedObject instanceof Parameter)) {
				error(ERROR_LEFT_HAND_ASSIGNMENT_MSG, ExpressionsPackage.Literals.ASSIGNMENT_EXPRESSION__VAR_REF,
					ERROR_LEFT_HAND_ASSIGNMENT_CODE)
			}
		} else {
			error(ERROR_LEFT_HAND_ASSIGNMENT_MSG, ExpressionsPackage.Literals.ASSIGNMENT_EXPRESSION__VAR_REF,
				ERROR_LEFT_HAND_ASSIGNMENT_CODE)
		}
	}

	@Check(CheckType.FAST)
	def void checkOperationArguments_FeatureCall(FeatureCall call) {
		if (call.getFeature() instanceof Operation) {
			var Operation operation = (call.getFeature() as Operation)
			assertOperationArguments(operation, call.getExpressions())
		}
	}
	
	public static final String DEPRECATED_ANNOTATABLE_ELEMENT = "This is a deprecated element and will be removed in the future. ";
	
	@Check(CheckType.FAST)
	def void checkDeprecatedInGuard(FeatureCall fc) {
		fc.expressions.filter(FeatureCall).forEach[a |
			if(a.feature.isDeprecated) {
				warning(DEPRECATED_ANNOTATABLE_ELEMENT + a.feature.deprecationMessage, a, null)
			}
		]		
	}


	def dispatch String deprecationMessage(AnnotatableElement it) {
		val annotation = getAnnotationOfType(DEPRECATED_ANNOTATION)
		if(annotation !== null && annotation.arguments.size > 0) {
			val expr = annotation.arguments.get(0).value
			if (expr instanceof PrimitiveValueExpression) {
				val msg = expr.value
				if (msg instanceof StringLiteral)
					return msg.value
			}
		}
		
		""
	}
	
	def dispatch String deprecationMessage(EObject it) {
		""		
	}

	@Check(CheckType.FAST)
	def void checkOperationArguments_TypedElementReferenceExpression(ElementReferenceExpression call) {
		if (call.getReference() instanceof Operation) {
			var Operation operation = (call.getReference() as Operation)
			assertOperationArguments(operation, call.getExpressions())
		}
	}

	@Check(CheckType.FAST)
	def void checkAnnotationArguments(Annotation annotation) {
		assertOperationArguments(annotation.type, annotation.expressions)
		typeInferrer.infer(annotation, this)
	}

	public static final String ERROR_WRONG_NUMBER_OF_ARGUMENTS_CODE = "WrongNrOfArgs"
	public static final String ERROR_WRONG_NUMBER_OF_ARGUMENTS_MSG = "Wrong number of arguments, expected %s ."

	def protected void assertOperationArguments(Operation operation, List<Expression> args) {
		var EList<Parameter> parameters = operation.getParameters()
		var List<Parameter> optionalParameters = filterOptionalParameters(parameters)
		if ((operation.isVariadic() && operation.getVarArgIndex() > args.size()) || (!operation.isVariadic() &&
			!(args.size() <= parameters.size() && args.size() >= parameters.size() - optionalParameters.size()))) {
			error(String.format(ERROR_WRONG_NUMBER_OF_ARGUMENTS_MSG, parameters), null,
				ERROR_WRONG_NUMBER_OF_ARGUMENTS_CODE)
		}
	}

	def protected List<Parameter> filterOptionalParameters(EList<Parameter> parameters) {
		var List<Parameter> optionalParameters = new ArrayList()
		for (Parameter p : parameters) {
			if (p.isOptional()) {
				optionalParameters.add(p)
			}
		}
		return optionalParameters
	}

	public static final String ERROR_WRONG_ANNOTATION_TARGET_CODE = "WrongAnnotationTarget"
	public static final String ERROR_WRONG_ANNOTATION_TARGET_MSG = "Annotation '%s' can not be applied on '%s'. It is applicable to %s."


	def protected AnnotatedElement annotatedElement(Annotation annotation) {
		val parent = annotation.eContainer
		
		if (parent instanceof AnnotatedElement) {
			return parent.annotatedElement
		} else {
			null
		}
	}
	
	def protected AnnotatedElement annotatedElement(AnnotatedElement e) {
		if ( e.eClass === TypesPackage.Literals.ANNOTATABLE_ELEMENT ) {
			return (e.eContainer.annotatedElement)
		}
		return e
	}
	
	def protected AnnotatedElement annotatedElement(EObject e) {
		if ( e instanceof AnnotatedElement ) {
			e
		} else {
			null
		}
	}
	
	@Check(CheckType.FAST)
	def void checkAnnotationTarget(Annotation annotation) {
		// Get annotated element and guard for AnnotableElement#getAnnotationInfo container
		val annotatedElement = annotation.annotatedElement
		if (annotatedElement === null ) return
		
		var EList<EObject> targets = annotation.getType().getTargets()
		if (!targets.empty) {
			var boolean found = 
				targets
					.filter(EClass)
					.map[ c |
						EPackage.Registry.INSTANCE.getEPackage(c.EPackage.nsURI).getEClassifier(c.name)
					]
					.exists[
						isInstance(annotatedElement)
					]
			if (!found) {
				error(
					String.format(
						ERROR_WRONG_ANNOTATION_TARGET_MSG, 
						annotation.getType().getName(),
						annotatedElement.eClass().name,
						'''«FOR t : targets.filter(ENamedElement) SEPARATOR ', '»'«t.name»'«ENDFOR»'''
					), 
					annotation, null, ERROR_WRONG_ANNOTATION_TARGET_CODE)
			}
		}
	}

	public static final String CONST_MUST_HAVE_VALUE_MSG = "A constant definition must specify an initial value."
	public static final String CONST_MUST_HAVE_VALUE_CODE = "ConstMustHaveAValue"
	public static final String REFERENCE_TO_VARIABLE = "Cannot reference a variable in a constant initialization."

	@Check(CheckType.FAST)
	def void checkValueDefinitionExpression(Property property) {
		// applies only to constants
		if(!property.isConst()) return;
		var Expression initialValue = property.getInitialValue()
		if (initialValue === null) {
			error(CONST_MUST_HAVE_VALUE_MSG, property, null, CONST_MUST_HAVE_VALUE_CODE)
			return;
		}
		var List<Expression> toCheck = Lists.newArrayList(initialValue)
		var TreeIterator<EObject> eAllContents = initialValue.eAllContents()
		while (eAllContents.hasNext()) {
			var EObject next = eAllContents.next()
			if(next instanceof Expression) toCheck.add(next)
		}
		for (Expression expression : toCheck) {
			var EObject referencedObject = null
			if (expression instanceof FeatureCall)
				referencedObject = expression.getFeature()
			else if (expression instanceof ElementReferenceExpression)
				referencedObject = expression.getReference()
			if (referencedObject instanceof Property) {
				if (!referencedObject.isConst()) {
					if (typeValidator.isAnyType(referencedObject.type)) {
						return
					}
					error(REFERENCE_TO_VARIABLE, TypesPackage.Literals.PROPERTY__INITIAL_VALUE)
				}
			}
		}
	}

	public static final String DECLARATION_WITH_READONLY = "The keyword '%s' has no effect for '%s' definitions. Can be removed."

	@Check(CheckType.FAST)
	def void checkConstAndReadOnlyDefinitionExpression(Property definition) {
		// applies only for readonly const definitions
		if(!definition.isReadonly() && !definition.isConst()) return;
		var ICompositeNode definitionNode = NodeModelUtils.getNode(definition)
		var String tokenText = NodeModelUtils.getTokenText(definitionNode)
		if(tokenText === null || tokenText.isEmpty()) return;
		if (tokenText.contains(TypesPackage.Literals.PROPERTY__READONLY.getName()) &&
			tokenText.contains(TypesPackage.Literals.PROPERTY__CONST.getName())) {
			warning(
				String.format(DECLARATION_WITH_READONLY, TypesPackage.Literals.PROPERTY__READONLY.getName(),
					TypesPackage.Literals.PROPERTY__CONST.getName()), definition,
				TypesPackage.Literals.PROPERTY__READONLY)
		}
	}

	public static final String ASSIGNMENT_EXPRESSION = "No nested assignment of the same variable allowed (different behavior in various programming languages).";

	@Check
	def void checkAssignmentExpression(AssignmentExpression exp) {
		if (exp.eAllContents.filter(AssignmentExpression).exists [
			varRef.featureOrReference == exp.varRef.featureOrReference
		]) {
			error(ASSIGNMENT_EXPRESSION, null)
		}

	}

	public static final String VALUE_OF_REQUIRES_EVENT = "valueof() expression requires event as argument.";

	@Check(CheckType.FAST)
	def void checkValueOfNoEvent(EventValueReferenceExpression exp) {
		var element = exp.getValue().featureOrReference
		if (element !== null && (!(element instanceof Event))) {
			var String msg = "Could not find event declaration."
			if (element instanceof NamedElement) {
				msg = ''''«element.getName()»' is no event.'''
			}
			error(msg, ExpressionsPackage.Literals.EVENT_VALUE_REFERENCE_EXPRESSION__VALUE, 0, VALUE_OF_REQUIRES_EVENT)
		}
	}
}
