/**
 * Copyright (c) 2020-2023 itemis AG - All rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * 
 */
package com.yakindu.sct.generator.cpp.files

import com.google.inject.Inject
import com.itemis.create.base.generator.core.types.Literals
import com.yakindu.sct.generator.c.extensions.GenmodelEntries
import com.yakindu.sct.generator.core.artifacts.IContentTemplate
import com.yakindu.sct.generator.core.artifacts.IGenArtifactConfigurations
import com.yakindu.sct.generator.cpp.CppFileNaming
import com.yakindu.sct.generator.cpp.CppNaming
import com.yakindu.sct.generator.cpp.CppPointers
import com.yakindu.sct.generator.cpp.CppSpecifiers
import com.yakindu.sct.generator.cpp.features.GenmodelEntriesExtension
import com.yakindu.sct.generator.cpp.types.CppTypes
import com.yakindu.sct.model.sexec.ExecutionFlow
import com.yakindu.sct.model.sexec.transformation.config.IFlowConfiguration
import com.yakindu.sct.model.sgen.GeneratorEntry

/**
 * @author Robin Herrmann
 * @author Axel Terfloth
 */
class TimerServiceHeader implements IContentTemplate<ExecutionFlow> {

	@Inject extension GenmodelEntries
	@Inject protected extension TimerServiceHeaderMethods
	@Inject protected extension CppFileNaming
	@Inject protected extension CppNaming
	@Inject protected extension CppTypes
	@Inject protected extension CppPointers
	@Inject protected extension Literals
	@Inject protected extension CppSpecifiers
	@Inject protected extension IFlowConfiguration
	@Inject protected GenmodelEntriesExtension genExt
	
	

	override content(ExecutionFlow flow, GeneratorEntry entry, IGenArtifactConfigurations locations) '''
		«entry.licenseText»
		#ifndef «timerServiceModule.define»_H_
		#define «timerServiceModule.define»_H_
		
		«includes»
		
		namespace sc {
		namespace timer {
			
		class «timerServiceImplementationTask» {
		private:
			struct TimeBased {
				«weakPtr»TimedInterface«pointerType» handle;
				«sc_eventid.fqName» pt_evid;
				«sc_bool.fqName» periodic;
				TimeBased() : handle(«nullptrInit»), pt_evid(0), periodic(false) {}
			};
			struct NonEmpty {
				«sc_time.fqName» time_ms;
				TimeBased time_event;
				«weakPtr»«cycleBasedInterface»«pointerType» run_cycle_handle;
				NonEmpty() : time_ms(0), time_event(«timeBasedInit»), run_cycle_handle(«nullptrInit») {}
			};
		public:
			struct TaskData {
				enum TaskType {
					EMPTY_TASK,
					TIME_EVENT_TASK,
					RUNCYCLE_TASK
				};
				TaskType type;
				NonEmpty get;
				TaskData() : type(EMPTY_TASK) {}
				TaskData(«weakPtr»TimedInterface«pointerType» time_event_handle, «sc_eventid.fqName» pt_evid, «sc_time.fqName» time_ms, «sc_bool.fqName» periodic) : 
					type(TIME_EVENT_TASK) {
					get.time_ms = time_ms;
					get.time_event.handle = time_event_handle;
					get.time_event.pt_evid = pt_evid;
					get.time_event.periodic = periodic;
				}
				TaskData(«weakPtr»«cycleBasedInterface»«pointerType» run_cycle_handle, «sc_time.fqName» time_ms) : 
					type(RUNCYCLE_TASK) {
					get.time_ms = time_ms;
					get.run_cycle_handle = run_cycle_handle;
				}
				
				void reset() {
					switch (type) {
						case TIME_EVENT_TASK:
							get.time_ms = 0;
							«resetHandle("get.time_event.handle")»
							get.time_event.pt_evid = 0;
							get.time_event.periodic = false;
							break;
						case RUNCYCLE_TASK:
							get.time_ms = 0;
							«resetHandle("get.run_cycle_handle")»
							break;
						default: return;
					}
					type = EMPTY_TASK;
				}
				
				virtual void execute() {
					switch (type) {
						case TIME_EVENT_TASK: {
							// fire the event
							«sharedPtr»TimedInterface«pointerType» time_event_handle = get.time_event.handle«lockIfNecessary»;
							if (time_event_handle != «NULL_LITERAL») {
								time_event_handle->raiseTimeEvent(get.time_event.pt_evid);
							}
							return;
						}
						case RUNCYCLE_TASK: {
							«sharedPtr»«cycleBasedInterface»«pointerType» run_cycle_handle = get.run_cycle_handle«lockIfNecessary»;
							if (run_cycle_handle != «NULL_LITERAL») {
								run_cycle_handle->runCycle();
							}
							return;
						}
						default: return;
					}
				}
				«taskDataDestructor»
			};
			
			TaskData data;
			«sc_time.fqName» elapsed_time_ms;
			size_t next_task_idx;
		
			«timerTaskDefaultConstructor»
		
			«timerServiceImplementationTask»(const TaskData &data_) : //
					data(data_), //
					elapsed_time_ms(0), //
					next_task_idx(0)
			{
			}
		
			virtual ~TimerTask() {
			}
		
			void updateElapsedTimeMs(«sc_time.fqName» elapsed_time_ms_ = 0) {
				elapsed_time_ms += elapsed_time_ms_;
			}
			
			virtual «sc_bool.fqName» isPeriodic() const {
				switch (data.type) {
					case TaskData::TIME_EVENT_TASK: return data.get.time_event.periodic;
					case TaskData::RUNCYCLE_TASK: return true;
					default: return false;
				}
			}
			
			virtual bool isRunCycleEvent() const {
				return data.type == TaskData::RUNCYCLE_TASK;
			}
			
			virtual void reset() {
				data.reset();
				elapsed_time_ms = 0;
			}
			
			/*
			Compare tasks based on their execution order.
			Return true if this task is always to be executed strictly before the other task if both are scheduled to run at the same time.
			Default behavior:
			- This task is to be scheduled strictly before the other task if its isRunCycleEvent() method does not return true and the other task's isRunCycleEvent() method returns true.
			*/
			virtual bool operator<(const TimerTask& other) const {
				return (!isRunCycleEvent() && other.isRunCycleEvent());
			}
			
			class TimerTaskMatcher {
			public:
				virtual ~TimerTaskMatcher() {}
				virtual bool match(const «timerServiceImplementationTask» &other) = 0;
			};
			class MatchTimeEvent : public TimerTaskMatcher {
			private:
					«weakPtr»TimedInterface«pointerType» time_event_handle;
					«sc_eventid.fqName» pt_evid;
			public:
				MatchTimeEvent(«weakPtr»TimedInterface«pointerType» time_event_handle_, «sc_eventid.fqName» pt_evid_): time_event_handle(time_event_handle_), pt_evid(pt_evid_) {}
				
				virtual bool match(const «timerServiceImplementationTask» &other) «_override» {
					return other.data.type == TaskData::TIME_EVENT_TASK && time_event_handle«lockIfNecessary» == other.data.get.time_event.handle«lockIfNecessary» && pt_evid == other.data.get.time_event.pt_evid;
				}
				
				virtual ~MatchTimeEvent() {}
			};
			
			class MatchRunCycleOf : public TimerTaskMatcher {
			private:
					«weakPtr»«cycleBasedInterface»«pointerType» run_cycle_handle;
			public:
				MatchRunCycleOf(«weakPtr»«cycleBasedInterface»«pointerType» run_cycle_handle_): run_cycle_handle(run_cycle_handle_) {}
				
				virtual bool match(const «timerServiceImplementationTask» &other) «_override» {
					return other.data.type == TaskData::RUNCYCLE_TASK && run_cycle_handle«lockIfNecessary» == other.data.get.run_cycle_handle«lockIfNecessary»;
				}
				
				virtual ~MatchRunCycleOf() {}
			};
		};
		
		class «timerServiceImplementation»: public «timerInterface» {
		private:
			«timerServicePrivateMembers»
		protected:
			void setGenericTimer(const «timerServiceImplementationTask»::TaskData &data);
			void unsetGenericTimer(«timerServiceImplementationTask»::TimerTaskMatcher &matcher);
		public:
		
			«timerServiceConstructor»
		
			virtual ~«timerServiceImplementation»() {}
		
			«setTimer»

			«unsetTimer»
			
			«IF genExt.useSmartPointers(entry)»
			«unsetTimerRaw»
			«ENDIF»
			
			«setRuncycleTimerFor»
			
			«unsetRuncycleTimerFor»
			
			/*!
			 * This function must be called by the user. The elapsed time must be calculated every time,
			 * the function gets called.
			 */
			virtual void proceed(«sc_time.fqName» elapsed_ms);
		
			/*! Cancel timer service. Use this to end possible timing threads and free
			 memory resources.
			 */
			virtual void cancel();
			
			/*! Obtain the time (in ms) required to proceed to the next task.
			 */
			«sc_time.fqName» time_till_next_task();
		};
		
		} /* namespace sc::timer */
		} /* namespace sc */
		
		#endif /* «timerServiceModule.define»_H_ */
		
	'''
	
	def protected includes()'''
		#include <cstddef>
		#include "«timerModule.h»"
		#include "«typesModule.h»"
		#include "«cycleBasedModule.h»"
	'''
	
	def protected tasksArrayMember()'''
	«sharedPtr»«timerServiceImplementationTask» «pointerType»tasks;
	'''
	
	def protected timerServicePrivateMembers()'''
	size_t length;
	«tasksArrayMember»
	size_t next_active_task;
	size_t next_free_task;
	'''
	
	def protected timerServiceConstructor()'''
	«timerServiceImplementation»(«sharedPtr»«timerServiceImplementationTask» «pointerType»tasks_, size_t length_) : 
		length(length_), //
		tasks(tasks_), //
		next_active_task(length_), //
		next_free_task(0) // 
	{
		for (size_t i = 0; i < length; ++i) {
			tasks[i].next_task_idx = i+1;
		}
	}
	'''
	
	def protected timerTaskDefaultConstructor()'''
	«timerServiceImplementationTask»() : //
		data(), //
		elapsed_time_ms(0), //
		next_task_idx(0)
	{
	}
	'''
	
	def protected resetHandle(String handle)'''«handle» = «NULL_LITERAL»;'''
	
	def protected taskDataDestructor()'''
	virtual ~TaskData() {};
	'''
	
	def protected nullptrInit() '''«NULL_LITERAL»'''
	
	def protected timeBasedInit() ''''''
}
