/**
 * Copyright (c) 2020-2023 itemis AG - All rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * 
 */
package com.yakindu.sct.generator.cpp.files

import com.google.inject.Inject
import com.itemis.create.base.generator.core.types.Literals
import com.yakindu.sct.generator.c.extensions.GenmodelEntries
import com.yakindu.sct.generator.core.artifacts.IContentTemplate
import com.yakindu.sct.generator.core.artifacts.IGenArtifactConfigurations
import com.yakindu.sct.generator.cpp.CppFileNaming
import com.yakindu.sct.generator.cpp.CppNaming
import com.yakindu.sct.generator.cpp.CppPointers
import com.yakindu.sct.generator.cpp.CppSpecifiers
import com.yakindu.sct.generator.cpp.types.CppTypes
import com.yakindu.sct.model.sexec.ExecutionFlow
import com.yakindu.sct.model.sgen.GeneratorEntry

/**
 * @author Robin Herrmann
 * @author Axel Terfloth
 */
class TimerServiceSource implements IContentTemplate<ExecutionFlow> {

	@Inject protected extension GenmodelEntries
	@Inject protected extension CppFileNaming
	@Inject protected extension CppNaming
	@Inject protected extension CppTypes
	@Inject protected extension Literals
	@Inject protected extension CppPointers
	@Inject protected extension CppSpecifiers
	

	override content(ExecutionFlow flow, GeneratorEntry entry, IGenArtifactConfigurations locations) '''
		«entry.licenseText»
		
		#include "«timerServiceModule.h»"
		#include <algorithm>
		
		using namespace sc::timer;
		
		«setGenericTimer»
		
		«unsetGenericTimer»
		
		«setTimer»
		
		«unsetTimer»
		
		«setRuncycleTimerFor»
		
		«unsetRuncycleTimerFor»
		
		«entry.proceed»
		
		«cancel»
		
		«timeTillNextTask»
		
	'''
	
	def protected cancel()'''
	/*! Cancel timer service. Use this to end possible timing threads.
	*/
	void «timerServiceImplementation»::cancel() {
		for (size_t idx = 0; idx < length; ++idx) {
			tasks[idx].reset();
			tasks[idx].next_task_idx = idx + 1;
		}
		next_active_task = length;
		next_free_task = 0;
	}
	'''
	
	def protected proceed(GeneratorEntry entry)'''
	/*!
	* This function must be called by the user. The elapsed time must be calculated every time,
	* the function gets called.
	*/
	void «timerServiceImplementation»::proceed(«sc_time.fqName» elapsed_ms) {
		if (next_active_task >= length) {
			return;
		}
		
		«sc_time.fqName» time_to_proceed = std::min(time_till_next_task(), elapsed_ms);
		«sc_time.fqName» proceeded_time = 0;
		while (time_to_proceed > 0) {
			/* go through all active timers and update their time */
			size_t idx = next_active_task;
			while (idx < length) {
				TimerTask &current_task = this->tasks[idx]; 
				current_task.updateElapsedTimeMs(time_to_proceed);
				idx = current_task.next_task_idx;
			}
			
			«sc_bool.fqName» task_fired;
			do {
				task_fired = false;
				size_t before_best = length;
				size_t best = next_active_task;
				«sc_time.fqName» best_remaining_time = tasks[best].data.get.time_ms - tasks[best].elapsed_time_ms;
				size_t last_task = best;
				size_t next_task = tasks[best].next_task_idx;
				while (next_task < length) {
					TimerTask &current_task = tasks[next_task];
					«sc_time.fqName» remaining_time = current_task.data.get.time_ms - current_task.elapsed_time_ms; 
					if (remaining_time < best_remaining_time || (remaining_time == best_remaining_time && !(tasks[best] < current_task))) {
						best = next_task;
						before_best = last_task;
						best_remaining_time = remaining_time;
					}
					last_task = next_task;
					next_task = current_task.next_task_idx;
				}
				if (best_remaining_time <= 0) {
					«timerServiceImplementationTask» &best_task = tasks[best];
					«timerServiceImplementationTask»::TaskData data(best_task.data);
					if (best_task.isPeriodic()) { // reset periodic time
						best_task.elapsed_time_ms = 0;
					} else { // remove the task
						best_task.reset();
						if (before_best < length) { // remove in list
							tasks[before_best].next_task_idx = best_task.next_task_idx;
						} else { // remove head
							next_active_task = best_task.next_task_idx;
						}
						best_task.next_task_idx = next_free_task;
						next_free_task = best;
					}
					
					// execute the task
					data.execute();
					task_fired = true;
				}
			} while (task_fired && next_active_task < length);
			
			proceeded_time += time_to_proceed;
			time_to_proceed = std::min(time_till_next_task(), elapsed_ms - proceeded_time);
		}
	}
	'''
	
	def protected unsetGenericTimer()'''
	void «timerServiceImplementation»::unsetGenericTimer(«timerServiceImplementationTask»::TimerTaskMatcher &matcher) {
		size_t last_position = length;
		size_t next_position = next_active_task;
		
		while (next_position < length) {
			«timerServiceImplementationTask» &current_task = tasks[next_position];
			if (matcher.match(current_task)) {
				current_task.reset();
				if (last_position < length) {
					tasks[last_position].next_task_idx = current_task.next_task_idx;
				} else {
					next_active_task = current_task.next_task_idx;
				}
				size_t current_position = next_position;
				next_position = current_task.next_task_idx;
				current_task.next_task_idx = next_free_task;
				next_free_task = current_position;
			} else {
				last_position = next_position;
				next_position = current_task.next_task_idx;
			}
		}
	}
	'''
	
	def protected unsetTimer()'''
	/*! Unset the given time event.
	 */
	void «timerServiceImplementation»::unsetTimer(«sharedPtr»TimedInterface «pointerType»statemachine, «sc_eventid.fqName» event) {
		«timerServiceImplementationTask»::MatchTimeEvent matcher(statemachine, event);
		unsetGenericTimer(matcher);
	}
	'''
	
	def protected setGenericTimer()'''
	void «timerServiceImplementation»::setGenericTimer(const «timerServiceImplementationTask»::TaskData &data) {
		// do nothing if there are no free slots
		if (next_free_task >= length) {
			return;
		}
		
		// Insert the task at the front
		size_t inserted_task_idx = next_free_task;
		«timerServiceImplementationTask» &inserted_task = tasks[inserted_task_idx];
		inserted_task.data = data;
		next_free_task = inserted_task.next_task_idx;
		inserted_task.next_task_idx = next_active_task;
		next_active_task = inserted_task_idx;
	}
	
	'''
	
	def protected setTimer()'''
	/*! Start the timing for a time event.
	 */
	void «timerServiceImplementation»::setTimer(«sharedPtr»TimedInterface «pointerType»statemachine, «sc_eventid.fqName» event,
			«sc_time.fqName» time_ms, «sc_bool.fqName» isPeriodic) {
		«timerServiceImplementationTask»::TaskData data(statemachine, event, time_ms, isPeriodic);
		setGenericTimer(data);
	}
	'''
	
	def protected setRuncycleTimerFor()'''
	/*! Set a timer for running cycles of the given statemachine.
	 */
	void «timerServiceImplementation»::setRuncycleTimerFor(«sharedPtr»«cycleBasedInterface» «pointerType»statemachine,  «sc_time.fqName» cycle_period) {
		«timerServiceImplementationTask»::TaskData data(statemachine, cycle_period);
		setGenericTimer(data);
	}
	'''
	
	def protected unsetRuncycleTimerFor()'''
	/*! Unset timers for running cycles of the given statemachine.
	 */
	void «timerServiceImplementation»::unsetRuncycleTimerFor(«sharedPtr»«cycleBasedInterface» «pointerType»statemachine) {
		«timerServiceImplementationTask»::MatchRunCycleOf matcher(statemachine);
		unsetGenericTimer(matcher);
	}
	'''
	
	
	def protected timeTillNextTask()'''
	«sc_time.fqName» «timerServiceImplementation»::time_till_next_task() {
		if (next_active_task == length) {
			return 0;
		}
		
		«sc_time.fqName» time = tasks[next_active_task].data.get.time_ms - tasks[next_active_task].elapsed_time_ms;
		size_t task = tasks[next_active_task].next_task_idx;
		while (task < length) {
			«timerServiceImplementationTask» &current_task = tasks[task]; 
			«sc_time.fqName» remaining_time = current_task.data.get.time_ms - current_task.elapsed_time_ms;
			if (remaining_time < time) {
				time = remaining_time;
			}
			task = current_task.next_task_idx; 
		}
		
		return time;
	}
	'''

}
