/**
 * Copyright (c) 2020 itemis AG - All rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * 
 */
package com.yakindu.base.expressions.interpreter.base

import com.google.inject.Inject
import com.google.inject.Singleton
import java.util.concurrent.BlockingQueue
import java.util.concurrent.Callable
import java.util.concurrent.LinkedBlockingDeque
import org.apache.commons.lang.time.StopWatch
import org.eclipse.emf.common.notify.impl.AdapterImpl
import org.eclipse.emf.ecore.EObject
import org.eclipse.xtend.lib.annotations.Accessors

import static extension org.eclipse.emf.ecore.util.EcoreUtil.*
import com.yakindu.core.TimerService
import com.yakindu.core.ITracingListener
import com.yakindu.core.rx.Observable
import com.yakindu.base.expressions.interpreter.scheduling.ITimeTaskScheduler
import com.yakindu.base.expressions.interpreter.scheduling.TaskScheduler
import com.yakindu.core.rx.Observer

/**
 * An interpreter session provides thread safe debug frontend for an interpreter. It executes all interpreter execution requestst in a worker thread 
 * and cares about correct session handling for suspending resuming, stepping, and terminating an interpreter session.
 * 
 * @author axel terfloth
 */
@Singleton
class InterpreterSession extends InterpreterSessionControl implements  InterpreterSessionControl.OperationCallback, ITracingListener<InterpreterSessionControl.State>, ITimeTaskScheduler {
	
	protected BlockingQueue<Callable<Object>> requestQueue = null
	
	protected Thread listenerThread = null
	protected Thread workerThread = null

	protected IInterpreter interpreter = null
	protected TimerService tService = null
	protected Observable<Throwable> onError = new Observable<Throwable>()
	protected Observer<Object> doEvaluateObserver = [ o | 
		execute [ 
			interpreter.evaluate(o)
			null
		]
	] 
	
	protected boolean debug = false 
	
	
	@Inject new(IInterpreter i, IInterpreter.Control iCtrl) {
		super()
		
		tService = new TimerService
		timerService = tService
		operationCallback =  this
		addTraceObserver(this)
		
		scheduler = new TaskScheduler	
		
		enter
		interpreterControl = iCtrl
		interpreter = i
		SessionAdapter.adaptSession(interpreter.heap as EObject, this)
	}
	
	def getOnError() {
		this.onError
	}
	
	def Observer<Object> doEvaluate() {
		return doEvaluateObserver
	}
	
	override void start() {	
		raiseStart
	}
	
	def boolean canTerminate() { return getCanTerminate	}

	override void terminate() { 
		raiseTerminate
		scheduler = new TaskScheduler // TODO: fix ITimeTaskScheduler interface to provide reset
	}
	
	def boolean isTerminated() { return getIsTerminated }
	
	def boolean canSuspend() { return getCanSuspend	}
	override void suspend() { raiseSuspend }
	def boolean isSuspended() { getIsSuspended }

	def boolean canResume() { getCanResume }
	override void resume() { raiseResume }

	def boolean canStep() { getCanStep }	
	override void step() { raiseStep }
	def boolean isStepping() { getIsStepping }	
	
		
	def void raiseEvent(Object event, Object value) {
		execute [
			interpreter.raiseEvent(event, value)
			return null
		]
	}

	def void setValue(Object variable, Object value) {
		execute [
			interpreter.setValue(variable, value)
			return null
		]
	}

	def void execute(Callable<Object> callable) {
		if (acceptExecutionRequest) {
			log("worker schedule work")
			requestQueue?.offer(callable)
		}	
	}

	override startWorker() {
		workerThread = new Thread( [runWorker] , "YSCT.InterpreterSession.worker")
		workerThread.start
		log("worker started")
	}
	
	override stopWorker() {
		workerThread.interrupt		
		log("worker stopped")	
	}

	override startListener() {
		requestQueue = new LinkedBlockingDeque
		listenerThread = new Thread( [runListener], "YSCT.InterpreterSession.listener")
		listenerThread.start
		
		log("listener started")
	}
	
	override stopListener() {
		listenerThread.interrupt
		requestQueue.clear
		requestQueue = null
		
		log("listener stopped")	
	}
	
	
	override interrupt() {
		workerThread?.interrupt
		log("worker interrupt")	
	}
	
	override hasWork() {
		return scheduler.scheduledTimeTasks.head !== null
	}


	def void runWorker(){
		log("begin work")	
		while (! inactive) {
			
			switch(this) {
				case workIdle: doWorkIdle
				case workPickTask: doWorkPickTask
				case workTimeLeap: doWorkTimeLeap
				case workProcess: doWorkProcess
				case workSuspending : doWorkSuspending
				case workSuspended : doWorkSuspended
				case workContinueSuspended : doWorkContinueSuspended
			}
		}	
		log("end work")			
	}	
	
	
	def void runListener() {
		log("begin listen")	
		
		while (! inactive) {
			
			switch(this) {
				case listenAwaitRequest : doListenAwaitRequest
			}
		}

		log("end listen")	
		
	}


	def protected inactive() {
		isStateActive(State.MAIN_INACTIVE)
	}

	def protected workIdle() {
		isStateActive(State.MAIN_ACTIVATED_WORK_IDLE)
	}

	def protected void doWorkIdle() {
		log("doWorkIdle")			
		sleep
	}	
	
	def protected workPickTask() {
		 isStateActive(State.MAIN_ACTIVATED_WORK_PICK_TASK)
	}
	
	def protected void doWorkPickTask() {
		log("doWorkPickTask")			
		try {
			nextTask = scheduler.scheduledTimeTasks.head
			if ( nextTask !== null ) {
				raiseDone
			}
			else {
				proceedTime(100);				
			}
		} catch (InterruptedException ie) {
		}					
	}


	def protected workTimeLeap() {
		 isStateActive(State.MAIN_ACTIVATED_WORK_TIME_LEAP)
	}
	
	def protected void doWorkTimeLeap() {
		log("doWorkTimeLeap")
		
		nextTask = scheduler.scheduledTimeTasks.head
		
		if (nextTask !== null) {
			if((nextTask.nextExecutionTime - scheduler.currentTime <= 0)) {
				raiseDone
			}
			else {
				var leapTime = Math.min(nextTask.getNextExecutionTime() - scheduler.currentTime, 100)
				proceedTime(leapTime);			
			}
		}		
	}

	def protected workProcess() {
		 isStateActive(State.MAIN_ACTIVATED_WORK_PROCESS)
	}

	def protected void doWorkProcess() {
		log("doWorkProcess")			
		try {
			scheduler.timeLeap(0)
			raiseDone 					
		} catch (InterpreterSuspendedException ise) {
			raiseWorkSuspended
		} catch (InterpreterException ie) {
			onError.next(ie.cause ?: ie)
		}
	}

	
	def protected workSuspending() {
		 isStateActive(State.MAIN_ACTIVATED_WORK_SUSPENDING)
	}
	
	def protected void doWorkSuspending() {
		log("doWorkSuspending")			
	}
		
	def protected workSuspended() {
		isStateActive(State.MAIN_ACTIVATED_WORK_SUSPENDED)
	}
	
	def protected void doWorkSuspended() {
		log("doWorkSuspended")			
		sleep
	}
	
	
	def protected workContinueSuspended() {
		isStateActive(State.MAIN_ACTIVATED_WORK_CONTINUE_SUSPENDED)
	}
	
	
	def protected void doWorkContinueSuspended() {
		log("doWorkContinueSuspended")			
		try {
			interpreterControl.resume
			raiseDone 					
		} catch (InterpreterSuspendedException ise) {
			raiseWorkSuspended
		}			
	}
		
	
	def protected listenAwaitRequest() {
		isStateActive(State.MAIN_ACTIVATED_LISTEN_AWAIT_REQUEST)
	}
	
	def protected void doListenAwaitRequest() {
		log("doListenAwaitRequest")			
		try {
			val request = requestQueue.take
			val newTask = new TimeTask("request", [request.call], TimeTask.Priority.HIGH )
			scheduler.scheduleTask(newTask, 0);
			if (scheduler.scheduledTimeTasks.head === newTask)
				raiseNext
		} catch (InterruptedException ie) {
		}							
	}
		
	def protected sleep() {
		try {
			Thread.sleep(1000)
		} catch (InterruptedException ie) {
		}				
	}
	
	
	def protected long proceedTime(long millis) {

		var long consumedTime = 0;
		var watch = new StopWatch

		try {
			watch.start
			Thread.sleep(millis)
		} catch (InterruptedException ie) {
		} finally {
			watch.stop
			consumedTime = watch.time
			scheduler.timeLeapWithoutExecuting(consumedTime)
		}		

		return consumedTime;			
	}
	
	
	override void onStateEntered(InterpreterSessionControl.State state) {
		log(System.identityHashCode(this) + " enter " + state)
	}
	
	override void onStateExited(InterpreterSessionControl.State state) {
		log(System.identityHashCode(this) + " exit " + state)		
	}
	
	protected String lastLogMsg = null
	def protected void log(String msg) {
		if (debug) {
			if(msg.equals(lastLogMsg)) {
				System.out.print(".")
			} else {
				lastLogMsg = msg
				System.out.println("")
				System.out.print(lastLogMsg)				
			}
		}			
	}
		
	static class SessionAdapter extends AdapterImpl {
		
		@Accessors InterpreterSession session
		
		override isAdapterForType(Object type) {
			return type == SessionAdapter
		}
		
		static def void adaptSession(EObject it, InterpreterSession s) {
			it.eAdapters += new SessionAdapter() => [
				session = s
			]
		} 
		
		static def InterpreterSession session(EObject it) {
			val adapter = it.getRootContainer.getExistingAdapter(SessionAdapter)
			
			if(adapter !== null) 
				(adapter as SessionAdapter).session
			else 
				null
		}
		
	}
	
	override getCurrentTime() {
		clock.time
	}
	
	override getScheduledTimeTasks() {
		scheduler.scheduledTimeTasks
	}
	
	override scheduleChanged() {
		scheduler.scheduleChanged
	}
	
	override scheduleTimeTask(TimeTask task, boolean isPeriodical, long duration) {
		scheduler.scheduleTask(task, isPeriodical, duration)
	}
	
	override timeLeap(long ms) {
		scheduler.timeLeap(ms)
	}
	
	override cycleLeap(int cycles) {
		scheduler.cycleLeap(cycles)
	}
	
	override unscheduleTimeTask(String taskName) {
		scheduler.unschedulTask(taskName)
	}
	
	
}