/**
 * Copyright (c) 2020-2024 itemis AG - All rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * 
 */
package com.yakindu.base.expressions.interpreter.base

import com.google.inject.Inject
import com.yakindu.base.expressions.interpreter.base.BaseExecution
import com.yakindu.base.expressions.interpreter.base.IInterpreter
import com.yakindu.base.expressions.interpreter.base.IInterpreter.Instance
import com.yakindu.base.expressions.interpreter.base.InstanceSemantics
import com.yakindu.base.expressions.interpreter.base.InterpreterException
import com.yakindu.sct.model.sruntime.CompositeSlot
import com.yakindu.sct.model.sruntime.ExecutionSlot
import com.yakindu.sct.model.sruntime.ReferenceSlot
import com.yakindu.base.types.adapter.OriginTracing

/**
 * Abstract implementation of runtime statechart instance which implements the logic for resolving instance members. 
 * 
 * @author axel terfloth - interpreter refactoring
 * 
 */
abstract class ResolvingRuntimeInstance extends BaseExecution implements Instance, Cloneable {
	
	protected CompositeSlot instanceSlot

	@Inject protected extension InstanceSemantics
	@Inject protected extension OriginTracing


	def void setUp(CompositeSlot instance, IInterpreter.Context context) {
		this.instanceSlot = instance	
		this.context = context		
	}

	override instanceMemory() {
		instanceSlot
	}
	
	//=========================================================================================
	//
	// resolving elements (implementation of the IInterpreter.Resolver)
	//
	
	def protected doProvideThis() {
		_return[instanceSlot]
	}
		
	override resolve(Object owner, Object symbol) {
		if (owner === instanceSlot) {
			owner.resolveSlot(symbol)
		}
	}

	def dispatch resolveSlot(Object slot, Object symbol) {
		null
	}	


	def dispatch Object resolveSlot(ReferenceSlot slot, String symbol) {

		if ( slot.reference !== null ) {
			return context.resolve(slot.reference, symbol)
		}

		throw new InterpreterException("Cannot resolve '" + symbol + "' for null reference")
	}

	def dispatch ExecutionSlot resolveSlot(CompositeSlot slot, String symbol) {
		var ExecutionSlot s = slot.slotByName(symbol)
		 
		if (s === null) {
			if (defaultInterface !== null && slot !== defaultInterface) {
				s = defaultInterface.slotByName(symbol)
			}	
		}
		if ( s === null ) {
			s = internalScopes.filter[slot !== it].map[slotByName(symbol)].filterNull.head
		}	
		if ( s === null ) {
			if (timeEventScope !== null && slot !== timeEventScope) {
				s = timeEventScope.slotByName(symbol)
			}
		}	
		if ( s === null ) {
			if (imports !== null) {
				for (import : imports.slots) {
					if (import instanceof CompositeSlot) {
						s = import.slotByName(symbol)	
					}
					if (s !== null) {
						if (s instanceof ReferenceSlot) {
							if (s.reference !== null) {
								if (s.origin === s.reference.origin) s = s.reference 
							}
						}
						return s
					}					
				}	
			}
		}
		
		return s
	}	


	def defaultInterface() {
		instanceSlot.slotByName("default") as CompositeSlot
	}
	
	def internalScopes() {
		instanceSlot.slotsByName("internal").filter(CompositeSlot)
	}
	
	def timeEventScope() {
		instanceSlot.slotByName("time events") as CompositeSlot
	}

	def imports() {
		instanceSlot.slotByName("imports") as CompositeSlot
	}
	
	def protected slotsByName(CompositeSlot slot, String symbol) {
		slot.slots.filter[ s | s.name == symbol || s.fqName == symbol] 
	}
	
	def protected slotByName(CompositeSlot slot, String symbol) {
		slot.slots.findFirst[ s | s.name == symbol || s.fqName == symbol] 
	}
	
	def protected slotByKey(CompositeSlot slot, String symbol) {
		slot.slots.findFirst[ s | s.key == symbol] 
	}

	override IInterpreter.Instance instance(Object it) {
		if ( it === instanceSlot ) return this
		else return _instanceSemantics.instance(it)
	}
	
	override clone() {
		super.clone()
	}
}
