/**
 * Copyright (c) 2020-2025 itemis AG - All rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * 
 */
package com.yakindu.base.expressions.interpreter.context

import com.google.inject.Inject
import com.yakindu.base.expressions.interpreter.base.PropertySemantics
import com.yakindu.base.types.ComplexType
import com.yakindu.base.types.Declaration
import com.yakindu.base.types.Direction
import com.yakindu.base.types.EnumerationType
import com.yakindu.base.types.Event
import com.yakindu.base.types.GenericElement
import com.yakindu.base.types.Operation
import com.yakindu.base.types.Property
import com.yakindu.base.types.Type
import com.yakindu.base.types.TypeParameter
import com.yakindu.base.types.TypeSpecifier
import com.yakindu.base.types.TypedDeclaration
import com.yakindu.base.types.inferrer.ITypeSystemInferrer
import com.yakindu.base.types.inferrer.ITypeSystemInferrer.InferenceResult
import com.yakindu.base.types.typesystem.ITypeSemantics
import com.yakindu.base.types.typesystem.ITypeSystem
import com.yakindu.sct.model.sruntime.CompositeSlot
import com.yakindu.sct.model.sruntime.ExecutionContext
import com.yakindu.sct.model.sruntime.ExecutionSlot
import com.yakindu.sct.model.sruntime.SRuntimeFactory
import java.util.List
import org.eclipse.emf.ecore.EObject
import org.eclipse.xtext.naming.IQualifiedNameProvider
import org.eclipse.xtext.util.PolymorphicDispatcher

import static com.yakindu.base.types.typesystem.ITypeSystem.ANY
import com.yakindu.base.expressions.interpreter.base.EventSemantics
import com.yakindu.sct.model.sruntime.ExecutionEvent

/**
 * This class implements the initialization logic for slots based on the type system. 
 * This includes scalar as well as complex types. 
 * 
 * @author andreas muelder - Initial contribution and API
 * @author axel terfloth - refactoring & extensions
 * 
 */
class DefaultExecutionContextInitializer implements IExecutionContextInitializer {

	@Inject protected extension IQualifiedNameProvider
	@Inject protected extension ITypeSemantics
	@Inject protected extension PropertySemantics
	@Inject protected extension EventSemantics
	@Inject protected extension ITypeSystem
	@Inject protected extension ITypeSystemInferrer
	
	@Inject protected extension IExecutionSlotInitialValueProvider
	
	@Inject protected BuiltInTypeContextInitializer builtInInitializer
	

	protected extension SRuntimeFactory = SRuntimeFactory.eINSTANCE

	override initialize(ExecutionContext context, EObject instance) {
		instance.define(context)
	}

	override ExecutionSlot newInstance(EObject instance) {
		instance.transform
	}

	def dispatch void define(EObject obj, CompositeSlot instance) { }
	
	def dispatch void define(ComplexType type, CompositeSlot instance) {
		type.features.filter[!(it instanceof Operation)].forEach [
			instance.slots += transform => [
				visible = false
			]
		]	
	}
	
	
	def dispatch ExecutionSlot transform(EObject scope) { null }

	def dispatch ExecutionSlot transform(Property prop) {
		val slot = createExecutionSlotFor(prop)
		slot.writable = slot.writable && !prop.const
		slot.adaptProperty(prop)
		slot
	}

	def dispatch ExecutionSlot transform(Operation op) {
		val slot = createExecutionSlotFor(op)
		slot
	}

	def dispatch ExecutionSlot transform(Event ev) {
		val slot = createExecutionSlotFor(ev)
		if (slot instanceof ExecutionEvent) slot.adaptEvent(ev)
		slot
	}


	def protected getSlotFor(List<ExecutionSlot> slots, String name) {
		val existingSlot = slots.findFirst[it.name == name]
		if (existingSlot instanceof CompositeSlot) {
			existingSlot
		} else
			createCompositeSlot => [ newSlot |
				newSlot.name = name
				slots += newSlot
			]
	}


	def dispatch ExecutionSlot createExecutionSlotFor(Declaration element) {
		val inferenceResult = element.infer
		val varType = inferenceResult.type
		if (varType.builtInType) {
			var dispatcher = new PolymorphicDispatcher('''createSlotFor«varType.name.toFirstUpper»''', 2,2,	newArrayList(builtInInitializer))
			return (dispatcher.invoke(element, this) as ExecutionSlot) => [ visible=true ]
		}

		transformByType(varType, element, inferenceResult)
	}
	
	def dispatch ExecutionSlot createExecutionSlotFor(Event event) {
		val inferenceResult = event.type.infer
		val eventType = inferenceResult?.type

		transformByType(eventType, event, inferenceResult)
	}



	def protected dispatch ExecutionSlot transformByType(ComplexType type, Declaration element,
		InferenceResult inferenceResult) {
		
		if (type.isReferenceType) {
			createReferenceSlot => [
				it.type = type
				it.init(element)	
			]		
		} else {
			createLazyPopulatedSlot(inferenceResult, element) => [
				it.type = type
				it.init(element)
				it.visible = (!(element instanceof ComplexType) || !(it.getSlots(true).filter([s|s.isVisible]).isEmpty()))
			]
		}
	}
	
	def protected dispatch ExecutionSlot transformByType(ComplexType eventType, Event event,
		InferenceResult inferenceResult) {
		
		if (eventType.isReferenceType) {
			createReferenceExecutionEvent => [
				it.type = eventType
				it.init(event)
				it.direction = Direction.get(event.direction.value)
			]		
		} else {
			createCompositeExecutionEvent => [
				it.type = eventType
				it.init(event)
				it.direction = Direction.get(event.direction.value)
				for (feature : eventType.allFeatures.filter[ f | ! (f instanceof Operation)]) {
					val featureType = feature.infer.type
					val featureSlot = transformByType(featureType, feature, inferenceResult)
					it.slots += featureSlot
					featureSlot.fqName = fqName + "." + featureSlot.name
				}
			]
		}
	}

	def protected dispatch ExecutionSlot transformByType(TypeParameter type, Declaration element,
		InferenceResult inferenceResult) {
		val typeParameterInferenceResult = inferTypeParameter(type, inferenceResult)
		val inferred = typeParameterInferenceResult.type
		if (!(inferred instanceof TypeParameter)) {
			return transformByType(inferred, element, typeParameterInferenceResult)
		} else {
			return element.createDeclarationSlot => [
				it.name = element.fullyQualifiedName.lastSegment
				it.fqName = element.fullyQualifiedName.toString
				it.type = inferred
				it.value = null
			]
		}
	}
	
	def protected dispatch ExecutionSlot transformByType(EnumerationType type, Declaration element,
		InferenceResult inferenceResult) {
		element.createDeclarationSlot => [
			it.type = type
			it.init(element)
		]
	}

	def protected dispatch ExecutionSlot transformByType(Type type, Declaration element,
		InferenceResult inferenceResult) {
		element.createDeclarationSlot => [
			it.type = type
			it.init(element)
		]
	}
	
	
	def protected dispatch ExecutionSlot transformByType(Void type, Event element,
		InferenceResult inferenceResult) {
		createExecutionEvent => [
			it.type = element.type
			it.init(element)
			it.direction = Direction.get(element.direction.value)
		]
	}


	def protected dispatch createDeclarationSlot(Operation decl) {
		createExecutionOperation
	}
	def protected dispatch createDeclarationSlot(Declaration decl) {
		createExecutionVariable
	}
	
	def protected dispatch createDeclarationSlot(Event decl) {
		createExecutionEvent => [
			it.direction = Direction.get(decl.direction.value)
		]
	}	
	
	def protected createLazyPopulatedSlot(InferenceResult inferenceResult, Declaration owner) {
		val slot = createLazyPopulatedSlot
		slot.setPopulator([ s, c |
			if (s.type instanceof ComplexType)
				for (feature : (s.type as ComplexType).features) {
					if (feature.dynamicallyVisibleFromOwner(owner)) {
						val featureType = feature.provideType
						var ExecutionSlot featureSlot
						if (featureType.isBuiltInType) {
							featureSlot = feature.transform
						} else {
							featureSlot = transformByType(featureType, feature, inferenceResult)
						}
						c += featureSlot
						featureSlot.fqName = s.fqName + "." + featureSlot.name
						if (feature instanceof Type)
							featureSlot.visible = feature.visible		
					}
				}
		])
		slot
	}


	def protected dispatch Type provideType(Declaration it) {
		it.infer.type
	}

	def protected dispatch Type provideType(Event it) {
		it.typeSpecifier.provideType
	}
	
	def protected dispatch Type provideType(TypeSpecifier it) {
		it.infer.type
	}
	
	def protected dispatch Type provideType(Void it) {
		ITypeSystem.VOID.type
	}
	

	def protected init(ExecutionSlot it, Declaration variable) {
		name = variable.name
		fqName = variable.fullyQualifiedName.toString
		value = initialSlotValue(type, variable)
	}


	def dynamicallyVisibleFromOwner(Declaration it, Declaration owner) {
		return !(owner instanceof Type) ||
		 (!(it instanceof TypedDeclaration) || it.isStatic)			
	}

	// copied from ExpressionsTypeInferrer
	def protected inferTypeParameter( TypeParameter typeParameter, InferenceResult ownerResult) {
		if (ownerResult.bindings.isEmpty() || !(ownerResult.type instanceof GenericElement)) {
			return InferenceResult.from(ANY.type)
		} else {
			val index = (ownerResult.type as GenericElement).typeParameters.indexOf(typeParameter)
			return InferenceResult.from(ownerResult.bindings.get(index).type, ownerResult.bindings.get(index).bindings)
		}
	}

}
