/**
 * Copyright (c) 2024 itemis AG - All rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * 
 */
package com.yakindu.base.expressions.interpreter.types

import com.google.inject.Inject
import com.yakindu.base.expressions.expressions.ArgumentExpression
import com.yakindu.base.expressions.interpreter.base.IInstanceFactory
import com.yakindu.base.expressions.interpreter.base.IInterpreter
import com.yakindu.base.types.ComplexType
import com.yakindu.base.types.Operation
import com.yakindu.base.types.Package
import com.yakindu.base.types.typesystem.ITypeSystem
import com.yakindu.sct.model.sruntime.CompositeSlot
import com.yakindu.sct.model.sruntime.ExecutionSlot
import com.yakindu.sct.model.sruntime.SRuntimeFactory

/**
 * Instance behavior implementation for arrays.
 * 
 * @author Axel Terfloth (terfloth@itemis.de)
 */
class ArrayInstance extends RuntimeInstance {

	@Inject extension ITypeSystem

	protected ComplexType arrayType
	protected Package declarationPackage

	override void setUp(CompositeSlot instance, IInterpreter.Context context, IInstanceFactory factory) {
		super.setUp(instance, context, factory)
		this.arrayType = ITypeSystem.ARRAY.type as ComplexType
		this.declarationPackage = this.arrayType.eContainer as Package
	}

	override declarationPackage() {
		declarationPackage
	}

	override declarations() {
		arrayType.allFeatures
	}

	override dispatch void doInvoke(ArgumentExpression expr) {
		val feature = expr.featureOrReference

		switch (feature) {
			Operation case "add" == feature.name:
				add()
			Operation case "addAll" == feature.name:
				addAll()
			Operation case "remove" == feature.name:
				remove()
			Operation case "removeAll" == feature.name:
				removeAll()
			Operation case "size" == feature.name:
				size()
			Operation case "contains" == feature.name:
				contains()
			Operation case "clear" == feature.name:
				clear()
			default:
				super._doInvoke(expr)
		}
	}

	def protected clear() {
		instanceSlot.slots.clear
	}

	def protected contains() {
		val value = popValue
		pushValue(instanceSlot.slots.exists[slot|slot.value == value])
	}

	def protected size() {
		pushValue(instanceSlot.slots.size)
	}

	def protected removeAll() {
		val removeAll = popValue as CompositeSlot
		removeAll.slots.forEach[value.pushValue remove]
	}

	def protected remove() {
		val value = popValue
		val slotValue = instanceSlot.slots.findFirst[slot|slot.value == value]
		if (slotValue !== null)
			instanceSlot.slots.remove(slotValue)
		updateIndex
	}

	def protected addAll() {
		val addAll = popValue as CompositeSlot
		addAll.slots.forEach[value.pushValue add]
	}

	def protected add() {
		val value = popValue
		val newSlot = SRuntimeFactory.eINSTANCE.createExecutionVariable => [ slot |
			slot.value = value
			slot.name = "" + '''[«instanceSlot.slots.size»]'''
			slot.type = getType(ITypeSystem.ANY)
		]
		instanceSlot.slots += newSlot
		updateIndex
		pushValue(IInterpreter.NULL)
	}

	def protected updateIndex() {
		instanceSlot.slots.forEach[variable, index|variable.name = '''[«index»]''']
	}

	// =========================================================================================
	//
	// Handling array assignment
	//
	override set(Object slot, Object value) {
		if (slot === instanceSlot) {
			if (slot instanceof CompositeSlot) {
				if (value instanceof CompositeSlot) {
					slot.slots.clear
					value.slots.forEach [ s, i |
						slot.slots += (factory.copyInstance(s) as ExecutionSlot) => [
							name = '''[«i»]'''
							fqName = name
						]
					]
				} else {
					_valueSemantics.setValue(slot, value)
				}
			}
		} else {
			_valueSemantics.setValue(slot, value)
		}
	}

}
