/**
 * Copyright (c) 2024 itemis AG - All rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * 
 */
package com.yakindu.base.expressions.interpreter.types

import com.google.inject.Inject
import com.google.inject.Injector
import com.yakindu.base.expressions.interpreter.base.IInstanceFactory
import com.yakindu.base.expressions.interpreter.base.IInterpreter
import com.yakindu.base.expressions.interpreter.base.InstanceSemantics
import com.yakindu.base.expressions.interpreter.base.InterpreterException
import com.yakindu.base.expressions.interpreter.base.SRuntimeInterpreter
import com.yakindu.base.expressions.interpreter.base.ValueSemantics
import com.yakindu.base.expressions.interpreter.scheduling.ITimeTaskScheduler
import com.yakindu.base.expressions.interpreter.scheduling.ITimeTaskScheduler.TimeTask
import com.yakindu.sct.model.sruntime.CompositeSlot
import com.yakindu.sct.model.sruntime.ExecutionEvent
import com.yakindu.sct.model.sruntime.ReferenceSlot
import org.eclipse.xtend.lib.annotations.Accessors
import com.yakindu.sct.model.sruntime.ExecutionArgument
import com.yakindu.sct.model.sruntime.ExecutionSlot

/**
 * 
 * @author Axel Terfloth
 */
abstract class TypesInterpreter extends SRuntimeInterpreter {
	
	@Inject protected extension Injector injector
	@Inject protected extension IInstanceFactory factory
	@Inject protected extension InstanceSemantics
	@Inject protected extension PackageHelper
	@Inject protected extension ValueSemantics
	
	@Inject @Accessors protected ITimeTaskScheduler timingService
	
	protected IInterpreter.Execution execution 	
	
	/** 
	 * Instantiation of instances is delegated to the instance factory.
	 */
	override newInstance(Object type) {
		process("newInstance(" + type.class.simpleName + ")", [
			factory.newInstance(type).pushValue
		])
	}


	override reset() {
		cleanupMemory
	}


	override void prepareExecution(Object program) {
		execution.provideExecution(program)
	}


	//===============================================================
	// implementation of the IInterpreter interface
	//


	override setValue(Object slot, Object value) {
		doSet(slot, value)
	}

	def dispatch void doSet(Object e, Object value) {
		throw new InterpreterException('''Don't know how to set member of type «e?.class.name» ! ''')
	}
	
	def dispatch void doSet(ExecutionSlot slot, Object value) {
		
		process("request set " + slot.name, [
			_execute("set " + slot.name, [
				var inst = slot.instance
				if (inst !== null) {
					inst.set(slot, value)
				} else {
					_valueSemantics.setValue(slot, value)
				}
			])
		])		
	}

	override set(Object slot, Object value) {
		do_Set(slot, value)
	}

	def dispatch void do_Set(Object e, Object value) {
		throw new InterpreterException('''Don't know how to set member of type «e?.class.name» ! ''')
	}
	
	def dispatch void do_Set(ExecutionSlot slot, Object value) {
		_execute("set " + slot.name, [
			var inst = slot.instance
			if (inst !== null) {
				inst.set(slot, value)
			} else {
				_valueSemantics.setValue(slot, value)
			}
		])
	}


	

	override get(Object slot) {
		slot.asValue
	}

	override getValue(Object slot) {
		slot.asValue
	}

	override _invoke(Object receiver, String operation, Object... args) {
		do_invoke(receiver, operation, args)
	}
	
	def dispatch void do_invoke(Object receiver, String operation, Object... args) {
		throw new InterpreterException('''Don't know how to invoke operation for receiver of type «receiver?.class.name» ! ''')
	}
	
	def dispatch void do_invoke(CompositeSlot receiver, String operation, Object... args) {
		_try('''call «receiver.name».«operation»''', [
			args.forEach[ pushValue ]
//			receiver.instance.invoke(receiver, operation)
			receiver.instance.provideExecution(operation)
		], [ onError |
			throw new InterpreterException(onError)
		])
	}


	override invokeOperation(Object receiver, String operation, Object... args) {		
		receiver.doInvoke(operation, args)
	}


	protected def dispatch Object doInvoke(Object receiver, String operation, Object... args) {
		throw new InterpreterException('''Don't know how to invoke operation for receiver of type «receiver?.class.name» ! ''')
	}


	protected def dispatch Object doInvoke(CompositeSlot receiver, String operation, Object... args) {
		process("request invoke " + operation, [
			args.forEach[ pushValue ]
			receiver.instance.provideExecution(operation)
		])				
	}


	override void raiseEvent(Object e, Object value) {
		doRaise(e, value)
	}
	
	def dispatch void doRaise(Object e, Object value) {
		throw new InterpreterException('''Don't know how to raise event of type «e?.class.name» ! ''')
	}
	
	def dispatch void doRaise(ExecutionEvent e, Object value) {
		
		process("request raise " + e.name, [
			_execute("raise " + e.name, [
				executeRaise(e, value)
			])		
		])		
	}

	def protected void executeRaise(ExecutionEvent e, Object value) {
		e.raise(value)
	}

	//===============================================================
	// implementation of the IInterpreter.Context interface
	//

	override _requestExecution(Object program, Execution requester) {
		if (requester === this.execution) {
			SELF?.instance?.provideExecution(program)
		} else {
			this.execution.provideExecution(program)
		}
	}


	override _schedule(String jobId, long duration, boolean periodic, Runnable action) {
		timingService.scheduleTimeTask(new TimeTask(jobId, [
			process('''timeout(«jobId»)''', action)
		]), periodic, duration) 
	}
	
	override _unschedule(String jobId) {
		timingService.unscheduleTimeTask(jobId)
	}


	override raise(Object o, Object value) {
			raiseSlot(o, value)
	}


	def dispatch raiseSlot(Object o, Object value) {
		throw new InterpreterException("Cannot raise on slot of type " + o.class.name);
	}


	def dispatch raiseSlot(ExecutionEvent it, Object value) {
		if (it.instance !== null) {
			it.instance.raise(it, value)
		} else {
			it.raised = true
			set(it, value)
		}
	}


	override currentTime() {
		return timingService.currentTime
	}


	//===============================================================
	// implementation of the IInterpreter.Resolver interface
	

	
	override resolve(Object owner, Object symbol) {
		if (owner === null) {
			val slot = localScope?.resolveSlot(symbol)
			if(slot !== null) return slot

			heap.resolveSlot(symbol)
		} else
			owner.resolveSlot(symbol)
	}

	def dispatch Object resolveSlot(Object slot, Object symbol) {
		throw new InterpreterException("Cannot resolve symbol '" + symbol + "' on slot '" + slot + "'!")
	}

	def dispatch Object resolveSlot(ExecutionArgument slot, String symbol) {
		slot.value.resolveSlot(symbol)
	}


	def dispatch Object resolveSlot(CompositeSlot slot, String symbol) {

		//
		var Object resolvedSlot = slot.slots.findFirst[s|s.name == symbol]
		if(resolvedSlot !== null) return resolvedSlot

		val inst = slot.instance
		if (inst !== null && inst instanceof IInterpreter.Resolver) {
			resolvedSlot = (inst as IInterpreter.Resolver).resolve(slot, symbol)
		}
		if(resolvedSlot !== null) return resolvedSlot

		val selfSlot = slot.slots.findFirst[s|s.name == InstanceSemantics.SELF_NAME]
		if (selfSlot !== null) {
			val SELF = selfSlot.value
			if(SELF !== slot) resolvedSlot = SELF.resolve(symbol)
		}

		return resolvedSlot
	}

	def dispatch Object resolveSlot(ReferenceSlot slot, String symbol) {

		if ( slot.reference !== null ) {
			return slot.reference.resolveSlot(symbol)
		}

		throw new InterpreterException("Cannot resolve '" + symbol + "' for null reference")
	}

	def dispatch Object resolveSlot(CompositeSlot slot, Long index) {

		if (slot.slots.size > index)  
			return slot.slots.get(index.intValue)

		throw new InterpreterException("Array index (" + index + ") out of bounds (0.." + (slot.slots.size-1) + "). ")
	}

	override IInterpreter.Instance instance(Object it) {
		_instanceSemantics.instance(it)
	}

}
