/**
 * Copyright (c) 2023 itemis AG - All rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * 
 * Contributors:
 * 	Axel Terfloth - itemis AG
 * 
 */
package com.itemis.create.base.generator.core

import com.google.inject.Inject
import com.yakindu.base.base.NamedElement
import com.yakindu.base.expressions.expressions.DeclarationExpression
import com.yakindu.base.expressions.expressions.ElementReferenceExpression
import com.yakindu.base.expressions.expressions.FeatureCall
import com.yakindu.base.types.Annotation
import com.yakindu.base.types.Declaration
import com.yakindu.base.types.Type
import com.yakindu.base.types.Package
import com.yakindu.base.types.TypeSpecifier
import com.yakindu.base.types.adapter.OriginTracing
import java.util.HashMap
import java.util.HashSet
import java.util.Map
import java.util.Set
import org.eclipse.emf.ecore.EObject
import org.eclipse.emf.ecore.util.EcoreUtil

/**
 * This class implements a basic model to model transformation which creates a new target model from a
 * source model which acts as the input. As a result the transformation makes sure that the created model 
 * is self contained in the sense that there are no regular model references from the target to the source model. 
 * 
 * The transformation is performed in three phases: 
 * 1. the transformation of compositions
 * 2. the transformation of (cross) references
 * 3. modifying the target model
 * 
 * During the first phase target elements are created from source elements. This transformation relation 
 * can be traced.As a result of the first phase the target model may reference source elements. 
 * In the second phase those references will be retargeted to the proper target elements using the transformation traces.  
 *
 * Client invoke the transform method. Subclasses must implement the toTarget method and call: 
 * - transformedRootFrom for each transformed root composite 
 * - transformedFrom for each other transformed element 
 * 
 * @author Axel Terfloth
 */
 
abstract class Transformation <S, T> {
	
	@Inject protected extension OriginTracing

	protected S source
	protected T target
	
	protected Set<EObject> sourceScopes = new HashSet
	protected Set<EObject> targetScopes = new HashSet
	
	
	def protected void transformedFrom(EObject target, EObject source) {

		if (! target.originTraces.contains(source)) {
			target.traceOrigin(source)		
		}
	}
	
	def protected void transformedRootFrom(EObject target, EObject source) {
		
		target.transformedFrom(source)
		sourceScopes += source
		targetScopes += target
	}
	
	def T transform(S source) {		
		this.source = source
		
		transformContainment
		transformReferences
		this.target.modifyTarget
		
		return this.target
	}
	
	
	def protected void transformContainment() {
		target = source.toTarget
	}
	
	abstract def protected T toTarget(S source)
	
	
	def protected void transformReferences() {
		
		val transformedTargetDeclarations = targetScopes.map[ eAllContents.filter(NamedElement).filter[origin !== null].toSet ].flatten.toSet
		val allSourceDeclarations = sourceScopes.map[ eAllContents.filter(NamedElement).toSet ].flatten.toSet

		val sourceToTargetMap = new HashMap<NamedElement, NamedElement>
		transformedTargetDeclarations.forEach[ 
			val origin = it.origin
			if ( origin instanceof NamedElement ) {
				if (allSourceDeclarations.contains(origin)) {
					sourceToTargetMap.put(origin, it)
				}				
			}
		]
		
		targetScopes.forEach[ eAllContents.forEach[ retargetDeclarationReferences(sourceToTargetMap)] ]
	}
	
	def protected void modifyTarget(T target) {}
	
	def protected dispatch retargetDeclarationReferences(EObject it, Map<NamedElement, NamedElement> map) {
	}
	
	
	def protected dispatch retargetDeclarationReferences(TypeSpecifier it, Map<NamedElement, NamedElement> map) {
		map.target(it.type).ifPresent[ target | it.type = target]
	}
	
	def protected dispatch retargetDeclarationReferences(ElementReferenceExpression it, Map<NamedElement, NamedElement> map) {
		if(reference instanceof NamedElement)
			map.target(it.reference as NamedElement).ifPresent[ target | it.reference = target]
	}
	
	def protected dispatch retargetDeclarationReferences(FeatureCall it, Map<NamedElement, NamedElement> map) {
		if(feature instanceof NamedElement)
			map.target(it.feature as NamedElement).ifPresent[ target | it.feature = target]
	}
	
	def protected dispatch retargetDeclarationReferences(DeclarationExpression it, Map<NamedElement, NamedElement> map) {
		map.target(it.declaration).ifPresent[ target | it.declaration = target]
	}
		
	def <T extends NamedElement> T target(Map<NamedElement, NamedElement> map, T source) {
		return map.get(source) as T		
	}
	
	def <T> T ifPresent(T it, (T)=>void action) {
		if (it !== null) {
			action.apply(it)
		}
		return it
	}
	
	def protected <T extends EObject> T copy(T it) {
		EcoreUtil.copy(it)
	}
	
	def protected Declaration createCopy(Declaration source) {

		val copier = new EcoreUtil.Copier {
			override copy(EObject source) {
				val target = super.copy(source)
				if (source instanceof NamedElement || source instanceof Annotation) {
					target.traceOrigin(source)	
				}
				
				if (target instanceof TypeSpecifier) {
					target.type = (source as TypeSpecifier).type
				}
				
				target.eAllContents.filter(Annotation).forEach[a |
					a.type =
						(a.origin as Annotation).type
				]

				return target
			}
		}
		
		copier.copy(source) as Declaration
	}

	def protected Set<Type> referencedTypes(EObject it) {
	    if (it === null) return #{}
	
	    // All types that are part of the current object's containment tree
	    val containedTypes = it.eAllContents.filter(Type).toSet
	
	    // Track visited objects and found types
	    val visited = newHashSet
	    val result = newHashSet
	
	    collectReferencedTypes(it, containedTypes, visited, result)
	
	    return result.filter[eContainer === null || eContainer instanceof Package].toSet
	}

	def private void collectReferencedTypes(EObject current, Set<Type> localTypes, Set<EObject> visited, Set<Type> result) {
	    if (!visited.add(current)) {
	    	return
		}
	
	    // Add direct type references
	    if (current instanceof TypeSpecifier) {
	    	val type = current.type
	        if (type !== null && !localTypes.contains(type)) {
	            result.add(type)
	            collectReferencedTypes(type, localTypes, visited, result)
	        }
	    }
	
	    // Traverse all containment children
	    for (child : current.eContents) {
	        collectReferencedTypes(child, localTypes, visited, result)
	    }
	
	    // Traverse all cross-references
	    for (xref : current.eCrossReferences) {
	        collectReferencedTypes(xref, localTypes, visited, result)
	    }
	}
	
}