/**
 * Copyright (c) 2025 itemis AG - All rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * 
 */
package com.yakindu.sct.json.transformation.json2xmi

import com.yakindu.sct.json.transformation.model.JsonChoice
import com.yakindu.sct.json.transformation.model.JsonEntry
import com.yakindu.sct.json.transformation.model.JsonExit
import com.yakindu.sct.json.transformation.model.JsonFinal
import com.yakindu.sct.json.transformation.model.JsonGraph
import com.yakindu.sct.json.transformation.model.JsonRegion
import com.yakindu.sct.json.transformation.model.JsonRoot
import com.yakindu.sct.json.transformation.model.JsonState
import com.yakindu.sct.json.transformation.model.JsonSynchronization
import com.yakindu.sct.json.transformation.model.JsonTransition
import com.yakindu.sct.json.transformation.model.util.NodeModelFactory
import com.yakindu.sct.json.transformation.model.util.NotationUtil
import com.yakindu.sct.model.sgraph.ChoiceKind
import com.yakindu.sct.model.sgraph.EntryKind
import com.yakindu.sct.model.sgraph.Region
import com.yakindu.sct.model.sgraph.SGraphFactory
import com.yakindu.sct.model.sgraph.Vertex
import java.util.HashMap
import java.util.Map
import org.eclipse.emf.common.util.URI
import org.eclipse.emf.ecore.EObject
import org.eclipse.emf.ecore.resource.Resource
import org.eclipse.gmf.runtime.emf.core.resources.GMFResource
import org.eclipse.gmf.runtime.notation.Diagram
import org.eclipse.gmf.runtime.notation.MeasurementUnit
import org.eclipse.gmf.runtime.notation.Node
import org.eclipse.gmf.runtime.notation.NotationFactory
import org.eclipse.gmf.runtime.notation.View

/**
 * 
 * @author andreas muelder - Initial contribution and API
 * 
 */
class JSONToXMITransformation {

	extension SGraphFactory = SGraphFactory.eINSTANCE
	extension NotationFactory = NotationFactory.eINSTANCE
	extension NotationUtil = new NotationUtil
	extension NodeModelFactory = new NodeModelFactory

	Map<String, Vertex> vertexMap = new HashMap
	Map<String, Node> nodeMap = new HashMap

	def Resource transform(JsonRoot root) {
		vertexMap.clear
		nodeMap.clear
		val resource = new GMFResource(URI.createFileURI("dummy.sct"))
		val jsonStatechart = root.graph.findStatechart
		val diagram = createDiagram => [
			measurementUnit = MeasurementUnit.PIXEL_LITERAL
			type = "itemis.create.ui.editor.editor.StatechartDiagramEditor"
			styles += createBooleanValueStyle => [
				name = "inlineDefinitionSection"
			]
		]
		val statechart = createStatechart => [
			name = jsonStatechart.name
			specification = jsonStatechart.specification
			regions += root.graph.findTopLevelRegions.map[transformElement(root.graph, it, diagram)].filter(Region)
		]
		val verticesWithoutRegion = root.graph.findTopLevelVertices
		if (verticesWithoutRegion.size > 0) {
			val topLevelRegion = createRegion => [
				name = "main"
				vertices += verticesWithoutRegion.map[transformElement(root.graph, it, diagram)].filter(Vertex)
			]
			// TODO: Calculate bounds
			val regionShape = topLevelRegion.createRegionNode
			diagram.insertChild(regionShape)
			statechart.regions += topLevelRegion
		}

		root.graph.cells.filter(JsonTransition).forEach[transformElement(root.graph, it, diagram)]
		diagram.element = statechart
		resource.contents += statechart
		resource.contents += diagram
		resource
	}

	def dispatch protected EObject transformElement(JsonGraph graph, JsonState jsonState, View container) {
		val state = createState => [
			name = jsonState.name
			specification = jsonState.specification
		]
		vertexMap.put(jsonState.id, state)

		val shape = state.createStateNode
		shape.layoutConstraint = graph.toRelativeBounds(jsonState)
		nodeMap.put(jsonState.id, shape)
		container.insertChild(shape)
		val compartmentNode = shape.children.filter(View).findFirst[type == "StateFigureCompartment"]

		val children = graph.findChildren(jsonState).filter(JsonRegion)
		val regions = children.map [ child |
			transformElement(graph, child, compartmentNode)
		].filter(Region)
		state.regions += regions
		state
	}

	def dispatch protected EObject transformElement(JsonGraph graph, JsonRegion jsonRegion, View container) {
		val region = createRegion => [
			name = jsonRegion.name
		]
		val shape = region.createRegionNode

		shape.layoutConstraint = graph.toRelativeBounds(jsonRegion) => [
			width = width
			height = height + 20 // offset for name label
		]

		container.insertChild(shape)
		val compartmentNode = shape.children.filter(View).findFirst[type == "RegionCompartment"]

		val children = graph.findChildren(jsonRegion).filter[!(it instanceof JsonTransition)]
		val vertices = children.map[child|transformElement(graph, child, compartmentNode)].filter(Vertex)
		region.vertices += vertices
		region
	}

	def dispatch protected EObject transformElement(JsonGraph graph, JsonEntry jsonEntry, View container) {
		val entry = createEntry => [
			name = graph.findNodeLabel(jsonEntry).name
			kind = EntryKind.get(jsonEntry.entryKind)
		]
		vertexMap.put(jsonEntry.id, entry)

		val entryShape = entry.createEntryNode
		entryShape.layoutConstraint = graph.toRelativeBounds(jsonEntry)
		nodeMap.put(jsonEntry.id, entryShape)
		container.insertChild(entryShape)

		entry
	}

	def dispatch protected EObject transformElement(JsonGraph graph, JsonExit jsonExit, View container) {
		val exit = createExit => [
			name = graph.findNodeLabel(jsonExit).name
		]
		vertexMap.put(jsonExit.id, exit)
		val exitShape = exit.createExitNode
		exitShape.layoutConstraint = graph.toRelativeBounds(jsonExit)

		nodeMap.put(jsonExit.id, exitShape)

		container.insertChild(exitShape)
		exit
	}

	def dispatch protected EObject transformElement(JsonGraph graph, JsonFinal jsonFinal, View container) {
		val finalState = createFinalState => [
			name = jsonFinal.name
		]
		vertexMap.put(jsonFinal.id, finalState)
		val finalShape = finalState.createFinalNode
		finalShape.layoutConstraint = graph.toRelativeBounds(jsonFinal)
		nodeMap.put(jsonFinal.id, finalShape)
		container.insertChild(finalShape)
		finalState
	}

	def dispatch protected EObject transformElement(JsonGraph graph, JsonChoice jsonChoice, View container) {
		val choice = createChoice => [
			name = jsonChoice.name
			kind = ChoiceKind.DYNAMIC
		]
		vertexMap.put(jsonChoice.id, choice)

		val choiceShape = choice.createChoiceNode
		choiceShape.layoutConstraint = graph.toRelativeBounds(jsonChoice)

		nodeMap.put(jsonChoice.id, choiceShape)

		container.insertChild(choiceShape)
		choice
	}

	def dispatch protected EObject transformElement(JsonGraph graph, JsonSynchronization jsonSync, View container) {
		val sync = createSynchronization
		vertexMap.put(jsonSync.id, sync)
		val syncShape = sync.createSynchNode
		syncShape.layoutConstraint = graph.toRelativeBounds(jsonSync)
		nodeMap.put(jsonSync.id, syncShape)
		container.insertChild(syncShape)
		sync
	}

	def dispatch protected EObject transformElement(JsonGraph graph, JsonTransition jsonTransition, View container) {
		val transition = createTransition => [
			specification = jsonTransition.specification
			source = vertexMap.get(jsonTransition.sourceId)
			target = vertexMap.get(jsonTransition.targetId)
		]

		val edge = transition.createTransitionEdge
		edge.source = nodeMap.get(jsonTransition.sourceId)
		edge.target = nodeMap.get(jsonTransition.targetId)
		if (jsonTransition?.source?.anchor !== null)
			edge.sourceAnchor = createAnchor(jsonTransition?.source?.anchor)
		if (jsonTransition?.target?.anchor !== null)
			edge.targetAnchor = createAnchor(jsonTransition?.target?.anchor)

		edge.insertChild(createDecorationNode => [
			type = "TransitionExpression"
			layoutConstraint = createLabelPosition(jsonTransition)
		])

		(container as Diagram).insertEdge(edge)

		transition
	}

}
